/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "visualsurface2d.h"
#include "drawable.h"
#include "stacks2d.h"

//#define SKIP_DRAW

M4Err VS2D_InitSurface(VisualSurface2D *surf)
{
	if (!surf->the_surface) {
		surf->the_surface = surf->render->g_hw->new_surface(surf->render->g_hw);
		if (!surf->the_surface) return M4IOErr;
	}
	if (!surf->the_brush) {
		surf->the_brush = surf->render->g_hw->new_stencil(surf->render->g_hw, M4StencilSolid);
		if (!surf->the_brush) return M4IOErr;
	}
	if (!surf->the_pen) {
		surf->the_pen = surf->render->g_hw->new_stencil(surf->render->g_hw, M4StencilSolid);
		if (!surf->the_pen) return M4IOErr;
	}
	return surf->GetSurfaceAccess(surf);
}

void VS2D_TerminateSurface(VisualSurface2D *surf)
{
	if (surf->the_surface) {
		surf->render->g_hw->surface_flush(surf->the_surface);
		surf->ReleaseSurfaceAccess(surf);
	}
}

void VS2D_ResetGraphics(VisualSurface2D *surf)
{
	if (surf->the_surface) surf->render->g_hw->delete_surface(surf->the_surface);
	surf->the_surface = NULL;
	if (surf->the_brush) surf->render->g_hw->delete_stencil(surf->the_brush);
	surf->the_brush = NULL;
	if (surf->the_pen) surf->render->g_hw->delete_stencil(surf->the_pen);
	surf->the_pen = NULL;
}

void VS2D_Clear(VisualSurface2D *surf, M4IRect *rc, u32 BackColor)
{
#ifdef SKIP_DRAW
	return;
#endif
	if (!surf->the_surface) return;

	if (!BackColor) {
		SFColor c = surf->render->compositor->clear_color;
#ifdef M4_DEF_CompositeTexture2D
		if (!surf->composite) BackColor = MAKE_ARGB_FLOAT(1.0, c.red, c.green, c.blue);
#else
		BackColor = MAKE_ARGB_FLOAT(1.0, c.red, c.green, c.blue);
#endif
	}
	surf->render->g_hw->surface_clear(surf->the_surface, rc, BackColor);
}


static void draw_clipper(VisualSurface2D *surf, struct _drawable_context *ctx)
{
	M4PenSettings clipset;
	LPM4PATH clippath, cliper;

	if (ctx->is_background) return;

	memset(&clipset, 0, sizeof(M4PenSettings));
	clipset.width = 2.0;

	clippath = m4_new_path();
	m4_path_add_rectangle(clippath, ctx->original.x + ctx->original.width/2, ctx->original.y - ctx->original.height/2, ctx->original.width, ctx->original.height);
	cliper = m4_path_get_outline(clippath, clipset);
	m4_path_delete(clippath);
	surf->render->g_hw->surface_set_matrix(surf->the_surface, &ctx->transform);
	surf->render->g_hw->surface_set_clipper(surf->the_surface, NULL);
	surf->render->g_hw->surface_set_path(surf->the_surface, cliper);
	surf->render->g_hw->set_brush_color(surf->the_pen, 0xFF000000);
	surf->render->g_hw->surface_fill(surf->the_surface, surf->the_pen);

	m4_path_delete(cliper);
}

static void VS2D_DoFill(VisualSurface2D *surf, DrawableContext *ctx, LPM4STENCIL stencil)
{
	u32 i;

	/*background rendering - direct rendering: use ctx clip*/
	if (ctx->is_background || (surf->render->top_effect->traversing_mode == TRAVERSE_RENDER_DIRECT)) {
		surf->render->g_hw->surface_set_clipper(surf->the_surface, &ctx->clip);
		surf->render->g_hw->surface_fill(surf->the_surface, stencil);
	} 
	/*indirect rendering, draw path in all dirty areas*/
	else {
		M4IRect clip;
		for (i=0; i<surf->to_redraw.count; i++) {
			/*there's an opaque region above, don't draw*/
			if (surf->draw_node_index<surf->to_redraw.opaque_node_index[i]) continue;
			clip = ctx->clip;
			m4_irect_intersect(&clip, &surf->to_redraw.list[i]);
			if (clip.width && clip.height) {
				surf->render->g_hw->surface_set_clipper(surf->the_surface, &clip);
				surf->render->g_hw->surface_fill(surf->the_surface, stencil);
			}
		}
	}
}

void VS2D_SetOptions(Render2D *sr, LPM4SURFACE rend, Bool forText, Bool no_antialias)
{
	if (no_antialias) {
		sr->g_hw->surface_set_raster_level(rend, sr->compositor->high_speed ? M4RasterHighSpeed : M4RasterInter);
	} else {
		switch (sr->compositor->antiAlias) {
		case M4_AL_None:
			sr->g_hw->surface_set_raster_level(rend, M4RasterHighSpeed);
			break;
		case M4_AL_Text:
			if (forText) {
				sr->g_hw->surface_set_raster_level(rend, M4RasterHighQuality);
			} else {
				sr->g_hw->surface_set_raster_level(rend, sr->compositor->high_speed ? M4RasterHighSpeed : M4RasterInter);
			}
			break;
		case M4_AL_All:
		default:
			sr->g_hw->surface_set_raster_level(rend, M4RasterHighQuality);
			break;
		}
	}
}


void VS2D_DrawPath(VisualSurface2D *surf, LPM4PATH path, DrawableContext *ctx, LPM4STENCIL brush, LPM4STENCIL pen)
{
	Float width, orig_width;
	Bool dofill, dostrike;
#ifdef SKIP_DRAW
	return;
#endif

	if (!surf->the_surface) return;
	if (ctx->path_filled && ctx->path_stroke) {
		if (surf->render->compositor->draw_bvol) draw_clipper(surf, ctx);
		return;
	}

	if (!ctx->is_background) VS2D_SetOptions(surf->render, surf->the_surface, ctx->is_text, 0);

	dofill = dostrike = 0;
	if (!ctx->path_filled && ctx->aspect.filled) {
		dofill = 1;
		if (!brush) {
			brush = surf->the_brush;
			surf->render->g_hw->set_brush_color(brush, ctx->aspect.fill_color);
		}
	}

	/*compute width based on transform and top_level transform*/
	width = orig_width = ctx->aspect.pen_props.width;
	if (!ctx->path_stroke && orig_width) {
		dostrike = 1;
		if (!pen) {
			pen = surf->the_pen;
			surf->render->g_hw->set_brush_color(pen, ctx->aspect.line_color);
		}
		if (ctx->aspect.pen_props.is_scalable) {
			ctx->aspect.line_scale = 1;
		} else {
			SFVec2f pt;
			pt.x = ctx->transform.m[0] + ctx->transform.m[1];
			pt.y = ctx->transform.m[3] + ctx->transform.m[4];
			ctx->aspect.line_scale = (Float) (sqrt(pt.x*pt.x + pt.y*pt.y) / sqrt(2));
		}
	}

	/*set path transform, except for background2D node which is directly build in the final coord system*/
	surf->render->g_hw->surface_set_matrix(surf->the_surface, ctx->is_background ? NULL : &ctx->transform);


	/*fill path*/
	if (dofill) {
		/*push path*/
		surf->render->g_hw->surface_set_path(surf->the_surface, path);
		VS2D_DoFill(surf, ctx, brush);
		surf->render->g_hw->surface_set_path(surf->the_surface, NULL);
	}

	if (dostrike) {
		StrikeInfo2D *si = drawctx_get_strikeinfo(ctx, path);
		if (si && si->outline) {
			surf->render->g_hw->surface_set_path(surf->the_surface, si->outline);
			VS2D_DoFill(surf, ctx, pen);
			/*that's ugly, but we cannot cache path outline for IFS2D/ILS2D*/
			if (path && !ctx->is_text) {
				m4_path_delete(si->outline);
				si->outline = NULL;
			}
		}
	}

	if (surf->render->compositor->draw_bvol) draw_clipper(surf, ctx);
}


void get_texture_transform(struct _drawable_context *ctx, M4Matrix2D *mat, Bool line_texture, Float final_width, Float final_height )
{
#ifdef M4_DEF_Appearance 
	u32 node_tag;
	B_Appearance *appear;		
	mx2d_init(*mat);

	if (!ctx->appear) return;
	appear = (B_Appearance *)ctx->appear;

	if (!line_texture) {
		if (!appear->textureTransform) return;

		/*gradient doesn't need bounds info in texture transform*/
		if (ctx->h_texture->compute_gradient_matrix) {
			final_width = final_height = 1.0;
		}
		
		node_tag = Node_GetTag((SFNode *) appear->textureTransform);
#ifdef M4_DEF_TextureTransform
		if (node_tag==TAG_TextureTransform) {
			/*VRML: Tc' = -C  S  R  C  T  Tc*/
			B_TextureTransform *txt = (B_TextureTransform *) appear->textureTransform;
			SFVec2f scale = txt->scale;
			if (!scale.x) scale.x = 0.01f;
			if (!scale.y) scale.y = 0.01f;

			mx2d_add_translation(mat, -txt->center.x * final_width, -txt->center.y * final_height);
			mx2d_add_scale(mat, scale.x, scale.y);
			mx2d_add_rotation(mat, 0, 0, txt->rotation);
			mx2d_add_translation(mat, txt->center.x * final_width, txt->center.y * final_height);
			mx2d_add_translation(mat, txt->translation.x * final_width, txt->translation.y * final_height);
			/*and inverse the matrix (this is texture transform, cf VRML)*/
			mx2d_inverse(mat);
			return;
		}
#endif

#ifdef M4_DEF_TransformMatrix2D
		if (node_tag==TAG_TransformMatrix2D) {
			TM2D_GetMatrix((SFNode *) appear->textureTransform, mat);
			mat->m[2] *= final_width;
			mat->m[5] *= final_height;
			mx2d_inverse(mat);
			return;
		}
#endif
	}

#endif

}


static void VS2D_DrawGradient(VisualSurface2D *surf, LPM4PATH path, struct _drawable_context *ctx)
{
	M4Rect rc;
	Bool fill;
	M4Matrix2D g_mat, txt_mat;

	rc = ctx->original;
	if (!rc.width || !rc.height || !ctx->h_texture->hwtx) return;
	ctx->h_texture->compute_gradient_matrix(ctx->h_texture, &rc, &g_mat);


	get_texture_transform(ctx, &txt_mat, 0, ctx->h_texture->active_window.width, ctx->h_texture->active_window.height);
	mx2d_add_matrix(&g_mat, &txt_mat);
	mx2d_add_matrix(&g_mat, &ctx->transform);

	surf->render->g_hw->stencil_set_matrix(ctx->h_texture->hwtx, &g_mat);

	fill = ctx->aspect.filled;
	ctx->aspect.filled = 1;
	VS2D_DrawPath(surf, path, ctx, ctx->h_texture->hwtx, NULL);
	ctx->aspect.filled = fill;
	ctx->path_filled = 1;
}


void VS2D_AddMeterMetrics(VisualSurface2D *surf, M4Matrix2D *mat)
{
	Float w, h;

	if (surf->is_pixel_metrics) return;
	surf->GetPixelSize(surf, &w, &h);
	w = 2.0f / w;
	h = 2.0f / h;
	mx2d_add_scale(mat, w, h);
}

/*
	Note: this may need improvements wrt how texturing is handled by the graphics plugin. 
	For	now all transformations are computed locally (including repeatS/T) and the final
	transform is applied to the texture brush
*/
void VS2D_TexturePath(VisualSurface2D *surf, LPM4PATH path, struct _drawable_context *ctx)
{
	Float sS, sT;
	M4Matrix2D mx2d_txt, texture_transform, mat;
	M4Rect rc, orig_rc;
	
#ifdef SKIP_DRAW
	return;
#endif
	if (!surf->the_surface || ctx->path_filled || !ctx->h_texture || surf->render->compositor->is_hidden) return;

	/*this is ambiguous in the spec, what if the material is filled and the texture is transparent ?
	let's draw, it's nicer */
	if (ctx->aspect.filled && ctx->h_texture->transparent) {
		VS2D_DrawPath(surf, path, ctx, NULL, NULL);
		ctx->path_filled = 0;
	}

	/*this is gradient draw*/
	if (ctx->h_texture->compute_gradient_matrix) {
		VS2D_DrawGradient(surf, path, ctx);
		return;
	}

	if (!ctx->h_texture->hwtx) return;

	/*setup quality even for background (since quality concerns images)*/
	VS2D_SetOptions(surf->render, surf->the_surface, ctx->is_text, ctx->no_antialias);

	/*get original bounds*/
	orig_rc = ctx->original;

	/*get active texture window in pixels*/
	rc = ctx->h_texture->active_window;

	/*if meter metrics get equivalent image size in meter*/
	if (!surf->is_pixel_metrics) {
		mx2d_init(mat);
		VS2D_AddMeterMetrics(surf, &mat);
		mx2d_apply_rect(&mat, &rc);
	}


	/*get scaling ratio so that active texture view is stretched to original bounds (std 2D shape texture mapping in MPEG4)*/
	sS = orig_rc.width / rc.width;
	sT = orig_rc.height / rc.height;
	
	mx2d_init(mx2d_txt);
	mx2d_add_scale(&mx2d_txt, sS, sT);
	/*apply texture transform*/
	get_texture_transform(ctx, &texture_transform, 0, ctx->h_texture->active_window.width * sS, ctx->h_texture->active_window.height * sT);
	mx2d_add_matrix(&mx2d_txt, &texture_transform);

	/*add meterMetrics transform (reverse transform meter->pixel to compensate the context transform)*/
	VS2D_AddMeterMetrics(surf, &mx2d_txt);
	/*move to bottom-left corner of bounds */
	mx2d_add_translation(&mx2d_txt, (orig_rc.x), (orig_rc.y - orig_rc.height));

	/*move to final coordinate system (except background which is built directly in final coord system)*/	
	if (!ctx->is_background) mx2d_add_matrix(&mx2d_txt, &ctx->transform);

	/*set path transform, except for background2D node which is directly build in the final coord system*/
	surf->render->g_hw->stencil_set_matrix(ctx->h_texture->hwtx, &mx2d_txt);
	surf->render->g_hw->set_texture_view(ctx->h_texture->hwtx, &ctx->h_texture->active_window);

	if (!ctx->is_background) {
		surf->render->g_hw->set_texture_alpha(ctx->h_texture->hwtx, M4C_A(ctx->aspect.fill_color) );
		surf->render->g_hw->surface_set_matrix(surf->the_surface, &ctx->transform);
	} else {
		surf->render->g_hw->surface_set_matrix(surf->the_surface, NULL);
	}

	/*push path*/
	surf->render->g_hw->surface_set_path(surf->the_surface, path);

	VS2D_DoFill(surf, ctx, ctx->h_texture->hwtx);

	surf->render->g_hw->surface_set_path(surf->the_surface, NULL);

	ctx->path_filled = 1;
}
