/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */



#include "stacks2d.h"
#include "visualsurface2d.h"

#ifdef M4_DEF_Text

#ifndef M4_DEF_FontStyle
#error "FontStyle MUST be defined when enabling Text"
#endif


/*default value when no fontStyle*/
#define FSFAMILY	fs ? (const char *)fs->family.vals[0]	: ""

/*here it's tricky since it depends on our metric system...*/
#define FSSIZE		(fs ? fs->size : -1)
#define FSSTYLE		(fs ? (const char *)fs->style.buffer : "")
#define FSMAJOR		( (fs && fs->justify.count && fs->justify.vals[0]) ? (const char *)fs->justify.vals[0] : "FIRST")
#define FSMINOR		( (fs && (fs->justify.count>1) && fs->justify.vals[1]) ? (const char *)fs->justify.vals[1] : "FIRST")
#define FSHORIZ		(fs ? fs->horizontal : 1)
#define FSLTR		(fs ? fs->leftToRight : 1)
#define FSTTB		(fs ? fs->topToBottom : 1)
#define FSLANG		(fs ? fs->language : "")
#define FSSPACE		(fs ? fs->spacing : 1)


void text_get_ascent_descent(DrawableContext *ctx, Float *a, Float *d)
{
	TextStack *stack = (TextStack *) Node_GetPrivate(ctx->node->owner);
	*a = stack->ascent;
	*d = stack->descent;
}

static void clean_paths(TextStack *stack, Graphics2DDriver *dr)
{
	LPM4PATH path;
	/*delete all path objects*/
	while (ChainGetCount(stack->text_lines)) {
		path = ChainGetEntry(stack->text_lines, 0);
		ChainDeleteEntry(stack->text_lines, 0);
		m4_path_delete(path);
	}
	m4_rect_reset(&stack->bounds);
	drawable_reset_path(stack->graph);
}

static void DestroyText(SFNode *node)
{
	TextStack *stack = (TextStack *) Node_GetPrivate(node);
	Render2D *sr = (Render2D *) stack->graph->compositor->visual_renderer->user_priv;
	clean_paths(stack, sr->g_hw);
	DeleteDrawableNode(stack->graph);
	DeleteChain(stack->text_lines);
	free(stack);
}

static void split_text_letters(TextStack *st, B_Text *txt, RenderEffect *eff)
{
	LPM4PATH path;
	unsigned short wcTemp[5000];
	unsigned short letter[2];
	DrawableContext *ctx;
	u32 i, j, len;
	Float fontSize, start_y, font_height, line_spacing;
	M4Rect rc;
	FontRaster *ft_dr = eff->surface->render->compositor->font_engine;
	B_FontStyle *fs = (B_FontStyle *)txt->fontStyle;

	fontSize = FSSIZE;
	if (fontSize <= 0) {
		fontSize = 12;
		if (!R2D_IsPixelMetrics((SFNode *)txt)) fontSize /= eff->surface->render->cur_width;
    }
	line_spacing = FSSPACE * fontSize;

	if (ft_dr->set_font(ft_dr, FSFAMILY, FSSTYLE) != M4OK) {
		if (ft_dr->set_font(ft_dr, NULL, FSSTYLE) != M4OK) {
			return;
		}
	}
	ft_dr->set_font_size(ft_dr, fontSize);
	ft_dr->get_font_metrics(ft_dr, &st->ascent, &st->descent, &font_height);
	/*adjust size*/
	fontSize *= fontSize / (font_height);
	ft_dr->set_font_size(ft_dr, fontSize);
	ft_dr->get_font_metrics(ft_dr, &st->ascent, &st->descent, &font_height);

			
	if (!strcmp(FSMINOR, "MIDDLE")) {
		start_y = (st->descent + st->ascent)/2;
	}
	else if (!strcmp(FSMINOR, "BEGIN")) {
		start_y = st->descent;
	}
	else if (!strcmp(FSMINOR, "END")) {
		start_y = st->descent + st->ascent;
	}
	else {
		start_y = st->ascent;
	}
	
	m4_rect_reset(&st->bounds);

	for (i=0; i < txt->string.count; i++) {

		char *str = txt->string.vals[i];
		if (!str) continue;
		len = utf8_mbstowcs(wcTemp, 5000, (const char **) &str);
		if (len == (size_t) (-1)) continue;

		letter[1] = (unsigned short) 0;
		for (j=0; j<len; j++) {
			if (FSLTR) {
				letter[0] = wcTemp[j];
			} else {
				letter[0] = wcTemp[len - j - 1];
			}
			/*request a context (first one is always valid when entering render)*/
			if (j) group2d_start_child(eff->parent);
			ctx = drawable_init_context(st->graph, eff);
			if (!ctx) return;

			ctx->is_text = 1;
			path = m4_new_path();

			ChainAddEntry(st->text_lines, path);
			ctx->sub_path_index = ChainGetCount(st->text_lines);

			ft_dr->add_text_to_path(ft_dr, path, letter, 0, start_y, 1, 1, st->ascent, &rc);
			ctx->original.width = rc.width;
			ctx->original.x = rc.x;
			ctx->original.height = MAX(st->ascent + st->descent, rc.height);
			ctx->original.y = start_y;

			drawable_finalize_render(ctx, eff);
			group2d_end_child(eff->parent);
		}
	}
}

static void split_text_words(TextStack *st, B_Text *txt, RenderEffect *eff)
{
	LPM4PATH path;
	unsigned short wcTemp[5000];
	unsigned short letter[5000];
	DrawableContext *ctx;
	u32 i, j, len, k, first_char;
	Float fontSize, font_height, line_spacing;
	M4Rect rc;
	FontRaster *ft_dr = eff->surface->render->compositor->font_engine;
	B_FontStyle *fs = (B_FontStyle *)txt->fontStyle;

	fontSize = FSSIZE;
	if (fontSize <= 0) {
		fontSize = 12;
		if (!R2D_IsPixelMetrics((SFNode *)txt)) fontSize /= eff->surface->render->cur_width;
    }
	line_spacing = FSSPACE * fontSize;

	if (ft_dr->set_font(ft_dr, FSFAMILY, FSSTYLE) != M4OK) {
		if (ft_dr->set_font(ft_dr, NULL, FSSTYLE) != M4OK) {
			return;
		}
	}
	ft_dr->set_font_size(ft_dr, fontSize);
	ft_dr->get_font_metrics(ft_dr, &st->ascent, &st->descent, &font_height);
	/*adjust size*/
	fontSize *= fontSize / (font_height);
	ft_dr->set_font_size(ft_dr, fontSize);
	ft_dr->get_font_metrics(ft_dr, &st->ascent, &st->descent, &font_height);

	m4_rect_reset(&st->bounds);

	for (i=0; i < txt->string.count; i++) {
		char *str = txt->string.vals[i];
		if (!str) continue;
		len = utf8_mbstowcs(wcTemp, 5000, (const char **) &str);
		if (len == (size_t) (-1)) continue;

		first_char = 0;
		for (j=0; j<len; j++) {
			/*we currently only split sentences at spaces*/
			if ((j+1!=len) && (wcTemp[j] != (unsigned short) ' ')) continue;

			if (FSLTR) {
				for (k=0; k<=j - first_char; k++) letter[k] = wcTemp[first_char+k];
			} else {
				for (k=0; k<=j - first_char; k++) letter[k] = wcTemp[len - first_char - k - 1];
			}
			letter[k] = (unsigned short) 0;
			/*request a context (first one is always valid when entering render)*/
			if (first_char) group2d_start_child(eff->parent);
			
			ctx = drawable_init_context(st->graph, eff);
			if (!ctx) return;

			ctx->is_text = 1;
			path = m4_new_path();

			ChainAddEntry(st->text_lines, path);
			ctx->sub_path_index = ChainGetCount(st->text_lines);

			/*word splitting only happen in layout, so we don't need top/left anchors*/
			ft_dr->add_text_to_path(ft_dr, path, letter, 0, 0, 1, 1, st->ascent, &rc);
			m4_path_get_bounds(path, &ctx->original);
			ctx->original.width = rc.width;
			if (ctx->original.x != 0) ctx->original.width -= ctx->original.x;
			ctx->original.x = 0;

			ctx->original.height = MAX(st->ascent + st->descent, ctx->original.height);
			if (ctx->original.y != 0) ctx->original.height -= ctx->original.y;
			ctx->original.y = 0;

			drawable_finalize_render(ctx, eff);
			group2d_end_child(eff->parent);

			first_char = j+1;
		}
	}
}

typedef struct
{
	unsigned short *wcText;
	u32 length;
	Float width, height;
	Float x_scaling, y_scaling;
} TextLine;

static void BuildVerticalTextGraph(TextStack *st, B_Text *txt, RenderEffect *eff)
{
	TextLine *lines;
	LPM4PATH path;
	unsigned short wcTemp[5000];
	u32 i, int_major, len, k;
	Float fontSize, start_x, start_y, space, line_spacing, tot_width, tot_height, max_scale;
	M4Rect rc, final;
	Float lw, lh, max_lw;
	unsigned short letter[2];
	FontRaster *ft_dr = eff->surface->render->compositor->font_engine;
	B_FontStyle *fs = (B_FontStyle *)txt->fontStyle;

	fontSize = FSSIZE;
	if (fontSize <= 0) {
		fontSize = 12;
		if (!R2D_IsPixelMetrics((SFNode *)txt)) fontSize /= eff->surface->render->cur_width;
    }

	line_spacing = FSSPACE * fontSize;

	if (ft_dr->set_font(ft_dr, FSFAMILY, FSSTYLE) != M4OK) {
		if (ft_dr->set_font(ft_dr, NULL, FSSTYLE) != M4OK) {
			return;
		}
	}
	ft_dr->set_font_size(ft_dr, fontSize);
	ft_dr->get_font_metrics(ft_dr, &st->ascent, &st->descent, &space);


	/*compute overall bounding box size*/
	tot_width = 0;
	tot_height = 0;
	lines = (TextLine *) malloc(sizeof(TextLine)*txt->string.count);
	memset(lines, 0, sizeof(TextLine)*txt->string.count);
		
	letter[1] = (unsigned short) '\0';

/*	space = st->ascent + st->descent;
	space = fontSize - st->ascent + st->descent;
*/
	for (i=0; i < txt->string.count; i++) {
		char *str = txt->string.vals[i];
		if (!str) continue;
		lines[i].length = 0;
		len = utf8_mbstowcs(wcTemp, 5000, (const char **) &str);
		if (len == (size_t) (-1)) continue;

		lines[i].wcText = malloc(sizeof(unsigned short) * len);
		memcpy(lines[i].wcText, wcTemp, sizeof(unsigned short) * len);
		lines[i].length = len;
		
		lines[i].y_scaling = lines[i].x_scaling = 1.0;
		lines[i].height = len * space;
		if (!lines[i].height) continue;

		if ((txt->length.count>i) && (txt->length.vals[i]>0) ) 
			lines[i].y_scaling = txt->length.vals[i] / lines[i].height;
	
		if (tot_height < lines[i].height * lines[i].y_scaling) tot_height = lines[i].height * lines[i].y_scaling;
	}

	tot_width = txt->string.count * line_spacing;
	st->bounds.width = tot_width;

	max_scale = 1.0;
	if ((txt->maxExtent>0) && (tot_height>txt->maxExtent)) {
		max_scale = txt->maxExtent / tot_height;
		tot_height = txt->maxExtent;
	}

	if (!strcmp(FSMINOR, "MIDDLE")) {
		if (FSLTR) {
			start_x = -tot_width/2;
			st->bounds.x = start_x;
		} else {
			start_x = tot_width/2 - line_spacing;
			st->bounds.x = - tot_width + line_spacing;
		}
	}
	else if (!strcmp(FSMINOR, "END")) {
		if (FSLTR) {
			start_x = -tot_width;
			st->bounds.x = start_x;
		} else {
			start_x = tot_width-line_spacing;
			st->bounds.x = 0;
		}
	}
	else {
		if (FSLTR) {
			start_x = 0;
			st->bounds.x = start_x;
		} else {
			start_x = -line_spacing;
			st->bounds.x = -tot_width;
		}
	}

	if (!strcmp(FSMAJOR, "MIDDLE")) {
		int_major = 0;
	} else if (!strcmp(FSMAJOR, "END")) {
		int_major = 1;
	} else {
		int_major = 2;
	}

	m4_rect_reset(&final);
	for (i=0; i < txt->string.count; i++) {
		switch (int_major) {
		case 0:
			if (FSTTB) 
				start_y = lines[i].height/2;
			else
				start_y = -lines[i].height/2 + space;
			break;
		case 1:
			if (FSTTB)
				start_y = lines[i].height;
			else
				start_y = -lines[i].height + space;
			break;
		default:
			if (FSTTB)
				start_y = 0;
			else
				start_y = space;
			break;
		}

		if (lines[i].length) {
			path = m4_new_path();
			ChainAddEntry(st->text_lines, path);

			/*adjust horizontal offset on first column*/
			if (!i) {
				max_lw = 0;
				for (k=0; k<lines[i].length; k++) {
					letter[0] = lines[i].wcText[k];
					/*get glyph width so that all letters are centered on the same vertical line*/
					ft_dr->get_text_size(ft_dr, letter, &lw, &lh);
					if (max_lw < lw) max_lw = lw;
				}
				st->bounds.width += max_lw/2;
				start_x += max_lw/2;
			}
			
			for (k=0; k<lines[i].length; k++) {
				letter[0] = lines[i].wcText[k];
				/*get glyph width so that all letters are centered on the same vertical line*/
				ft_dr->get_text_size(ft_dr, letter, &lw, &lh);
				ft_dr->add_text_to_path(ft_dr, path, letter, start_x - lw/2, start_y, lines[i].x_scaling, lines[i].y_scaling*max_scale, st->ascent, &rc);

				if (FSTTB)
					start_y -= space;
				else
					start_y += space;
			}
			m4_path_get_bounds(path, &rc);
			m4_rect_union(&final, &rc);
		}

		if (FSLTR) {
			start_x += line_spacing;
		} else {
			start_x -= line_spacing;
		}

		if (eff->parent) {
			rc = final;
			mx2d_apply_rect(&eff->transform, &rc);
			if (FSLTR && (rc.x > eff->surface->top_clipper.x + eff->surface->top_clipper.width) ) {
				break;
			}
			else if (!FSLTR && (rc.x + rc.width < eff->surface->top_clipper.x) ) {
				break;
			}
		}

		/*free unicode buffer*/
		free(lines[i].wcText);
	}

	/*free remaining unicode buffers*/
	for (; i < txt->string.count; i++) free(lines[i].wcText);

	free(lines);

	st->bounds.height = final.height;
	st->bounds.y = final.y;
}


static void BuildTextGraph(TextStack *st, B_Text *txt, RenderEffect *eff)
{
	TextLine *lines;
	LPM4PATH path;
	unsigned short wcTemp[5000];
	u32 i, int_major, len, k;
	Float fontSize, start_x, start_y, font_height, line_spacing, tot_width, tot_height, max_scale;
	M4Rect rc, final;
	FontRaster *ft_dr = eff->surface->render->compositor->font_engine;
	B_FontStyle *fs = (B_FontStyle *)txt->fontStyle;

	if (!FSHORIZ) {
		BuildVerticalTextGraph(st, txt, eff);
		return;
	}

	fontSize = FSSIZE;
	if (fontSize <= 0) {
		fontSize = 12;
		if (!R2D_IsPixelMetrics((SFNode *)txt)) fontSize /= eff->surface->render->cur_width;
    }

	if (ft_dr->set_font(ft_dr, FSFAMILY, FSSTYLE) != M4OK) {
		if (ft_dr->set_font(ft_dr, NULL, FSSTYLE) != M4OK) {
			return;
		}
	}
	ft_dr->set_font_size(ft_dr, fontSize);
	ft_dr->get_font_metrics(ft_dr, &st->ascent, &st->descent, &font_height);

	/*spacing= FSSPACING * fontSize and fontSize adjusted to m-ascent+m-descent (removed COR)*/
#if 0
	line_spacing = FSSPACE * fontSize;
	/*adjust size*/
	fontSize *= fontSize / (font_height);
	ft_dr->set_font_size(ft_dr, fontSize);
	ft_dr->get_font_metrics(ft_dr, &st->ascent, &st->descent, &font_height);
#else
	/*spacing= FSSPACING * (font_height) and fontSize not adjusted */
//	line_spacing = FSSPACE * (font_height);
	line_spacing = FSSPACE * fontSize;
#endif
	
	tot_width = 0;
	lines = (TextLine *) malloc(sizeof(TextLine)*txt->string.count);
	memset(lines, 0, sizeof(TextLine)*txt->string.count);
	
	for (i=0; i < txt->string.count; i++) {
		char *str = txt->string.vals[i];
		if (!str) continue;
		lines[i].length = 0;
		len = utf8_mbstowcs(wcTemp, 5000, (const char **) &str);
		if (len == (size_t) (-1)) continue;

		lines[i].length = len;
		lines[i].wcText = malloc(sizeof(unsigned short) * (len+1));
		if (!FSLTR) {
			for (k=0; k<len; k++) lines[i].wcText[k] = wcTemp[len-k-1];
		} else {
			memcpy(lines[i].wcText, wcTemp, sizeof(unsigned short) * len);
		}
		lines[i].wcText[len] = (unsigned short) '\0';

		lines[i].y_scaling = lines[i].x_scaling = 1.0;
		ft_dr->get_text_size(ft_dr, lines[i].wcText, &lines[i].width, &lines[i].height);

		if (!lines[i].width) continue;
		if ((txt->length.count>i) && (txt->length.vals[i]>0)) {
			lines[i].x_scaling = txt->length.vals[i] / lines[i].width;
		}
		if (tot_width < lines[i].width * lines[i].x_scaling) tot_width = lines[i].width * lines[i].x_scaling;
	}
	
	max_scale = 1.0;
	if ((txt->maxExtent > 0) && (tot_width>txt->maxExtent)) {
		max_scale = txt->maxExtent / tot_width;
		tot_width = txt->maxExtent;
	}

	tot_height = (txt->string.count-1) * line_spacing + (st->ascent + st->descent);
	st->bounds.height = tot_height;
	
	if (!strcmp(FSMINOR, "MIDDLE")) {
		if (FSTTB) {
			start_y = tot_height/2;
			st->bounds.y = start_y;
		} else {
			start_y = st->descent + st->ascent - tot_height/2;
			st->bounds.y = tot_height/2;
		}
	}
	else if (!strcmp(FSMINOR, "BEGIN")) {
		if (FSTTB) {
			start_y = st->descent;
			start_y = 0;
			st->bounds.y = start_y;
		} else {
			st->bounds.y = st->bounds.height;
			start_y = st->descent + st->ascent;
		}
	}
	else if (!strcmp(FSMINOR, "END")) {
		if (FSTTB) {
			start_y = tot_height;
			st->bounds.y = start_y;
		} else {
			start_y = -tot_height + 2*st->descent + st->ascent;
			st->bounds.y = start_y - (st->descent + st->ascent) + tot_height;
		}
	}
	else {
		start_y = st->ascent;
		st->bounds.y = FSTTB ? start_y : (tot_height - st->descent);
	}
	
	/*major-justification*/
	if (!strcmp(FSMAJOR, "MIDDLE") ) {
		int_major = 0;
	} else if (!strcmp(FSMAJOR, "END") ) {
		int_major = 1;
	} else {
		int_major = 2;
	}
	m4_rect_reset(&final);


	for (i=0; i < txt->string.count; i++) {
		switch (int_major) {
		/*major-justification MIDDLE*/
		case 0:
			start_x = -lines[i].width/2;
			break;
		/*major-justification END*/
		case 1:
			start_x = (FSLTR) ? -lines[i].width : 0;
			break;
		/*BEGIN, FIRST or default*/
		default:
			start_x = (FSLTR) ? 0 : -lines[i].width;
			break;
		}

		if (lines[i].length) {
			path = m4_new_path();

			/*if using the font engine the font is already configured*/
			ft_dr->add_text_to_path(ft_dr, path, lines[i].wcText, start_x, start_y, lines[i].x_scaling*max_scale, lines[i].y_scaling, st->ascent, &rc);

			ChainAddEntry(st->text_lines, path);
			m4_rect_union(&final, &rc);
		}
		if (FSTTB) {
			start_y -= line_spacing;
		} else {
			start_y += line_spacing;
		}

		if (! eff->parent) {
			rc = final;
			mx2d_apply_rect(&eff->transform, &rc);
			if (FSTTB && (rc.y < eff->surface->top_clipper.y - eff->surface->top_clipper.height) ) {
				break;
			}
			else if (! FSTTB && (rc.y - rc.height > eff->surface->top_clipper.y) ) {
				break;
			}
		}

		/*free unicode buffer*/
		free(lines[i].wcText);
	}
	/*free remaining unicode buffers*/
	for (; i < txt->string.count; i++) free(lines[i].wcText);

	free(lines);

	st->bounds.width = final.width;
	st->bounds.x = final.x;
}


static void Text_Render(SFNode *n, void *rs)
{
	DrawableContext *ctx;
	B_Text *txt = (B_Text *) n;
	TextStack *st = (TextStack *) Node_GetPrivate(n);
	RenderEffect *eff = (RenderEffect *)rs;

	if (!st->graph->compositor->font_engine) return;

	if (!txt->string.count) return;

	if (eff->text_split_mode == 2) {
		split_text_letters(st, txt, eff);
		return;
	}
	else if (eff->text_split_mode == 1) {
		split_text_words(st, txt, eff);
		return;
	}

	/*check for geometry change*/
	if (Node_GetDirty(n)) {
		clean_paths(st, eff->surface->render->g_hw);
		BuildTextGraph(st, txt, eff);
		Node_ClearDirty(n);
		st->graph->node_changed = 1;
	}

	/*get the text bounds*/
	ctx = drawable_init_context(st->graph, eff);
	if (!ctx) return;

	/*store bounds*/
	ctx->original = st->bounds;

	ctx->is_text = 1;
	if (!ctx->aspect.filled) {
		/*override line join*/
		ctx->aspect.pen_props.join = M4LineJoinMiter;
		ctx->aspect.pen_props.cap = M4LineCapFlat;
	}
	drawable_finalize_render(ctx, eff);
}

static void Text_Draw(DrawableContext *ctx)
{
	u32 i;
	LPM4PATH path;
	TextStack *st = (TextStack *) Node_GetPrivate((SFNode *) ctx->node->owner);

	/*text has been splited*/
	if (ctx->sub_path_index > 0) {
		path = ChainGetEntry(st->text_lines, ctx->sub_path_index - 1);
		if (!path) return;
		VS2D_TexturePath(ctx->surface, path, ctx);
		VS2D_DrawPath(ctx->surface, path, ctx, NULL, NULL);
		return;
	}

	for (i=0; i<ChainGetCount(st->text_lines); i++) {
		path = ChainGetEntry(st->text_lines, i);
		VS2D_TexturePath(ctx->surface, path, ctx);
		VS2D_DrawPath(ctx->surface, path, ctx, NULL, NULL);
		/*reset fill/strike flags since we perform several draw per context*/
		ctx->path_filled = ctx->path_stroke = 0;
	}
}

static Bool Text_PointOver(DrawableContext *ctx, Float x, Float y, Bool check_outline)
{
	LPM4PATH path;
	M4Matrix2D inv;
	u32 i;
	TextStack *st;
	/*this is not documented anywhere but it speeds things up*/
	if (!check_outline) return 1;
	
	
	st = (TextStack *) Node_GetPrivate((SFNode *) ctx->node->owner);
	
	mx2d_copy(inv, ctx->transform);
	mx2d_inverse(&inv);
	mx2d_apply_coords(&inv, &x, &y);

	/*otherwise get all paths*/
	if (ctx->sub_path_index > 0) {
		path = ChainGetEntry(st->text_lines, ctx->sub_path_index - 1);
		if (!path) return 0;
		return m4_path_point_over(path, x, y);
	}

	for (i=0; i<ChainGetCount(st->text_lines); i++) {
		path = ChainGetEntry(st->text_lines, i);
		if (!path) return 0;
		if (m4_path_point_over(path, x, y)) return 1;
	}
	return 0;
}

void R2D_InitText(Render2D *sr, SFNode *node)
{
	TextStack *stack = malloc(sizeof(TextStack));
	stack->graph = NewDrawableNode();
	/*override all funct*/
	stack->graph->Draw = Text_Draw;
	stack->graph->IsPointOver = Text_PointOver;
	stack->ascent = stack->descent = 0;
	stack->text_lines = NewChain();
	
	traversable_setup(stack->graph, node, sr->compositor);
	Node_SetPrivate(node, stack);
	Node_SetRenderFunction(node, Text_Render);
	Node_SetPreDestroyFunction(node, DestroyText);
}


#endif

