/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Graph sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

/*math.h is not included in main config (C++ clash on win32)*/
#include <math.h>
#include <gpac/intern/m4_scenegraph_dev.h>
#include <gpac/intern/m4_node_tables.h>

#ifdef M4_DEF_Script
#ifdef M4_USE_SPIDERMONKEY

static JSClass globalClass;
static JSClass browserClass;
static JSClass SFNodeClass;
static JSClass SFVec2fClass;
static JSClass SFVec3fClass;
static JSClass SFColorClass;
static JSClass MFInt32Class;
static JSClass MFBoolClass;
static JSClass MFFloatClass;
static JSClass MFTimeClass;
static JSClass MFVec2fClass;
static JSClass MFVec3fClass;
static JSClass MFRotationClass;
static JSClass MFColorClass;
static JSClass MFStringClass;
static JSClass MFUrlClass;
static JSClass MFNodeClass;


static M4INLINE JSInterface *JS_GetInterface(JSContext *c)
{
	SFNode *n = JS_GetContextPrivate(c);
	return n->sgprivate->scenegraph->js_ifce;
}

static M4INLINE JSField *NewJSField(Bool attached)
{
	JSField *ptr;
	SAFEALLOC(ptr, sizeof(JSField));
	ptr->is_attached = attached;
	return ptr;
}

static M4INLINE B_Script *JS_GetScript(JSContext *c)
{
	return (B_Script *) JS_GetContextPrivate(c);
}
static M4INLINE ScriptPriv *JS_GetScriptStack(JSContext *c)
{
	B_Script *script = (B_Script *) JS_GetContextPrivate(c);
	return script->sgprivate->privateStack;
}

static void script_error(JSContext *c, const char *msg, JSErrorReport *jserr)
{
	JSInterface *ifce = JS_GetInterface(c);
	if (ifce)
		ifce->Error(ifce->callback, msg);
}

static JSBool JSPrint(JSContext *c, JSObject *p, uintN argc, jsval *argv, jsval *rval)
{
	u32 i;
	char buf[5000];
	JSInterface *ifce = JS_GetInterface(c);
	if (!ifce) return JS_FALSE;

	strcpy(buf, "");
	for (i = 0; i < argc; i++) {
		JSString *str = JS_ValueToString(c, argv[i]);
		if (!str) return JS_FALSE;
		if (i) strcat(buf, " ");
		strcat(buf, JS_GetStringBytes(str));
	}
	ifce->Print(ifce->callback, buf);
	return JS_TRUE;
}

static JSFunctionSpec globalFunctions[] = {
    {"print",           JSPrint,          0},
    { 0 }
};

static JSBool getName(JSContext *c, JSObject *obj, uintN n, jsval *v, jsval *rval)
{
	const char *name;
	JSInterface *ifce = JS_GetInterface(c);
	if (!ifce) return 0;
	name = ifce->GetName(ifce->callback);
	*rval = STRING_TO_JSVAL(JS_InternString(c, name ));
	return JS_TRUE;
}
static JSBool getVersion(JSContext*c, JSObject*obj, uintN n, jsval *v, jsval *rval)
{
	const char *name;
	JSInterface *ifce = JS_GetInterface(c);
	if (!ifce) return JS_FALSE;
	name = ifce->GetVersion(ifce->callback);
	*rval = STRING_TO_JSVAL(JS_InternString( c, name ));
	return JS_TRUE;
}
static JSBool getCurrentSpeed(JSContext *c, JSObject *o, uintN n, jsval *v, jsval *rval)
{
	*rval = DOUBLE_TO_JSVAL(JS_NewDouble( c, 30.0 ));
	return JS_TRUE;
}
static JSBool getCurrentFrameRate(JSContext *c, JSObject*o, uintN n, jsval *v, jsval*rval)
{
	*rval = DOUBLE_TO_JSVAL(JS_NewDouble(c, 30.0 ));
	return JS_TRUE;
}
static JSBool getWorldURL(JSContext*c, JSObject*obj, uintN n, jsval *v, jsval *rval)
{
	const char *name;
	JSInterface *ifce = JS_GetInterface(c);
	if (!ifce) return JS_FALSE;
	name = ifce->GetWorldURL(ifce->callback);
	*rval = STRING_TO_JSVAL(JS_InternString( c, name ));
	return JS_TRUE;
}
static JSBool replaceWorld(JSContext*c, JSObject*o, uintN n, jsval *v, jsval *rv)
{
	return JS_TRUE;
}
static JSBool addRoute(JSContext*c, JSObject*o, uintN n, jsval *v, jsval *rv)
{
	return JS_TRUE;
}
static JSBool deleteRoute(JSContext*c, JSObject*o, uintN n, jsval *v, jsval *rv)
{
	return JS_TRUE;
}

static JSBool loadURL(JSContext*c, JSObject*obj, uintN argc, jsval *argv, jsval *rval)
{
	u32 i;
	jsval item;
	jsuint len;
	JSObject *p;
	JSField *f;
	JSInterface *ifce = JS_GetInterface(c);
	if (!ifce) return JS_FALSE;
	if (argc < 1) return JS_FALSE;

	if (JSVAL_IS_STRING(argv[0])) {
		JSString *str = JSVAL_TO_STRING(argv[0]);
		if (ifce->LoadURL(ifce->callback, JS_GetStringBytes(str)) )
			return JS_TRUE;
		return JS_FALSE;
	}
	if (!JSVAL_IS_OBJECT(argv[0])) return JS_FALSE;
	
	JS_ValueToObject(c, argv[0], &p);

	f = (JSField *) JS_GetPrivate(c, p);
	if (!f || !f->js_list) return JS_FALSE;
	JS_GetArrayLength(c, f->js_list, &len);

	for (i=0; i<len; i++) {
		JS_GetElement(c, f->js_list, (jsint) i, &item);

		if (JSVAL_IS_STRING(item)) {
			JSString *str = JSVAL_TO_STRING(item);
			if (ifce->LoadURL(ifce->callback, JS_GetStringBytes(str)) )
				return JS_TRUE;
		}
	}
	return JS_TRUE;
}

static JSBool setDescription(JSContext*c, JSObject*o, uintN n, jsval *v, jsval *rv)
{
	return JS_TRUE;
}


static JSFunctionSpec browserFunctions[] = {
  {"getName", getName, 0},
  {"getVersion", getVersion, 0},
  {"getCurrentSpeed", getCurrentSpeed, 0},
  {"getCurrentFrameRate", getCurrentFrameRate, 0},
  {"getWorldURL", getWorldURL, 0},
  {"replaceWorld", replaceWorld, 0},
  {"addRoute", addRoute, 0},
  {"deleteRoute", deleteRoute, 0},
  {"loadURL", loadURL, 0},
  {"setDescription", setDescription, 0},
  {0}
};

void Script_FieldChanged(SFNode *parent, JSField *parent_owner, FieldInfo *field)
{
	ScriptPriv *priv;
	u32 i;
	ScriptField *sf;


	if (!parent) {
		parent = parent_owner->owner;
		field = &parent_owner->field;
	}
	if (!parent) return;

	if (parent->sgprivate->tag != TAG_Script) {
		/*field has changed, set routes...*/
		Node_OnEventOut(parent, field->allIndex);
		SG_NodeChanged(parent, field);
		return;
	}
	/*otherwise mark field if eventOut*/
	if (parent_owner) {
		priv = parent_owner->owner->sgprivate->privateStack;
		for (i=0; i<ChainGetCount(priv->fields); i++) {
			sf = ChainGetEntry(priv->fields, i);
			if (sf->ALL_index == field->allIndex) {
				/*queue eventOut*/
				if (sf->eventType == ET_EventOut) {
					sf->activate_event_out = 1;
				}
			}
		}
	}
}

JSBool eventOut_setProperty(JSContext *c, JSObject *obj, jsval id, jsval *val)
{
	u32 i;
	const char *eventName;
	ScriptPriv *script;
	SFNode *n;
	FieldInfo info;
	JSString *str = JS_ValueToString(c, id);
	if (!str) return JS_FALSE;
	
	eventName = JS_GetStringBytes(str);

	script = JS_GetScriptStack(c);
	if (!script) return JS_FALSE;
	n = (SFNode *) JS_GetScript(c);

	for (i=0; i<ChainGetCount(script->fields); i++) {
		ScriptField *sf = ChainGetEntry(script->fields, i);
		if (!stricmp(sf->name, eventName)) {
			Node_GetField(n, sf->ALL_index, &info);
			JS_ToNodeField(c, *val, &info, n, NULL);
			sf->activate_event_out = 1;
			return JS_TRUE;
		}
	}
	return JS_FALSE;
}


/*generic ToString method*/
static M4INLINE void sffield_toString(char *str, void *f_ptr, u32 fieldType)
{
	char temp[1000];

	switch (fieldType) {
	case FT_SFVec2f:
	{
		SFVec2f val = * ((SFVec2f *) f_ptr);
		sprintf(temp, "%f %f", val.x, val.y);
		strcat(str, temp);
		break;
	}
	case FT_SFVec3f:
	{
		SFVec3f val = * ((SFVec3f *) f_ptr);
		sprintf(temp, "%f %f %f", val.x, val.y, val.z);
		strcat(str, temp);
		break;
	}
	case FT_SFVec4f:
	{
		SFVec4f val = * ((SFVec4f *) f_ptr);
		sprintf(temp, "%f %f %f %f", val.x, val.y, val.z, val.q);
		strcat(str, temp);
		break;
	}
	case FT_SFRotation:
	{
		SFRotation val = * ((SFRotation *) f_ptr);
		sprintf(temp, "%f %f %f %f", val.xAxis, val.xAxis, val.zAxis, val.angle);
		strcat(str, temp);
		break;
	}
	case FT_SFColor:
	{
		SFColor val = * ((SFColor *) f_ptr);
		sprintf(temp, "%f %f %f", val.red, val.green, val.blue);
		strcat(str, temp);
		break;
	}
	case FT_SFImage:
	{
		SFImage *val = ((SFImage *)f_ptr);
		sprintf(temp, "%dx%dx%d", val->width, val->height, val->numComponents);
		strcat(str, temp);
		break;
	}
	
	}
}


static JSBool field_toString(JSContext *c, JSObject *obj, uintN n, jsval *v, jsval *rval)
{
	u32 i, sftype;
	GenMFField *mf;
	char str[5000];
	JSString *s;
	JSField *f = (JSField *) JS_GetPrivate(c, obj);
	if (!f) return JS_FALSE;

	strcpy(str, "");

	if (SG_IsSFField(f->field.fieldType)) {
		sffield_toString(str, f->field.far_ptr, f->field.fieldType);
		return JS_TRUE;
	}
	if (f->field.fieldType == FT_MFNode) return JS_TRUE;

	mf = *((GenMFField **)f->field.far_ptr);
	sftype = SG_GetSFType(f->field.fieldType);
	for (i=0; i<mf->count; i++) {
		sffield_toString(str, &mf->array[i], sftype);
		if (i<mf->count-1) strcat(str, ", ");
	}

	s = JS_NewStringCopyZ(c, str);
	if (!s) return JS_FALSE;
	*rval = STRING_TO_JSVAL(s); 
	return JS_TRUE; 
}



static JSBool SFNodeConstructor(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	u32 tag, ID;
	SFNode *new_node;
	JSField *field;
	JSString *str;
	LPPROTO proto;
	LPSCENEGRAPH sg;
	char *node_name;
	B_Script *sc = JS_GetScript(c);
	if (argc != 1) return JS_FALSE;
	
	str = JS_ValueToString(c, argv[0]);
	if (!str) return JS_FALSE;

	node_name = JS_GetStringBytes(str);
	if (!strnicmp(node_name, "_proto", 6)) {
		ID = atoi(node_name+6);

		/*locate proto in current graph and all parents*/
		sg = sc->sgprivate->scenegraph;
		while (1) {
			proto = SG_FindProto(sg, ID, NULL);
			if (proto) break;
			if (!sg->parent_scene) break;
			sg = sg->parent_scene;
		}
		if (!proto) return JS_FALSE;
		/* create interface and load code in current graph*/
		new_node = Proto_CreateInstance(sc->sgprivate->scenegraph, proto);
		if (!new_node) return JS_FALSE;
		/*OK, instanciate proto code*/
		if (Proto_LoadCode(new_node) != M4OK) {
			Node_Unregister(new_node, NULL);
			return JS_FALSE;
		}
	} else {
		tag = Node_GetTagByName(node_name);
		if (!tag) return JS_FALSE;
		new_node = SG_NewNode(sc->sgprivate->scenegraph, tag);

		if (!new_node) return JS_FALSE;
		Node_Init(new_node);
	}

	field = NewJSField(0);
	field->field.fieldType = FT_SFNode;
	field->temp_node = new_node;
	field->field.far_ptr = &field->temp_node;

	JS_SetPrivate(c, obj, field);
	return JS_TRUE;
}
static void node_finalize(JSContext *c, JSObject *obj)
{
	JSField *ptr = (JSField *) JS_GetPrivate(c, obj);
	if (ptr) {
		/*this node is not inserted in the graph, destroy*/
		if (! ptr->is_attached) {
			Node_Delete( *((SFNode**) ptr->field.far_ptr), NULL);
		}
		free(ptr);
	}
	JS_SetPrivate(c, obj, 0 );
}
static JSBool node_toString(JSContext *c, JSObject *obj, uintN i, jsval *v, jsval *rval)
{
	char str[1000];
	SFNode *n;
	JSString *s;
	JSField *f = (JSField *) JS_GetPrivate(c, obj);
	if (!f) return JS_FALSE;

	n = * ((SFNode **)f->field.far_ptr);
	if (n->sgprivate->NodeID) {
		if (n->sgprivate->NodeName) {
			sprintf(str , "DEF %s ", n->sgprivate->NodeName);
		} else {
			sprintf(str , "DEF %d ", n->sgprivate->NodeID - 1);
		}
	}
	strcat(str, Node_GetName(n));
	s = JS_NewStringCopyZ(c, (const char *) str);
	if (!s) return JS_FALSE;
	*rval = STRING_TO_JSVAL(s); 
	return JS_TRUE; 
}
static JSBool node_getProperty(JSContext *c, JSObject *obj, jsval id, jsval *vp)
{
	SFNode *n;
	u32 index;
	JSString *str;
	FieldInfo info;
	JSField *ptr = (JSField *) JS_GetPrivate(c, obj);
	assert(ptr->field.fieldType==FT_SFNode);
	n = * ((SFNode **)ptr->field.far_ptr);

	if (n && JSVAL_IS_STRING(id) && ( (str = JSVAL_TO_STRING(id)) != 0) ) {
		char *fieldName = JS_GetStringBytes(str);
		/*fieldID indexing*/
		if (!strnicmp(fieldName, "_field", 6)) {
			index = atoi(fieldName+6);
			if ( Node_GetField(n, index, &info) != M4OK) return JS_FALSE;
		} else {
			if ( Node_GetFieldByName(n, fieldName, &info) != M4OK) return JS_FALSE;
		}
		*vp = JS_ToJSField(JS_GetScriptStack(c), &info, n);
		return (*vp == JSVAL_NULL) ? JS_FALSE : JS_TRUE;
    }
	return JS_FALSE;
}

static JSBool node_setProperty(JSContext *c, JSObject *obj, jsval id, jsval *vp)
{
	SFNode *n;
	FieldInfo info;
	u32 index;
	char *fieldname;
	JSField *ptr = (JSField *) JS_GetPrivate(c, obj);
	assert(ptr->field.fieldType==FT_SFNode);
	n = * ((SFNode **)ptr->field.far_ptr);

	if (n && JSVAL_IS_STRING(id)) {
		JSString *str = JSVAL_TO_STRING(id);
		fieldname = JS_GetStringBytes(str);
		
		/*fieldID indexing*/
		if (!strnicmp(fieldname, "_field", 6)) {
			index = atoi(fieldname+6);
			if ( Node_GetField(n, index, &info) != M4OK) return JS_FALSE;
		} else {
			if (Node_GetFieldByName(n, fieldname, &info) != M4OK) return JS_FALSE;
		}
		JS_ToNodeField(c, *vp, &info, n, ptr);
	}
	return JS_TRUE;
}
static JSFunctionSpec SFNodeMethods[] = {
	{"toString",        node_toString,       0},
	{0}
};



/* Generic field destructor */
static void field_finalize(JSContext *c, JSObject *obj)
{
	JSField *ptr = (JSField *) JS_GetPrivate(c, obj);
	if (ptr) {
		if (!ptr->is_attached) SG_DeleteFieldPointer(ptr->field.far_ptr, ptr->field.fieldType);
		free(ptr);
	}
	JS_SetPrivate(c, obj, 0 );
}



/*SFVec2f class functions */
static M4INLINE JSField *SFVec2f_Create(JSContext *c, JSObject *obj, Bool attached, Float x, Float y)
{
	JSField *field;
	SFVec2f *v;
	field = NewJSField(attached);
	v = SG_NewFieldPointer(FT_SFVec2f);
	field->field.far_ptr = v;
	v->x = x;
	v->y = y;
	JS_SetPrivate(c, obj, field);
	return field;
}
static JSBool SFVec2fConstructor(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rv)
{
	jsdouble x = 0.0, y = 0.0;
	if (argc > 0) JS_ValueToNumber(c, argv[0], &x);
	if (argc > 1) JS_ValueToNumber(c, argv[1], &y);
	SFVec2f_Create(c, obj, 0, (Float) x, (Float) y);
	return JS_TRUE;
}
static JSBool vec2f_getProperty(JSContext *c, JSObject *obj, jsval id, jsval *vp)
{
	JSField *val = (JSField *) JS_GetPrivate(c, obj);
	if (JSVAL_IS_INT(id)) {
		switch (JSVAL_TO_INT(id)) {
		case 0: *vp = DOUBLE_TO_JSVAL(JS_NewDouble(c, ((SFVec2f*)val->field.far_ptr)->x)); break;
		case 1: *vp = DOUBLE_TO_JSVAL(JS_NewDouble(c, ((SFVec2f*)val->field.far_ptr)->y)); break;
		default: return JS_FALSE;
		}
	}
	return JS_TRUE;
}

static JSBool vec2f_setProperty(JSContext *c, JSObject *obj, jsval id, jsval *vp)
{
	jsdouble d;
	Bool changed = 0;
	JSField *ptr = (JSField *) JS_GetPrivate(c, obj);

	if (JSVAL_IS_INT(id) && JSVAL_TO_INT(id) >= 0 && JSVAL_TO_INT(id) < 2 && JS_ValueToNumber(c, *vp, &d)) {
		switch (JSVAL_TO_INT(id)) {
		case 0: 
			changed = ! ( ((SFVec2f*)ptr->field.far_ptr)->x == (Float) d);
			((SFVec2f*)ptr->field.far_ptr)->x = (Float) d;
			break;
		case 1: 
			changed = ! ( ((SFVec2f*)ptr->field.far_ptr)->y == (Float) d);
			((SFVec2f*)ptr->field.far_ptr)->y = (Float) d;
			break;
		default: return JS_FALSE;
		}
		if (changed) Script_FieldChanged(NULL, ptr, NULL);
		return JS_TRUE;
    }
	return JS_FALSE;
}
static JSPropertySpec SFVec2fProps[] = {
	{"x",       0,       JSPROP_ENUMERATE | JSPROP_PERMANENT},
	{"y",       1,       JSPROP_ENUMERATE | JSPROP_PERMANENT},
	{0}
};
static JSBool vec2f_add(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	SFVec2f *v1, *v2;
	JSObject *pNew;
	if (argc<=0 || !JSVAL_IS_OBJECT(argv[0]) || (&SFVec2fClass != JS_GetClass(JSVAL_TO_OBJECT(argv[0]))))
		return JS_FALSE;

	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
    v2 = ((JSField *) JS_GetPrivate(c, JSVAL_TO_OBJECT(argv[0])))->field.far_ptr;
	pNew = JS_NewObject(c, &SFVec2fClass, 0, JS_GetParent(c, obj));  
	SFVec2f_Create(c, pNew, 0, v1->x + v2->x, v1->y + v2->y);
	*rval = OBJECT_TO_JSVAL(pNew);
	return JS_TRUE;
}
static JSBool vec2f_subtract(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	SFVec2f *v1, *v2;
	JSObject *pNew;
	if (argc<=0 || !JSVAL_IS_OBJECT(argv[0]) || (&SFVec2fClass != JS_GetClass(JSVAL_TO_OBJECT(argv[0]))))
		return JS_FALSE;

	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
    v2 = ((JSField *) JS_GetPrivate(c, JSVAL_TO_OBJECT(argv[0])))->field.far_ptr;
	pNew = JS_NewObject(c, &SFVec2fClass, 0, JS_GetParent(c, obj));  
	SFVec2f_Create(c, pNew, 0, v1->x - v2->x, v1->y - v2->y);
	*rval = OBJECT_TO_JSVAL(pNew);
	return JS_TRUE;
}
static JSBool vec2f_negate(JSContext *c, JSObject *obj, uintN n, jsval *v, jsval *rval)
{
	SFVec2f *v1;
	JSObject *pNew;
	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
	pNew = JS_NewObject(c, &SFVec2fClass, 0, JS_GetParent(c, obj));  
	SFVec2f_Create(c, pNew, 0, -v1->x , -v1->y );
	*rval = OBJECT_TO_JSVAL(pNew);
	return JS_TRUE;
}
static JSBool vec2f_multiply(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	SFVec2f *v1;
	JSObject *pNew;
	jsdouble d;
	if (argc<=0) return JS_FALSE;
	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
	pNew = JS_NewObject(c, &SFVec2fClass, 0, JS_GetParent(c, obj));  
	JS_ValueToNumber(c, argv[0], &d );
	SFVec2f_Create(c, pNew, 0, v1->x * (Float) d, v1->y * (Float) d);
	*rval = OBJECT_TO_JSVAL(pNew);
	return JS_TRUE;
}
static JSBool vec2f_divide(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	SFVec2f *v1;
	JSObject *pNew;
	jsdouble d;
	if (argc<=0) return JS_FALSE;
	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
	pNew = JS_NewObject(c, &SFVec2fClass, 0, JS_GetParent(c, obj));  
	JS_ValueToNumber(c, argv[0], &d );
	SFVec2f_Create(c, pNew, 0, v1->x / (Float) d, v1->y / (Float) d);
	*rval = OBJECT_TO_JSVAL(pNew);
	return JS_TRUE;
}
static JSBool vec2f_length(JSContext *c, JSObject *obj, uintN n, jsval *val, jsval *rval)
{
	Double res;
	SFVec2f *v1;
	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
	res = sqrt(v1->x*v1->x + v1->y*v1->y);
	*rval = DOUBLE_TO_JSVAL(JS_NewDouble(c, res) );
	return JS_TRUE;
}
static JSBool vec2f_normalize(JSContext *c, JSObject *obj, uintN n, jsval *val, jsval *rval)
{
	SFVec2f *v1;
	Double res;
	JSObject *pNew;
	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
	res = sqrt(v1->x*v1->x + v1->y*v1->y);
	pNew = JS_NewObject(c, &SFVec2fClass, 0, JS_GetParent(c, obj));  
	SFVec2f_Create(c, pNew, 0, v1->x / (Float) res, v1->y / (Float) res);
	*rval = OBJECT_TO_JSVAL(pNew);
	return JS_TRUE;
}
static JSBool vec2f_dot(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	SFVec2f *v1, *v2;
	JSObject *pNew;
	if (argc<=0 || !JSVAL_IS_OBJECT(argv[0]) || (&SFVec2fClass != JS_GetClass(JSVAL_TO_OBJECT(argv[0]))))
		return JS_FALSE;

	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
    v2 = ((JSField *) JS_GetPrivate(c, JSVAL_TO_OBJECT(argv[0])))->field.far_ptr;
	pNew = JS_NewObject(c, &SFVec2fClass, 0, JS_GetParent(c, obj));  
	SFVec2f_Create(c, pNew, 0, v1->x * v2->x, v1->y * v2->y);
	*rval = OBJECT_TO_JSVAL(pNew);
	return JS_TRUE;
}

static JSFunctionSpec SFVec2fMethods[] = {
    {"add",             vec2f_add,      1},
    {"divide",          vec2f_divide,   1},
    {"dot",             vec2f_dot,      1},
    {"length",          vec2f_length,   0},
    {"multiply",        vec2f_multiply, 1},
    {"normalize",       vec2f_normalize,0},
    {"subtract",        vec2f_subtract, 1},
    {"negate",          vec2f_negate,   0},
	{"toString",        field_toString,       0},
	{0}
};


/*SFVec3f class functions */
static M4INLINE JSField *SFVec3f_Create(JSContext *c, JSObject *obj, Bool attached, Float x, Float y, Float z)
{
	JSField *field;
	SFVec3f *v;
	field = NewJSField(attached);
	v = SG_NewFieldPointer(FT_SFVec3f);
	field->field.far_ptr = v;
	v->x = x;
	v->y = y;
	v->z = z;
	JS_SetPrivate(c, obj, field);
	return field;
}
static JSBool SFVec3fConstructor(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rv)
{
	jsdouble x = 0.0, y = 0.0, z = 0.0;
	if (argc > 0) JS_ValueToNumber(c, argv[0], &x);
	if (argc > 1) JS_ValueToNumber(c, argv[1], &y);
	if (argc > 2) JS_ValueToNumber(c, argv[2], &z);
	SFVec3f_Create(c, obj, 0, (Float) x, (Float) y, (Float) z);
	return JS_TRUE;
}
static JSBool vec3f_getProperty(JSContext *c, JSObject *obj, jsval id, jsval *vp)
{
	JSField *val = (JSField *) JS_GetPrivate(c, obj);
	if (JSVAL_IS_INT(id)) {
		switch (JSVAL_TO_INT(id)) {
		case 0: *vp = DOUBLE_TO_JSVAL(JS_NewDouble(c, ((SFVec3f*)val->field.far_ptr)->x)); break;
		case 1: *vp = DOUBLE_TO_JSVAL(JS_NewDouble(c, ((SFVec3f*)val->field.far_ptr)->y)); break;
		case 2: *vp = DOUBLE_TO_JSVAL(JS_NewDouble(c, ((SFVec3f*)val->field.far_ptr)->z)); break;
		default: return JS_FALSE;
		}
	}
	return JS_TRUE;
}
static JSBool vec3f_setProperty(JSContext *c, JSObject *obj, jsval id, jsval *vp)
{
	jsdouble d;
	Bool changed = 0;
	JSField *ptr = (JSField *) JS_GetPrivate(c, obj);

	if (JSVAL_IS_INT(id) && JSVAL_TO_INT(id) >= 0 && JSVAL_TO_INT(id) < 3 && JS_ValueToNumber(c, *vp, &d)) {
		switch (JSVAL_TO_INT(id)) {
		case 0: 
			changed = ! ( ((SFVec3f*)ptr->field.far_ptr)->x == (Float) d);
			((SFVec3f*)ptr->field.far_ptr)->x = (Float) d;
			break;
		case 1: 
			changed = ! ( ((SFVec3f*)ptr->field.far_ptr)->y == (Float) d);
			((SFVec3f*)ptr->field.far_ptr)->y = (Float) d;
			break;
		case 2: 
			changed = ! ( ((SFVec3f*)ptr->field.far_ptr)->z == (Float) d);
			((SFVec3f*)ptr->field.far_ptr)->z = (Float) d;
			break;
		default: return JS_FALSE;
		}
		if (changed) Script_FieldChanged(NULL, ptr, NULL);
		return JS_TRUE;
    }
	return JS_FALSE;
}
static JSPropertySpec SFVec3fProps[] = {
	{"x",       0,       JSPROP_ENUMERATE | JSPROP_PERMANENT},
	{"y",       1,       JSPROP_ENUMERATE | JSPROP_PERMANENT},
	{"z",       2,       JSPROP_ENUMERATE | JSPROP_PERMANENT},
	{0}
};
static JSBool vec3f_add(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	SFVec3f *v1, *v2;
	JSObject *pNew;
	if (argc<=0 || !JSVAL_IS_OBJECT(argv[0]) || (&SFVec2fClass != JS_GetClass(JSVAL_TO_OBJECT(argv[0]))))
		return JS_FALSE;

	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
    v2 = ((JSField *) JS_GetPrivate(c, JSVAL_TO_OBJECT(argv[0])))->field.far_ptr;
	pNew = JS_NewObject(c, &SFVec3fClass, 0, JS_GetParent(c, obj));  
	SFVec3f_Create(c, pNew, 0, v1->x + v2->x, v1->y + v2->y, v1->z + v2->z);
	*rval = OBJECT_TO_JSVAL(pNew);
	return JS_TRUE;
}
static JSBool vec3f_subtract(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	SFVec3f *v1, *v2;
	JSObject *pNew;
	if (argc<=0 || !JSVAL_IS_OBJECT(argv[0]) || (&SFVec2fClass != JS_GetClass(JSVAL_TO_OBJECT(argv[0]))))
		return JS_FALSE;

	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
    v2 = ((JSField *) JS_GetPrivate(c, JSVAL_TO_OBJECT(argv[0])))->field.far_ptr;
	pNew = JS_NewObject(c, &SFVec3fClass, 0, JS_GetParent(c, obj));  
	SFVec3f_Create(c, pNew, 0, v1->x - v2->x, v1->y - v2->y, v1->z - v2->z);
	*rval = OBJECT_TO_JSVAL(pNew);
	return JS_TRUE;
}
static JSBool vec3f_negate(JSContext *c, JSObject *obj, uintN n, jsval *v, jsval *rval)
{
	SFVec3f *v1;
	JSObject *pNew;
	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
	pNew = JS_NewObject(c, &SFVec3fClass, 0, JS_GetParent(c, obj));  
	SFVec3f_Create(c, pNew, 0, -v1->x , -v1->y , -v1->z );
	*rval = OBJECT_TO_JSVAL(pNew);
	return JS_TRUE;
}
static JSBool vec3f_multiply(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	SFVec3f *v1;
	JSObject *pNew;
	jsdouble d;
	if (argc<=0) return JS_FALSE;

	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
	pNew = JS_NewObject(c, &SFVec3fClass, 0, JS_GetParent(c, obj));  
	JS_ValueToNumber(c, argv[0], &d );
	SFVec3f_Create(c, pNew, 0, v1->x * (Float) d, v1->y * (Float) d, v1->z * (Float) d);
	*rval = OBJECT_TO_JSVAL(pNew);
	return JS_TRUE;
}
static JSBool vec3f_divide(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	SFVec3f *v1;
	JSObject *pNew;
	jsdouble d;
	if (argc<=0) return JS_FALSE;
	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
	pNew = JS_NewObject(c, &SFVec2fClass, 0, JS_GetParent(c, obj));  
	JS_ValueToNumber(c, argv[0], &d );
	SFVec3f_Create(c, pNew, 0, v1->x / (Float) d, v1->y / (Float) d, v1->z / (Float) d);
	*rval = OBJECT_TO_JSVAL(pNew);
	return JS_TRUE;
}
static JSBool vec3f_length(JSContext *c, JSObject *obj, uintN n, jsval *val, jsval *rval)
{
	Double res;
	SFVec3f *v1;
	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
	res = sqrt(v1->x*v1->x + v1->y*v1->y + v1->z*v1->z);
	*rval = DOUBLE_TO_JSVAL(JS_NewDouble(c, res) );
	return JS_TRUE;
}
static JSBool vec3f_normalize(JSContext *c, JSObject *obj, uintN n, jsval *val, jsval *rval)
{
	SFVec3f *v1;
	Double res;
	JSObject *pNew;

	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
	res = sqrt(v1->x*v1->x + v1->y*v1->y + v1->z*v1->z);
	pNew = JS_NewObject(c, &SFVec3fClass, 0, JS_GetParent(c, obj));  
	SFVec3f_Create(c, pNew, 0, v1->x / (Float) res, v1->y / (Float) res, v1->z / (Float) res);
	*rval = OBJECT_TO_JSVAL(pNew);
	return JS_TRUE;
}
static JSBool vec3f_dot(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	SFVec3f *v1, *v2;
	JSObject *pNew;
	if (argc<=0 || !JSVAL_IS_OBJECT(argv[0]) || (&SFVec2fClass != JS_GetClass(JSVAL_TO_OBJECT(argv[0]))))
		return JS_FALSE;

	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
    v2 = ((JSField *) JS_GetPrivate(c, JSVAL_TO_OBJECT(argv[0])))->field.far_ptr;
	pNew = JS_NewObject(c, &SFVec3fClass, 0, JS_GetParent(c, obj));  
	SFVec3f_Create(c, pNew, 0, v1->x * v2->x, v1->y * v2->y, v1->z * v2->z);
	*rval = OBJECT_TO_JSVAL(pNew);
	return JS_TRUE;
}
static JSBool vec3f_cross(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	SFVec3f *v1, *v2;
	JSObject *pNew;

	if (argc<=0 || !JSVAL_IS_OBJECT(argv[0]) || (&SFVec2fClass != JS_GetClass(JSVAL_TO_OBJECT(argv[0]))))
		return JS_FALSE;

	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
    v2 = ((JSField *) JS_GetPrivate(c, JSVAL_TO_OBJECT(argv[0])))->field.far_ptr;
	pNew = JS_NewObject(c, &SFVec3fClass, 0, JS_GetParent(c, obj));  
	
	SFVec3f_Create(c, pNew, 0, 
					v1->y * v2->z - v1->z * v2->y,
					v1->z * v2->x - v1->x * v2->z, 
					v1->x * v2->y - v1->y * v2->x);

	*rval = OBJECT_TO_JSVAL(pNew);
	return JS_TRUE;
}

static JSFunctionSpec SFVec3fMethods[] = {
    {"add",             vec3f_add,      1},
    {"divide",          vec3f_divide,   1},
    {"dot",             vec3f_dot,      1},
    {"length",          vec3f_length,   0},
    {"multiply",        vec3f_multiply, 1},
    {"normalize",       vec3f_normalize,0},
    {"subtract",        vec3f_subtract, 1},
    {"cross",			vec3f_cross,	1},
    {"negate",          vec3f_negate,   0},
	{"toString",        field_toString,	0},
	{0}
};




/* SFColor class functions */
static M4INLINE JSField *SFColor_Create(JSContext *c, JSObject *obj, Bool attached, Float r, Float g, Float b)
{
	JSField *field;
	SFColor *v;
	field = NewJSField(attached);
	v = SG_NewFieldPointer(FT_SFColor);
	field->field.far_ptr = v;
	v->red = r;
	v->green = g;
	v->blue = b;
	JS_SetPrivate(c, obj, field);
	return field;
}
static JSBool SFColorConstructor(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rv)
{
	jsdouble r = 0.0, g = 0.0, b = 0.0;
	if (argc > 0) JS_ValueToNumber(c, argv[0], &r);
	if (argc > 1) JS_ValueToNumber(c, argv[1], &g);
	if (argc > 2) JS_ValueToNumber(c, argv[2], &b);
	SFVec3f_Create(c, obj, 0, (Float) r, (Float) g, (Float) b);
	return JS_TRUE;
}
static JSBool color_getProperty(JSContext *c, JSObject *obj, jsval id, jsval *vp)
{
	JSField *val = (JSField *) JS_GetPrivate(c, obj);
	if (JSVAL_IS_INT(id)) {
		switch (JSVAL_TO_INT(id)) {
		case 0: *vp = DOUBLE_TO_JSVAL(JS_NewDouble(c, ((SFColor*)val->field.far_ptr)->red)); break;
		case 1: *vp = DOUBLE_TO_JSVAL(JS_NewDouble(c, ((SFColor*)val->field.far_ptr)->green)); break;
		case 2: *vp = DOUBLE_TO_JSVAL(JS_NewDouble(c, ((SFColor*)val->field.far_ptr)->blue)); break;
		default: return JS_FALSE;
		}
	}
	return JS_TRUE;
}

static JSBool color_setProperty(JSContext *c, JSObject *obj, jsval id, jsval *vp)
{
	jsdouble d;
	Bool changed = 0;
	JSField *ptr = (JSField *) JS_GetPrivate(c, obj);

	if (JSVAL_IS_INT(id) && JSVAL_TO_INT(id) >= 0 && JSVAL_TO_INT(id) < 3 && JS_ValueToNumber(c, *vp, &d)) {
		switch (JSVAL_TO_INT(id)) {
		case 0: 
			changed = ! ( ((SFColor*)ptr->field.far_ptr)->red == (Float) d);
			((SFColor*)ptr->field.far_ptr)->red = (Float) d;
			break;
		case 1: 
			changed = ! ( ((SFColor*)ptr->field.far_ptr)->green == (Float) d);
			((SFColor*)ptr->field.far_ptr)->green = (Float) d;
			break;
		case 2: 
			changed = ! ( ((SFColor*)ptr->field.far_ptr)->blue == (Float) d);
			((SFColor*)ptr->field.far_ptr)->blue = (Float) d;
			break;
		default: return JS_FALSE;
		}
		if (changed) Script_FieldChanged(NULL, ptr, NULL);
		return JS_TRUE;
    }
	return JS_FALSE;
}
static JSPropertySpec SFColorProps[] = {
	{"r",       0,       JSPROP_ENUMERATE | JSPROP_PERMANENT},
	{"g",       1,       JSPROP_ENUMERATE | JSPROP_PERMANENT},
	{"b",       2,       JSPROP_ENUMERATE | JSPROP_PERMANENT},
	{0}
};
static JSBool color_setHSV(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rv)
{
	SFColor *v1, hsv;
	jsdouble h, s, v;
	JSField *ptr = (JSField *) JS_GetPrivate(c, obj);
	if (argc != 3) return JS_FALSE;
	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
	JS_ValueToNumber( c, argv[0], &h);
	JS_ValueToNumber( c, argv[1], &s);
	JS_ValueToNumber( c, argv[2], &v);
	hsv.red = (Float) h;
	hsv.green = (Float) s;
	hsv.blue = (Float) v;
	SFColor_fromHSV(&hsv);
	SG_CopyField(v1, &hsv, FT_SFColor);
	Script_FieldChanged(NULL, ptr, NULL);
	return JS_TRUE;
}

static JSBool color_getHSV(JSContext *c, JSObject *obj, uintN n, jsval *va, jsval *rval)
{
	SFColor *v1, hsv;
	jsval vec[3];
	JSObject *arr;

	v1 = ((JSField *) JS_GetPrivate(c, obj))->field.far_ptr;
	hsv = *v1;
	SFColor_toHSV(&hsv);
	vec[0] = DOUBLE_TO_JSVAL(JS_NewDouble(c, hsv.red));
	vec[1] = DOUBLE_TO_JSVAL(JS_NewDouble(c, hsv.green));
	vec[2] = DOUBLE_TO_JSVAL(JS_NewDouble(c, hsv.blue));
	arr = JS_NewArrayObject(c, 3, vec);
	*rval = OBJECT_TO_JSVAL(arr);
	return JS_TRUE;
}
static JSFunctionSpec SFColorMethods[] = {
    {"setHSV",          color_setHSV,   3, 0, 0},
    {"getHSV",          color_getHSV,   0, 0, 0},
	{"toString",        field_toString,       0, 0, 0},
	{0, 0, 0, 0, 0}
};



static JSBool MFArrayConstructor(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	JSField *ptr = NewJSField(1);
	ptr->js_list = JS_NewArrayObject(c, (jsint) argc, argv);
	JS_SetPrivate(c, obj, ptr);
	*rval = OBJECT_TO_JSVAL(obj);
	return obj == 0 ? JS_FALSE : JS_TRUE;
}

static void array_finalize(JSContext *c, JSObject *obj)
{
	JSField *ptr = (JSField *) JS_GetPrivate(c, obj);
	if (ptr) free(ptr);
	JS_SetPrivate(c, obj, 0 );
}

JSBool array_getElement(JSContext *c, JSObject *obj, jsval id, jsval *rval)
{
	u32 i;
	JSField *ptr = (JSField *) JS_GetPrivate(c, obj);
	if (JSVAL_IS_INT(id)) {
		i = JSVAL_TO_INT(id);
		JS_GetElement(c, ptr->js_list, i, rval);
	}
	return JS_TRUE;
}

//this could be overloaded for each MF type...
JSBool array_setElement(JSContext *c, JSObject *obj, jsval id, jsval *rval)
{
	u32 ind, i;
	JSBool ret;
	jsdouble d;
	void *slot;
	JSField *ptr = (JSField *) JS_GetPrivate(c, obj);
	
	ind = JSVAL_TO_INT(id);
	
	ret = JS_SetElement(c, ptr->js_list, ind, rval);
	if (ret!=JS_TRUE) return JS_FALSE;

	if (SG_IsSFField(ptr->field.fieldType)) return JS_FALSE;
	
	/*insert till index if needed*/
	if (ptr->field.fieldType != FT_MFNode) {
		i = ((GenMFField *)ptr->field.far_ptr)->count;
		while (i <= ind) {
			MFField_Insert(ptr->field.far_ptr, ptr->field.fieldType, &slot, i);
			i++;
		}
	}

	switch (ptr->field.fieldType) {
	case FT_MFBool:
		if (! JSVAL_IS_BOOLEAN(*rval)) return JS_FALSE;
	{
		Bool val = JSVAL_TO_BOOLEAN(*rval);
		MFBool *f = (MFBool *) ptr->field.far_ptr;
		f->vals[ind] = val;
	}
		break;
	case FT_MFInt32:
		if (! JSVAL_IS_INT(*rval)) return JS_FALSE;
	{
		s32 val = JSVAL_TO_INT(*rval);
		MFInt32 *f = (MFInt32 *) ptr->field.far_ptr;
		f->vals[ind] = val;
	}
		break;
	case FT_MFFloat:
		if (! JSVAL_IS_NUMBER(*rval)) return JS_FALSE;
	{
		MFFloat *f = (MFFloat *) ptr->field.far_ptr;
		JS_ValueToNumber(c, *rval, &d );
		f->vals[ind] = (Float) d;
	}
		break;
	case FT_MFTime:
		if (! JSVAL_IS_NUMBER(*rval)) return JS_FALSE;
	{
		MFTime *f = (MFTime *) ptr->field.far_ptr;
		JS_ValueToNumber(c, *rval, &d );
		f->vals[ind] = d;
	}
		break;
	case FT_MFString:
	{
		char *str;
		JSString *jsstr = JS_ValueToString(c, *rval);
		MFString *f = (MFString *) ptr->field.far_ptr;
		if (!jsstr) return JS_FALSE;
		str = JS_GetStringBytes(jsstr);
		if (f->vals[ind]) free(f->vals[ind]);
		f->vals[ind] = strdup(str);
	}
		break;
	case FT_MFURL:
	{
		JSString *str = JS_ValueToString(c, *rval);
		MFURL *f = (MFURL *) ptr->field.far_ptr;
		f->vals[ind].OD_ID = 0;
		if (f->vals[ind].url) free(f->vals[ind].url);
		f->vals[ind].url = strdup(JS_GetStringBytes(str));
	}
		break;

	case FT_MFVec2f:
		if (! JSVAL_IS_OBJECT(*rval)) return JS_FALSE;
	{
		SFVec2f *item = (SFVec2f *) ((JSField *) JS_GetPrivate(c, JSVAL_TO_OBJECT(*rval) ))->field.far_ptr;
		MFVec2f *f = (MFVec2f *) ptr->field.far_ptr;
		SG_CopyField(&f->vals[ind], item, FT_SFVec2f);
	}
		break;
	case FT_MFVec3f:
		if (! JSVAL_IS_OBJECT(*rval)) return JS_FALSE;
	{
		SFVec3f *item = (SFVec3f *) ((JSField *) JS_GetPrivate(c, JSVAL_TO_OBJECT(*rval) ))->field.far_ptr;
		MFVec3f *f = (MFVec3f *) ptr->field.far_ptr;
		SG_CopyField(&f->vals[ind], item, FT_SFVec3f);
	}
		break;
	case FT_MFColor:
		if (! JSVAL_IS_OBJECT(*rval)) return JS_FALSE;
	{
		SFColor *item = (SFColor *) ((JSField *) JS_GetPrivate(c, JSVAL_TO_OBJECT(*rval) ))->field.far_ptr;
		MFColor *f = (MFColor *) ptr->field.far_ptr;
		SG_CopyField(&f->vals[ind], item, FT_SFColor);
	}
		break;
	case FT_MFNode:
		if (!JSVAL_IS_OBJECT(*rval)) return JS_FALSE;
		if (!ptr->owner) return JS_FALSE;
	{

		Chain *f = * (Chain **) ptr->field.far_ptr;
		if (*rval) {
			JSField *jsf = (JSField *) JS_GetPrivate(c, JSVAL_TO_OBJECT(*rval) );
			SFNode *item = * (SFNode **) jsf->field.far_ptr;

			/*we're attached to the parent*/
			jsf->is_attached = 1;

			Node_Register(item, ptr->owner);

			if (ChainGetCount(f)>ind) {
				Node_ReplaceChild(ptr->owner, f, ind, item);
			} else {
				ChainAddEntry(f, item);
			}
		} else {
			Node_ReplaceChild(ptr->owner, f, ind, NULL);
		}
	}
		break;

	default:
		return JS_FALSE;
	}
	Script_FieldChanged(NULL, ptr, NULL);
	return ret;
}

static JSBool array_toString(JSContext *c, JSObject *obj, uintN n, jsval *v, jsval *rval)
{
	jsuint len;
	jsval item;
	JSString *s;
	u32 i;
	char str[5000];
	JSField *arr = (JSField *) JS_GetPrivate(c, obj);
	if (!arr || !arr->js_list) return JS_FALSE;

	JS_GetArrayLength(c, arr->js_list, &len);

	strcpy(str, "");
	for (i=0; i<len;i++) {
		JS_GetElement(c, arr->js_list, i, &item);
		strcat(str , JS_GetStringBytes( JS_ValueToString(c, item) ) );
		strcat(str , ", ");
	}
	s = JS_NewStringCopyZ(c, (const char *) str);
	if (!s) return JS_FALSE;
	*rval = STRING_TO_JSVAL(s); 
	return JS_TRUE;
}

JSBool array_setLength(JSContext *c, JSObject *obj, jsval v, jsval *val)
{
	u32 len;
	JSField *ptr = (JSField *) JS_GetPrivate(c, obj);
	if (!JSVAL_IS_INT(*val) || JSVAL_TO_INT(*val) < 0) return JS_FALSE;
	len = JSVAL_TO_INT(*val);
	return JS_SetArrayLength(c, ptr->js_list, len);
}
JSBool array_getLength(JSContext *c, JSObject *obj, jsval v, jsval *val)
{
	jsuint len;
	JSField *ptr = (JSField *) JS_GetPrivate(c, obj);
	JSBool ret = JS_GetArrayLength(c, ptr->js_list, &len);
	*val = INT_TO_JSVAL(len);
	return ret;
}
static JSPropertySpec MFArrayProp[] = {
	{ "length", 0, JSPROP_PERMANENT, array_getLength, array_setLength },
	{ 0, 0, 0, 0, 0 } 
};
static JSFunctionSpec MFArrayMethods[] = {
	{"toString",        array_toString,       0},
	{0}
};


/* MFVec2f class constructor */
static JSBool MFVec2fConstructor(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	jsval val;
	JSObject *item;
	u32 i;
	JSField *ptr = NewJSField(0);
	ptr->js_list = JS_NewArrayObject(c, 0, 0);
	JS_SetArrayLength(c, ptr->js_list, argc);
	JS_SetPrivate(c, obj, ptr);

	for (i=0; i<argc; i++) {
		if (!JSVAL_IS_OBJECT(argv[i]) || ( &SFVec2fClass != JS_GetClass( JSVAL_TO_OBJECT(argv[i])) ) ) {
			item = JS_ConstructObject(c, &SFVec2fClass, 0, obj);
			val = OBJECT_TO_JSVAL(item);
			JS_SetElement(c, ptr->js_list, i, &val);
		} else {
			JS_SetElement(c, ptr->js_list, i, &argv[i]);
		}
	}
	*rval = OBJECT_TO_JSVAL(obj);
	return obj == 0 ? JS_FALSE : JS_TRUE;
}

/* MFVec3f class constructor */
static JSBool MFVec3fConstructor(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	jsval val;
	JSObject *item;
	u32 i;
	JSField *ptr = NewJSField(0);
	ptr->js_list = JS_NewArrayObject(c, 0, 0);
	JS_SetArrayLength(c, ptr->js_list, argc);
	JS_SetPrivate(c, obj, ptr);

	for (i=0; i<argc; i++) {
		if (!JSVAL_IS_OBJECT(argv[i]) || ( &SFVec3fClass != JS_GetClass( JSVAL_TO_OBJECT(argv[i])) ) ) {
			item = JS_ConstructObject(c, &SFVec3fClass, 0, obj);
			val = OBJECT_TO_JSVAL(item);
			JS_SetElement(c, ptr->js_list, i, &val);
		} else {
			JS_SetElement(c, ptr->js_list, i, &argv[i]);
		}
	}
	*rval = OBJECT_TO_JSVAL(obj);
	return obj == 0 ? JS_FALSE : JS_TRUE;
}


/*MFColor class constructor */
static JSBool MFColorConstructor(JSContext *c, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	jsval val;
	JSObject *item;
	u32 i;
	JSField *ptr = NewJSField(0);
	ptr->js_list = JS_NewArrayObject(c, 0, 0);
	JS_SetArrayLength(c, ptr->js_list, argc);
	JS_SetPrivate(c, obj, ptr);

	for (i=0; i<argc; i++) {
		if (!JSVAL_IS_OBJECT(argv[i]) || ( &SFColorClass != JS_GetClass( JSVAL_TO_OBJECT(argv[i])) ) ) {
			item = JS_ConstructObject(c, &SFColorClass, 0, obj);
			val = OBJECT_TO_JSVAL(item);
			JS_SetElement(c, ptr->js_list, i, &val);
		} else {
			JS_SetElement(c, ptr->js_list, i, &argv[i]);
		}
	}
	*rval = OBJECT_TO_JSVAL(obj);
	return obj == 0 ? JS_FALSE : JS_TRUE;
}



#define M4_SETUP_JS(vrmlclass, cname, flag, addp, delp, getp, setp, enump, resp, conv, fin)	\
	vrmlclass.name = cname;	\
	vrmlclass.flags = flag;	\
	vrmlclass.addProperty = addp;	\
	vrmlclass.delProperty = delp;	\
	vrmlclass.getProperty = getp;	\
	vrmlclass.setProperty = setp;	\
	vrmlclass.enumerate = enump;	\
	vrmlclass.resolve = resp;		\
	vrmlclass.convert = conv;		\
	vrmlclass.finalize = fin;	

void init_spidermonkey_api(ScriptPriv *sc, SFNode *script)
{

	/*GCC port: classes are declared within code since JS_PropertyStub and co are exported symbols
	from JS runtime lib, so with non-constant addresses*/
	M4_SETUP_JS(globalClass, "global", 0, 
		JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, 
		JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub);

	M4_SETUP_JS(browserClass , "Browser", 0,
		JS_PropertyStub,  JS_PropertyStub,  JS_PropertyStub, JS_PropertyStub,
		JS_EnumerateStub, JS_ResolveStub,   JS_ConvertStub,  JS_FinalizeStub);

	M4_SETUP_JS(SFNodeClass, "SFNode", JSCLASS_HAS_PRIVATE,
		JS_PropertyStub,  JS_PropertyStub,  node_getProperty, node_setProperty,
		JS_EnumerateStub, JS_ResolveStub,   JS_ConvertStub,  node_finalize);

	M4_SETUP_JS(SFVec2fClass , "SFVec2f", JSCLASS_HAS_PRIVATE,
	  JS_PropertyStub,  JS_PropertyStub,  vec2f_getProperty, vec2f_setProperty,
	  JS_EnumerateStub, JS_ResolveStub,   JS_ConvertStub,  field_finalize);

	M4_SETUP_JS(SFVec3fClass , "SFVec3f", JSCLASS_HAS_PRIVATE,
	  JS_PropertyStub,  JS_PropertyStub,  vec3f_getProperty, vec3f_setProperty,
	  JS_EnumerateStub, JS_ResolveStub,   JS_ConvertStub,  field_finalize);

	M4_SETUP_JS(SFColorClass , "SFColor", JSCLASS_HAS_PRIVATE,
	  JS_PropertyStub,  JS_PropertyStub,  color_getProperty, color_setProperty,
	  JS_EnumerateStub, JS_ResolveStub,   JS_ConvertStub,  field_finalize);

	M4_SETUP_JS(MFInt32Class , "MFInt32", JSCLASS_HAS_PRIVATE,
	  JS_PropertyStub,  JS_PropertyStub, array_getElement,  array_setElement,  
	  JS_EnumerateStub, JS_ResolveStub,   JS_ConvertStub,  array_finalize);
	
	M4_SETUP_JS(MFBoolClass , "MFBool", JSCLASS_HAS_PRIVATE,
	  JS_PropertyStub,  JS_PropertyStub, array_getElement,  array_setElement,  
	  JS_EnumerateStub, JS_ResolveStub,   JS_ConvertStub,  array_finalize);
	M4_SETUP_JS(MFTimeClass , "MFTime", JSCLASS_HAS_PRIVATE,
	  JS_PropertyStub,  JS_PropertyStub, array_getElement,  array_setElement,  
	  JS_EnumerateStub, JS_ResolveStub,   JS_ConvertStub,  array_finalize);

	M4_SETUP_JS(MFFloatClass , "MFFloat", JSCLASS_HAS_PRIVATE,
	  JS_PropertyStub,  JS_PropertyStub, array_getElement,  array_setElement,  
	  JS_EnumerateStub, JS_ResolveStub,   JS_ConvertStub,  array_finalize);

	M4_SETUP_JS(MFUrlClass , "MFUrl", JSCLASS_HAS_PRIVATE,
	  JS_PropertyStub,  JS_PropertyStub, array_getElement,  array_setElement,  
	  JS_EnumerateStub, JS_ResolveStub,   JS_ConvertStub,  array_finalize);

	M4_SETUP_JS(MFStringClass , "MFString", JSCLASS_HAS_PRIVATE,
	  JS_PropertyStub,  JS_PropertyStub, array_getElement,  array_setElement,  
	  JS_EnumerateStub, JS_ResolveStub,   JS_ConvertStub,  array_finalize);

	M4_SETUP_JS(MFNodeClass , "MFNode", JSCLASS_HAS_PRIVATE,
	  JS_PropertyStub,  JS_PropertyStub, array_getElement,  array_setElement,  
	  JS_EnumerateStub, JS_ResolveStub,   JS_ConvertStub,  array_finalize);

	M4_SETUP_JS(MFVec2fClass , "MFVec2f", JSCLASS_HAS_PRIVATE,
	  JS_PropertyStub,  JS_PropertyStub, array_getElement,  array_setElement,  
	  JS_EnumerateStub, JS_ResolveStub,   JS_ConvertStub,  array_finalize);

	M4_SETUP_JS(MFVec3fClass , "MFVec3f", JSCLASS_HAS_PRIVATE,
	  JS_PropertyStub,  JS_PropertyStub, array_getElement,  array_setElement,  
	  JS_EnumerateStub, JS_ResolveStub,   JS_ConvertStub,  array_finalize);

	M4_SETUP_JS(MFColorClass , "MFColor", JSCLASS_HAS_PRIVATE,
	  JS_PropertyStub,  JS_PropertyStub, array_getElement,  array_setElement,  
	  JS_EnumerateStub, JS_ResolveStub,   JS_ConvertStub,  array_finalize);

	JS_SetErrorReporter(sc->js_ctx, script_error);

	sc->js_obj = JS_NewObject(sc->js_ctx, &globalClass, 0, 0 );
	JS_InitStandardClasses(sc->js_ctx, sc->js_obj);
	JS_DefineFunctions(sc->js_ctx, sc->js_obj, globalFunctions );

	JS_DefineProperty(sc->js_ctx, sc->js_obj, "FALSE", BOOLEAN_TO_JSVAL(0), 0, 0, JSPROP_READONLY | JSPROP_PERMANENT );
	JS_DefineProperty(sc->js_ctx, sc->js_obj, "TRUE", BOOLEAN_TO_JSVAL(1), 0, 0, JSPROP_READONLY | JSPROP_PERMANENT );
	JS_DefineProperty(sc->js_ctx, sc->js_obj, "_this", PRIVATE_TO_JSVAL(script), 0, 0, JSPROP_READONLY | JSPROP_PERMANENT );

	sc->js_browser = JS_DefineObject(sc->js_ctx, sc->js_obj, "Browser", &browserClass, 0, 0 );
	JS_DefineProperty(sc->js_ctx, sc->js_browser, "_this", PRIVATE_TO_JSVAL(script), 0, 0, JSPROP_READONLY | JSPROP_PERMANENT );
	JS_DefineFunctions(sc->js_ctx, sc->js_browser, browserFunctions);

	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &SFNodeClass, SFNodeConstructor, 1, 0, SFNodeMethods, 0, 0);
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &SFVec2fClass, SFVec2fConstructor, 0, SFVec2fProps, SFVec2fMethods, 0, 0);
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &SFVec3fClass, SFVec3fConstructor, 0, SFVec3fProps, SFVec3fMethods, 0, 0);
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &SFColorClass, SFColorConstructor, 0, SFColorProps, SFColorMethods, 0, 0);
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &MFInt32Class, MFArrayConstructor, 0, MFArrayProp, 0, 0, 0);
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &MFBoolClass, MFArrayConstructor, 0, MFArrayProp, 0, 0, 0);
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &MFFloatClass, MFArrayConstructor, 0, MFArrayProp, 0, 0, 0);
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &MFTimeClass, MFArrayConstructor, 0, MFArrayProp, 0, 0, 0);
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &MFStringClass, MFArrayConstructor, 0, MFArrayProp, 0, 0, 0);
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &MFUrlClass, MFArrayConstructor, 0, MFArrayProp, 0, 0, 0);
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &MFVec2fClass, MFVec2fConstructor, 0, MFArrayProp, MFArrayMethods, 0, 0);
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &MFVec3fClass, MFVec3fConstructor, 0, MFArrayProp, 0, 0, 0);
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &MFColorClass, MFColorConstructor, 0, MFArrayProp, 0, 0, 0);
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &MFNodeClass, MFArrayConstructor, 0, MFArrayProp, 0, 0, 0);

/*
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &SFVec4fClass, SFVec4fConstructor, 0, SFVec4fProps, SFVec4fMethods, 0, 0);
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &SFRotationClass, SFRotationConstructor, 0, SFRotationProps, SFRotationMethods, 0, 0);
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &MFVec4fClass, MFVec4fCons, 0, MFArrayProp, 0, 0, 0);
	JS_InitClass(sc->js_ctx, sc->js_obj, 0, &MFRotationClass, MFRotationCons, 0, MFArrayProp, 0, 0, 0);
*/

}



void JS_ToNodeField(JSContext *c, jsval val, FieldInfo *field, SFNode *owner, JSField *parent)
{
	jsdouble d;
	JSObject *obj;
	JSField *p, *from;
	jsuint len;
	jsval item;
	u32 i;

	if (JSVAL_IS_VOID(val)) return;
	if ((field->fieldType != FT_SFNode) && JSVAL_IS_NULL(val)) return;

	switch (field->fieldType) {
	case FT_SFBool:
	{
		if (JSVAL_IS_BOOLEAN(val)) {
			*((SFBool *) field->far_ptr) = JSVAL_TO_BOOLEAN(val);
			Script_FieldChanged(owner, parent, field);
		}
		return;
	}
	case FT_SFInt32:
	{
		if (JSVAL_IS_INT(val) ) {
			* ((SFInt32 *) field->far_ptr) = JSVAL_TO_INT(val);
			Script_FieldChanged(owner, parent, field);
		}
		return;
	}
	case FT_SFFloat:
	{
		if (JSVAL_IS_NUMBER(val) ) {
			JS_ValueToNumber(c, val, &d );
			*((SFFloat *) field->far_ptr) = (Float) d;
			Script_FieldChanged(owner, parent, field);
		}
		return;
    }
	case FT_SFTime:
	{
		if (JSVAL_IS_NUMBER(val) ) {
			JS_ValueToNumber(c, val, &d );
			*((SFTime *) field->far_ptr) = (Double) d;
			Script_FieldChanged(owner, parent, field);
		}
		return;
    }
	case FT_SFString:
	{
		if (JSVAL_IS_STRING(val)) {
			JSString *str = JSVAL_TO_STRING(val);
			if ( ((SFString*)field->far_ptr)->buffer) free(((SFString*)field->far_ptr)->buffer);
			((SFString*)field->far_ptr)->buffer = strdup(JS_GetStringBytes(str));
			Script_FieldChanged(owner, parent, field);
		}
		return;
	}
	case FT_SFURL:
	{
		if (JSVAL_IS_STRING(val)) {
			JSString *str = JSVAL_TO_STRING(val);
			if (((SFURL*)field->far_ptr)->url) free(((SFURL*)field->far_ptr)->url);
			((SFURL*)field->far_ptr)->url = strdup(JS_GetStringBytes(str));
			((SFURL*)field->far_ptr)->OD_ID = 0;
			Script_FieldChanged(owner, parent, field);
		}
		return;
	}

	default:
		break;
	}

	//from here we must have an object
	if (! JSVAL_IS_OBJECT(val)) return;
	obj = JSVAL_TO_OBJECT(val) ;

	switch (field->fieldType) {
	case FT_SFVec2f:
	{
		if ( &SFVec2fClass == JS_GetClass( obj ) ) {
			p = (JSField *) JS_GetPrivate(c, obj);
			SG_CopyField(field->far_ptr, p->field.far_ptr, FT_SFVec2f);
			Script_FieldChanged(owner, parent, field);
		}
		return;
	}
	case FT_SFVec3f:
	{
		if ( &SFVec3fClass == JS_GetClass( obj ) ) {
			p = (JSField *) JS_GetPrivate(c, obj);
			SG_CopyField(field->far_ptr, p->field.far_ptr, FT_SFVec3f);
			Script_FieldChanged(owner, parent, field);
		}
		return;
	}
	case FT_SFColor:
	{
		if ( &SFColorClass == JS_GetClass( obj ) ) {
			p = (JSField *) JS_GetPrivate(c, obj);
			SG_CopyField(field->far_ptr, p->field.far_ptr, FT_SFColor);
			Script_FieldChanged(owner, parent, field);
		}
		return;
	}
	case FT_SFNode:
	{
		/*replace object*/
		if (*((SFNode**)field->far_ptr)) 
			Node_Unregister(*((SFNode**)field->far_ptr), owner);

		if (JSVAL_IS_NULL(val)) {
			field->far_ptr = NULL;
			Script_FieldChanged(owner, parent, field);
		} else if ( &SFNodeClass == JS_GetClass( obj ) ) {
			SFNode *n;
			JSField *jsf = (JSField *) JS_GetPrivate(c, obj);

			* ((SFNode **)field->far_ptr) = * ((SFNode**)jsf->field.far_ptr);

			n = * ((SFNode**)field->far_ptr);
			Node_Register(n, owner);
			/*first assignment*/
			if (!jsf->is_attached) {
				jsf->is_attached = 1;
				jsf->temp_node = NULL;
				Node_GetField(owner, field->allIndex, &jsf->field);
			}
			Script_FieldChanged(owner, parent, field);
		}
		return;
	}
	default:
		break;
	}

	//from here we handle only MF fields 
	if ( ( &MFBoolClass != JS_GetClass( obj ) )
		&& ( &MFInt32Class != JS_GetClass( obj ) )
		&& ( &MFFloatClass != JS_GetClass( obj ) )
		&& ( &MFTimeClass != JS_GetClass( obj ) )
		&& ( &MFStringClass != JS_GetClass( obj ) )
		&& ( &MFUrlClass != JS_GetClass( obj ) )
		&& ( &MFVec2fClass != JS_GetClass( obj ) )
		&& ( &MFVec3fClass != JS_GetClass( obj ) )
		&& ( &MFNodeClass != JS_GetClass( obj ) )
/*		&& ( &MFVec4fClass != JS_GetClass( obj ) )
		&& ( &MFRotationClass != JS_GetClass( obj ) )
*/
		&& ( &MFColorClass != JS_GetClass( obj ) )
		) return;


	p = (JSField *) JS_GetPrivate(c, obj);
	JS_GetArrayLength(c, p->js_list, &len);


	/*special handling for MF node, reset list first*/
	if ( &MFNodeClass == JS_GetClass( obj ) ) {
		SFNode *child;
		Chain *list = * (Chain **) field->far_ptr;
		while (ChainGetCount(list)) {
			child = ChainGetEntry(list, 0);
			ChainDeleteEntry(list, 0);
			Node_Unregister(child, owner);
		}
	
		for (i=0; i<len; i++) {
			JS_GetElement(c, p->js_list, (jsint) i, &item);
			if (JSVAL_IS_NULL(item)) break;
			if ( &SFNodeClass != JS_GetClass( obj ) ) break;
			from = (JSField *) JS_GetPrivate(c, JSVAL_TO_OBJECT(item));

			child = * ((SFNode**)from->field.far_ptr);

			Node_Register(child, owner);
			/*first assignment, detach*/
			if (!from->is_attached) {
				from->is_attached = 1;
				from->temp_node = NULL;
			}
		}
		Script_FieldChanged(owner, parent, field);
		return;
	}
	

	/*realloc*/
	MFField_Reset(field->far_ptr, field->fieldType);
	MFField_Alloc(field->far_ptr, field->fieldType, len);

	/*assign each slot*/
	for (i=0; i<len; i++) {
		JS_GetElement(c, p->js_list, (jsint) i, &item);

		switch (field->fieldType) {
		case FT_MFBool:
		{
			if (JSVAL_IS_BOOLEAN(item)) {
				((MFBool*)field->far_ptr)->vals[i] = (Bool) JSVAL_TO_BOOLEAN(item);
			}
			break;
		}
		case FT_MFInt32:
		{
			if (JSVAL_IS_INT(item)) {
				((MFInt32 *)field->far_ptr)->vals[i] = (s32) JSVAL_TO_INT(item);
			}
			break;
		}
		case FT_MFFloat:
		{
			if (JSVAL_IS_NUMBER(item)) {
				JS_ValueToNumber(c, item, &d);
				((MFFloat *)field->far_ptr)->vals[i] = (Float) d;
			}
			break;
		}
		case FT_MFTime:
		{
			if (JSVAL_IS_NUMBER(item)) {
				JS_ValueToNumber(c, item, &d);
				((MFTime *)field->far_ptr)->vals[i] = d;
			}
			break;
		}
		case FT_MFString:
		{
			if (JSVAL_IS_STRING(item)) {
				JSString *str = JSVAL_TO_STRING(item);
				((MFString *)field->far_ptr)->vals[i] = strdup(JS_GetStringBytes(str));
			}
			break;
		}
		case FT_MFURL:
		{
			if (JSVAL_IS_STRING(item)) {
				JSString *str = JSVAL_TO_STRING(item);
				((MFURL *)field->far_ptr)->vals[i].url = strdup(JS_GetStringBytes(str));
				((MFURL *)field->far_ptr)->vals[i].OD_ID = 0;
			}
			break;
		}

		case FT_MFVec2f:
		{
			if (JSVAL_IS_OBJECT(item)) {
				from = (JSField *) JS_GetPrivate(c, JSVAL_TO_OBJECT(item));
				SG_CopyField(& ((MFVec2f*)field->far_ptr)->vals[i], from->field.far_ptr, FT_SFVec2f);
			}
			break;
		}
		case FT_MFVec3f:
		{
			if (JSVAL_IS_OBJECT(item)) {
				from = (JSField *) JS_GetPrivate(c, JSVAL_TO_OBJECT(item));
				SG_CopyField(& ((MFVec3f*)field->far_ptr)->vals[i], from->field.far_ptr, FT_SFVec3f);
			}
			break;
		}
		case FT_MFColor:
		{
			if (JSVAL_IS_OBJECT(item)) {
				from = (JSField *) JS_GetPrivate(c, JSVAL_TO_OBJECT(item));
				SG_CopyField(& ((MFColor*)field->far_ptr)->vals[i], from->field.far_ptr, FT_SFColor);
			}
			break;
		}

		default:
			return;
		}
	}
	Script_FieldChanged(owner, parent, field);
}

#define SETUP_FIELD	\
		jsf = NewJSField(1);	\
		jsf->owner = parent;	\
		Node_GetField(parent, field->allIndex, &jsf->field);	\

#define SETUP_MF_FIELD	\
		jsf = (JSField *) JS_GetPrivate(priv->js_ctx, obj);	\
		jsf->owner = parent;		\
		Node_GetField(parent, field->allIndex, &jsf->field);	\



jsval JS_ToJSField(ScriptPriv *priv, FieldInfo *field, SFNode *parent)
{
	u32 i;
	JSObject *obj = NULL;
	JSField *jsf = NULL;
	JSField *slot = NULL;
	SFNode *n;
	jsdouble *d;
	jsval newVal;
	JSString *s;

	switch (field->fieldType) {
    case FT_SFBool:
		return BOOLEAN_TO_JSVAL( * ((SFBool *) field->far_ptr) );
	case FT_SFInt32:
		return INT_TO_JSVAL(  * ((SFInt32 *) field->far_ptr));
    case FT_SFFloat:
		d = JS_NewDouble(priv->js_ctx, * ((SFFloat *) field->far_ptr));
		return DOUBLE_TO_JSVAL(d);
	case FT_SFTime:
		d = JS_NewDouble(priv->js_ctx, * ((SFTime *) field->far_ptr));
		return DOUBLE_TO_JSVAL(d);
	case FT_SFString:
    {
		s = JS_NewStringCopyZ(priv->js_ctx, ((SFString *) field->far_ptr)->buffer);
		return STRING_TO_JSVAL( s );
    }
	case FT_SFURL:
    {
		SFURL *url = (SFURL *)field->far_ptr;
		if (url->OD_ID > 0) {
			char msg[30];
			sprintf(msg, "od:%d", url->OD_ID);
			s = JS_NewStringCopyZ(priv->js_ctx, (const char *) msg);
		} else {
			s = JS_NewStringCopyZ(priv->js_ctx, (const char *) url->url);
		}
		return STRING_TO_JSVAL( s );
    }
    case FT_SFVec2f:
		SETUP_FIELD
		obj = JS_NewObject(priv->js_ctx, &SFVec2fClass, 0, priv->js_obj);
		break;
    case FT_SFVec3f:
		SETUP_FIELD
		obj = JS_NewObject(priv->js_ctx, &SFVec3fClass, 0, priv->js_obj);
		break;
    case FT_SFColor:
		SETUP_FIELD
		obj = JS_NewObject(priv->js_ctx, &SFColorClass, 0, priv->js_obj);
		break;
	case FT_SFNode:
		SETUP_FIELD
		obj = JS_NewObject(priv->js_ctx, &SFNodeClass, 0, priv->js_obj);
		break;


	case FT_MFBool:
	{
		MFBool *f = (MFBool *) field->far_ptr;
		obj = JS_ConstructObject(priv->js_ctx, &MFBoolClass, 0, priv->js_obj);
		SETUP_MF_FIELD
		for (i = 0; i<f->count; i++) {
			jsval newVal = BOOLEAN_TO_JSVAL(f->vals[i]);
			JS_SetElement(priv->js_ctx, obj, (jsint) i, &newVal);
		}
		break;
	}
	case FT_MFInt32:
	{
		MFInt32 *f = (MFInt32 *) field->far_ptr;
		obj = JS_ConstructObject(priv->js_ctx, &MFInt32Class, 0, priv->js_obj);
		SETUP_MF_FIELD
		for (i=0; i<f->count; i++) {
			jsval newVal = INT_TO_JSVAL(f->vals[i]);
			JS_SetElement(priv->js_ctx, obj, (jsint) i, &newVal);
		}
		break;
	}
	case FT_MFFloat:
	{
		MFFloat *f = (MFFloat *) field->far_ptr;
		obj = JS_ConstructObject(priv->js_ctx, &MFFloatClass, 0, priv->js_obj);
		SETUP_MF_FIELD
		for (i=0; i<f->count; i++) {
			jsval newVal = DOUBLE_TO_JSVAL(JS_NewDouble(priv->js_ctx, f->vals[i]));
			JS_SetElement(priv->js_ctx, obj, (jsint) i, &newVal);
		}
		break;
	}
	case FT_MFTime:
	{
		MFTime *f = (MFTime *) field->far_ptr;
		obj = JS_ConstructObject(priv->js_ctx, &MFTimeClass, 0, priv->js_obj);
		SETUP_MF_FIELD
		for (i=0; i<f->count; i++) {
			jsval newVal = DOUBLE_TO_JSVAL( JS_NewDouble(priv->js_ctx, f->vals[i]) );
			JS_SetElement(priv->js_ctx, obj, (jsint) i, &newVal);
		}
		break;
	}
	case FT_MFString:
	{
		MFString *f = (MFString *) field->far_ptr;
		obj = JS_ConstructObject(priv->js_ctx, &MFStringClass, 0, priv->js_obj);
		SETUP_MF_FIELD
		for (i=0; i<f->count; i++) {
			s = JS_NewStringCopyZ(priv->js_ctx, f->vals[i]);
			newVal = STRING_TO_JSVAL( s );
			JS_SetElement(priv->js_ctx, obj, (jsint) i, &newVal);
		}
		break;
	}
	case FT_MFURL:
	{
		MFURL *f = (MFURL *) field->far_ptr;
		obj = JS_ConstructObject(priv->js_ctx, &MFUrlClass, 0, priv->js_obj);
		SETUP_MF_FIELD
		for (i=0; i<f->count; i++) {
			if (f->vals[i].OD_ID > 0) {
				char msg[30];
				sprintf(msg, "od:%d", f->vals[i].OD_ID);
				s = JS_NewStringCopyZ(priv->js_ctx, (const char *) msg);
			} else {
				s = JS_NewStringCopyZ(priv->js_ctx, f->vals[i].url);
			}
			newVal = STRING_TO_JSVAL( s );
			JS_SetElement(priv->js_ctx, obj, (jsint) i, &newVal);
		}
		break;
	}

	case FT_MFVec2f:
	{
		MFVec2f *f = (MFVec2f *) field->far_ptr;
		obj = JS_ConstructObject(priv->js_ctx, &MFVec2fClass, 0, priv->js_obj);
		SETUP_MF_FIELD
		for (i=0; i<f->count; i++) {
			JSObject *pf = JS_NewObject(priv->js_ctx, &SFVec2fClass, 0, obj);
			newVal = OBJECT_TO_JSVAL(pf);
			slot = SFVec2f_Create(priv->js_ctx, pf, 0, f->vals[i].x, f->vals[i].y);
			slot->owner = parent;
			JS_SetElement(priv->js_ctx, obj, (jsint) i, &newVal);
		}
		break;
	}
	case FT_MFVec3f:
	{
		MFVec3f *f = (MFVec3f *) field->far_ptr;
		obj = JS_ConstructObject(priv->js_ctx, &MFVec3fClass, 0, priv->js_obj);
		SETUP_MF_FIELD
		for (i=0; i<f->count; i++) {
			JSObject *pf = JS_NewObject(priv->js_ctx, &SFVec3fClass, 0, obj);
			newVal = OBJECT_TO_JSVAL(pf);
			slot = SFVec3f_Create(priv->js_ctx, pf, 0, f->vals[i].x, f->vals[i].y, f->vals[i].z);
			slot->owner = parent;
			JS_SetElement(priv->js_ctx, obj, (jsint) i, &newVal);
		}
		break;
	}
	case FT_MFColor:
	{
		MFColor *f = (MFColor *) field->far_ptr;
		obj = JS_ConstructObject(priv->js_ctx, &MFColorClass, 0, priv->js_obj);
		SETUP_MF_FIELD
		for (i=0; i<f->count; i++) {
			JSObject *pf = JS_NewObject(priv->js_ctx, &SFColorClass, 0, obj);
			newVal = OBJECT_TO_JSVAL(pf);
			slot = SFColor_Create(priv->js_ctx, pf, 0, f->vals[i].red, f->vals[i].green, f->vals[i].blue);
			slot->owner = parent;
			JS_SetElement(priv->js_ctx, obj, (jsint) i, &newVal);
		}
		break;
	}

	case FT_MFNode:
	{
		Chain *f = * ((Chain**)field->far_ptr);
		obj = JS_ConstructObject(priv->js_ctx, &MFNodeClass, 0, priv->js_obj);
		SETUP_MF_FIELD
		for (i=0; i<ChainGetCount(f); i++) {
			JSObject *pf = JS_NewObject(priv->js_ctx, &SFNodeClass, 0, obj);
			n = ChainGetEntry(f, i);
			slot = NewJSField(1);
			slot->owner = parent;
			slot->temp_node = n;
			slot->field.far_ptr = & slot->temp_node;
			slot->field.fieldType = FT_SFNode;
			JS_SetPrivate(priv->js_ctx, pf, slot);

			newVal = OBJECT_TO_JSVAL(pf);
			JS_SetElement(priv->js_ctx, obj, (jsint) i, &newVal);
		}
		break;
	}

	//not supported
    default:
		return JSVAL_NULL;
    }

	if (!obj) return JSVAL_NULL;
	//store field associated with object if needed
	if (jsf) JS_SetPrivate(priv->js_ctx, obj, jsf);
	
	return OBJECT_TO_JSVAL(obj);

}



/*all spidermonkey specific stuff*/
static JSRuntime *js_runtime = 0;
static u32 nb_inst = 0;
const long MAX_HEAP_BYTES = 4L * 1024L * 1024L;
const long STACK_CHUNK_BYTES = 4024L;

static void JS_PreDestroy(SFNode *node)
{
	jsval fval, rval;
	ScriptPriv *priv = node->sgprivate->privateStack;
	if (!priv) return;
	
	if (JS_LookupProperty(priv->js_ctx, priv->js_obj, "shutdown", &fval))
		if (! JSVAL_IS_VOID(fval))
			JS_CallFunctionValue(priv->js_ctx, priv->js_obj, fval, 0, NULL, &rval);

	JS_DestroyContext(priv->js_ctx);
	nb_inst --;
	if (nb_inst == 0) {
		JS_DestroyRuntime(js_runtime);
		js_runtime = 0;
	}
}

static void JS_InitScriptFields(ScriptPriv *priv, SFNode *sc)
{
	u32 i;
	ScriptField *sf;
	FieldInfo info;
	jsval val;

    for (i=0; i < ChainGetCount(priv->fields); i++) { 
		sf = ChainGetEntry(priv->fields, i);

		switch (sf->eventType) {
		case ET_EventIn:
			break;
		case ET_EventOut:
			Node_GetField(sc, sf->ALL_index, &info);
			val = JS_ToJSField(priv, &info, sc);
			/*for native types directly modified*/
			JS_DefineProperty(priv->js_ctx, priv->js_obj, (const char *) sf->name, val, 0, eventOut_setProperty, JSPROP_PERMANENT );
			break;
		default:
			Node_GetField(sc, sf->ALL_index, &info);
			val = JS_ToJSField(priv, &info, sc);
			JS_DefineProperty(priv->js_ctx, priv->js_obj, (const char *) sf->name, val, 0, 0, JSPROP_PERMANENT);
			break;
		}
    }
}

static void JS_EventIn(SFNode *node, FieldInfo *in_field)
{
	jsval fval, rval;
	Double time;
	jsval argv[2];
	ScriptField *sf;
	FieldInfo t_info;
	ScriptPriv *priv;
	u32 i;	
	priv = node->sgprivate->privateStack;

	/*no support for change of static fields*/
	if (in_field->allIndex<3) return;
	
	sf = ChainGetEntry(priv->fields, in_field->allIndex - 3);
	time = Node_GetSceneTime(node);

//	if (sf->last_route_time == time) return; 
	sf->last_route_time = time;

	//locate function
	if (! JS_LookupProperty(priv->js_ctx, priv->js_obj, sf->name, &fval)) return;
	if (JSVAL_IS_VOID(fval)) return;

	argv[0] = JS_ToJSField(priv, in_field, node);

	memset(&t_info, 0, sizeof(FieldInfo));
	t_info.far_ptr = &sf->last_route_time;
	t_info.fieldType = FT_SFTime;
	t_info.allIndex = -1;
	argv[1] = JS_ToJSField(priv, &t_info, node);

	/*protect args*/
	if (JSVAL_IS_GCTHING(argv[0])) JS_AddRoot(priv->js_ctx, &argv[0]);
	if (JSVAL_IS_GCTHING(argv[1])) JS_AddRoot(priv->js_ctx, &argv[1]);

	JS_CallFunctionValue(priv->js_ctx, priv->js_obj, fval, 2, argv, &rval);

	/*release args*/
	if (JSVAL_IS_GCTHING(argv[0])) JS_RemoveRoot(priv->js_ctx, &argv[0]);
	if (JSVAL_IS_GCTHING(argv[1])) JS_RemoveRoot(priv->js_ctx, &argv[1]);

	/*flush event out*/
	for (i=0; i<ChainGetCount(priv->fields); i++) {
		sf = ChainGetEntry(priv->fields, i);
		if (sf->activate_event_out) {
			sf->activate_event_out = 0;
			Node_OnEventOut(node, sf->ALL_index);
		}
	}
}


static void JSScript_Load(SFNode *node)
{
	char *str;
	JSBool ret;
	jsval rval, fval;
	B_Script *script = (B_Script *)node;
	ScriptPriv *priv = (ScriptPriv *) node->sgprivate->privateStack;

	if (!priv || priv->is_loaded) return;
	if (!script->url.count) return;
	priv->is_loaded = 1;
	priv->JS_PreDestroy = JS_PreDestroy;
	priv->JS_EventIn = JS_EventIn;

	/*JS load*/
	if (!js_runtime) {
		js_runtime = JS_NewRuntime(MAX_HEAP_BYTES);
		assert(js_runtime);
	}
	priv->js_ctx = JS_NewContext(js_runtime, STACK_CHUNK_BYTES);
	assert(priv->js_ctx);
	nb_inst++;
	JS_SetContextPrivate(priv->js_ctx, node);
	init_spidermonkey_api(priv, node);

	/*setup fields interfaces*/
	JS_InitScriptFields(priv, node);

	str = script->url.vals[0].script_text;

	if (!strnicmp(str, "javascript:", 11)) str += 11;
	else if (!strnicmp(str, "vrmlscript:", 11)) str += 11;
	else if (!strnicmp(str, "ecmascript:", 11)) str += 11;
	else if (!strnicmp(str, "mpeg4script:", 12)) str += 12;
	
	ret = JS_EvaluateScript(priv->js_ctx, priv->js_obj, str, strlen(str), 0, 0, &rval);
	if (ret==JS_FALSE) return;

	/*call initialize if present*/
	if (! JS_LookupProperty(priv->js_ctx, priv->js_obj, "initialize", &fval)) return;
	if (JSVAL_IS_VOID(fval)) return;
	JS_CallFunctionValue(priv->js_ctx, priv->js_obj, fval, 0, NULL, &rval);
}

#endif

#endif


/*set JavaScript interface*/
void SG_SetJavaScriptAPI(LPSCENEGRAPH scene, JSInterface *ifce)
{
#ifdef M4_DEF_Script
	if (!scene) return;
	scene->js_ifce = ifce;

#ifdef M4_USE_SPIDERMONKEY
	scene->Script_Load = JSScript_Load;
#endif

#endif
}
