/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Stream Management sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include <gpac/m4_author.h>
#include <gpac/intern/m4_esm_dev.h>
#include "MediaMemory.h"


static Bool check_in_scene(InlineScene *scene, LPODMANAGER odm)
{
	u32 i;
	LPODMANAGER root;
	if (!scene) return 0;
	root = scene->root_od;
	while (1) {
		if (odm == root) return 1;
		if (!root->remote_OD) break;
		root = root->remote_OD;
	}
	scene = root->subscene;

	for (i=0; i<ChainGetCount(scene->ODlist); i++) {
		ODManager *ptr = ChainGetEntry(scene->ODlist, i);
		while (1) {
			if (ptr == odm) return 1;
			if (!ptr->remote_OD) break;
			ptr = ptr->remote_OD;
		}
		if (check_in_scene(ptr->subscene, odm)) return 1;
	}
	return 0;
}

static Bool M4T_CheckODM(MPEG4CLIENT term, LPODMANAGER odm)
{
	if (!term->root_scene) return 0;
	return check_in_scene(term->root_scene, odm);
}


/*returns top-level OD of the presentation*/
LPODMANAGER M4T_GetRootOD(MPEG4CLIENT term)
{
	if (!term) return NULL;
	if (!term->root_scene) return NULL;
	return term->root_scene->root_od;
}

/*returns number of sub-ODs in the current root. scene_od must be an inline OD*/
u32 M4T_GetODCount(MPEG4CLIENT term, LPODMANAGER scene_od)
{
	if (!term || !scene_od) return 0;
	if (!M4T_CheckODM(term, scene_od)) return 0;
	if (!scene_od->subscene) return 0;
	return ChainGetCount(scene_od->subscene->ODlist);
}

/*returns indexed (0-based) OD manager in the scene*/
LPODMANAGER M4T_GetODManager(MPEG4CLIENT term, LPODMANAGER scene_od, u32 index)
{
	if (!term || !scene_od) return NULL;
	if (!M4T_CheckODM(term, scene_od)) return NULL;
	if (!scene_od->subscene) return NULL;
	return (LPODMANAGER ) ChainGetEntry(scene_od->subscene->ODlist, index);
}

/*returns remote ODManager of this OD if any, NULL otherwise*/
LPODMANAGER M4T_GetRemoteOD(MPEG4CLIENT term, LPODMANAGER odm)
{
	if (!term || !odm) return NULL;
	if (!M4T_CheckODM(term, odm)) return NULL;
	return odm->remote_OD;
}


u32 M4T_IsInlineOD(MPEG4CLIENT term, LPODMANAGER odm)
{
	Bool IS_IsProtoLibObject(InlineScene *is, ODManager *odm);

	if (!term || !odm) return 0;
	if (!M4T_CheckODM(term, odm)) return 0;

	if (!odm->subscene) return 0;
	if (odm->parentscene) return IS_IsProtoLibObject(odm->parentscene, odm) ? 2 : 1;
	return 1;
}

M4Err M4T_GetODInfo(MPEG4CLIENT term, LPODMANAGER odm, ODInfo *info)
{
	if (!term || !odm || !info) return M4BadParam;
	if (!M4T_CheckODM(term, odm)) return M4BadParam;

	memset(info, 0, sizeof(ODInfo));
	info->od = odm->OD;

	info->duration = odm->duration;
	info->duration /= 1000;
	if (odm->codec) {
		info->current_time = odm->codec->CB ? odm->current_time : CK_GetTime(odm->codec->ck);
		info->current_time /= 1000;
	} else if (odm->subscene) {
		info->current_time = CK_GetTime(odm->subscene->bifs_codec->ck);
		info->current_time /= 1000;
	}

	if (odm->is_open) {
		Clock *ck = ODM_GetMediaClock(odm);
		if (CK_IsStarted(ck)) {
			info->status = 1;
		} else {
			info->status = 2;
		}
	}
	
	info->buffer = -2;
	info->db_unit_count = 0;
	if (odm->is_open) {
		u32 i, buf;
		Clock *ck;
		info->buffer = -1;
		buf = 0;
		for (i=0; i<ChainGetCount(odm->channels); i++) {
			Channel *ch = ChainGetEntry(odm->channels, i);
			info->db_unit_count += ch->AU_Count;
			if (!ch->is_pulling) {
				if (ch->MaxBuffer) info->buffer = 0;
				buf += ch->BufferTime;
			}
		}
		if (buf) info->buffer = (s32) buf;

		ck = ODM_GetMediaClock(odm);
		if (ck) info->clock_drift = ck->drift;
	}


	info->has_profiles = (odm->Audio_PL<0) ? 0 : 1;
	if (info->has_profiles) {
		info->inline_pl = odm->ProfileInlining;
		info->OD_pl = odm->OD_PL; 
		info->scene_pl = odm->Scene_PL;
		info->audio_pl = odm->Audio_PL;
		info->visual_pl = odm->Visual_PL;
		info->graphics_pl = odm->Graphics_PL;
	}	

	info->service_handler = odm->net_service->ifce->plugin_name;
	info->service_url = odm->net_service->url;
	if (odm->net_service->owner == odm) info->owns_service = 1;

	if (odm->codec && odm->codec->decio) {
		if (!odm->codec->decio->GetCodecName) {
			info->codec_name = odm->codec->decio->plugin_name;
		} else {
			info->codec_name = odm->codec->decio->GetCodecName(odm->codec->decio);
		}
		info->od_type = odm->codec->type;
		if (odm->codec->CB) {
			info->cb_max_count = odm->codec->CB->Capacity;
			info->cb_unit_count = odm->codec->CB->UnitCount;
		}
	}
	
	if (odm->mo) {
		switch (info->od_type) {
		case M4ST_VISUAL:
			info->width = odm->mo->width;
			info->height = odm->mo->height;
			info->pixelFormat = odm->mo->pixelFormat;
			break;
		case M4ST_AUDIO:
			info->sample_rate = odm->mo->sample_rate;
			info->bits_per_sample = odm->mo->bits_per_sample;
			info->num_channels = odm->mo->num_channels;
			info->clock_drift = 0;
			break;
		}
	}
	return M4OK;
}


#ifdef M4_DEF_WorldInfo

B_WorldInfo *M4T_GetWorldInfo(MPEG4CLIENT term, LPODMANAGER scene_od)
{
	LPODMANAGER odm;
	if (!term) return NULL;
	if (!scene_od) {
		if (!term->root_scene) return NULL;
		return term->root_scene->world_info;
	}
	if (!M4T_CheckODM(term, scene_od)) return NULL;

	odm = scene_od;
	while (odm->remote_OD) odm = odm->remote_OD;
	if (odm->subscene) return odm->subscene->world_info;
	return odm->parentscene->world_info;
}

#endif

M4Err M4T_DumpSceneGraph(MPEG4CLIENT term, FILE *dump, Bool xmt_dump, Bool skip_protos, LPODMANAGER scene_od)
{
	LPSCENEGRAPH sg;
	LPODMANAGER odm;
	LPSCENEDUMPER dumper;

	if (!term || !term->root_scene) return M4BadParam;
	if (!scene_od) {
		if (!term->root_scene) return M4BadParam;
		odm = term->root_scene->root_od;
	} else {
		odm = scene_od;
		if (!M4T_CheckODM(term, scene_od)) 
			odm = term->root_scene->root_od;
	}

	while (odm->remote_OD) odm = odm->remote_OD;
	if (odm->subscene) {
		if (!odm->subscene->graph) return M4IOErr;
		sg = odm->subscene->graph;
	} else {
		if (!odm->parentscene->graph) return M4IOErr;
		sg = odm->parentscene->graph;
	}

	dumper = NewSceneDumper(dump, ' ', xmt_dump);
	if (!dumper) return M4IOErr;
	SD_SetSceneGraph(dumper, sg);
	SD_DumpGraph(dumper, skip_protos, 0);
	DeleteSceneDumper(dumper);
	return M4OK;
}

