/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Stream Management sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include <gpac/intern/m4_esm_dev.h>
#include "MediaMemory.h"
#include "MediaControl.h"


MediaObject *MO_FindObject(SFNode *node, MFURL *url)
{
	u32 obj_type;
	InlineScene *is;
	LPSCENEGRAPH sg = Node_GetParentGraph(node);
	if (!sg) return NULL;
	is = SG_GetPrivate(sg);
	if (!is) return NULL;

	/*keep track of the kind of object expected if URL is not using OD scheme*/
	switch (Node_GetTag(node)) {
#ifdef M4_DEF_MovieTexture
	case TAG_MovieTexture: obj_type = NM_OD_VIDEO; break;
#endif
#ifdef M4_DEF_ImageTexture
	case TAG_ImageTexture: obj_type = NM_OD_VIDEO; break;
#endif
#ifdef M4_DEF_AudioSource
	case TAG_AudioSource: obj_type = NM_OD_AUDIO; break;
#endif
#ifdef M4_DEF_AudioClip
	case TAG_AudioClip: obj_type = NM_OD_AUDIO; break;
#endif
#ifdef M4_DEF_AnimationStream
	case TAG_AnimationStream: obj_type = NM_OD_BIFS; break;
#endif
#ifdef M4_DEF_Inline
	case TAG_Inline: obj_type = NM_OD_SCENE; break;
#endif
#ifdef M4_DEF_InputSensor
	case TAG_InputSensor: obj_type = NM_OD_INTERACT; break;
#endif
	default: obj_type = NM_OD_UNDEF; break;
	}
	return IS_GetMediaObject(is, url, obj_type);
}

MediaObject *NewMediaObject()
{
	MediaObject *mo;
	mo = (MediaObject *) malloc(sizeof(MediaObject));
	memset(mo, 0, sizeof(MediaObject));
	mo->speed = 1.0;
	return mo;
}


void MO_UpdateCaps(MediaObject *mo)
{
	CapObject cap;
	ODManager * odm = (ODManager * )mo->od_man;
	mo->mo_flags &= ~MO_IS_INIT;

	if (mo->type == NM_OD_VIDEO) {
		cap.CapCode = CAP_WIDTH;
		Codec_GetCap(odm->codec, &cap);
		mo->width = cap.cap.valueINT;
		cap.CapCode = CAP_HEIGHT;
		Codec_GetCap(odm->codec, &cap);
		mo->height = cap.cap.valueINT;
		cap.CapCode = CAP_PITCH;
		Codec_GetCap(odm->codec, &cap);
		mo->stride = cap.cap.valueINT;
		cap.CapCode = CAP_COLORMODE;
		Codec_GetCap(odm->codec, &cap);
		mo->pixelFormat = cap.cap.valueINT;
		odm->codec->bytes_per_sec = 0;
	}
	else if (mo->type == NM_OD_AUDIO) {
		cap.CapCode = CAP_SAMPLERATE;
		Codec_GetCap(odm->codec, &cap);
		mo->sample_rate = cap.cap.valueINT;
		cap.CapCode = CAP_NBCHANNELS;
		Codec_GetCap(odm->codec, &cap);
		mo->num_channels = cap.cap.valueINT;
		cap.CapCode = CAP_BITSPERSAMPLE;
		Codec_GetCap(odm->codec, &cap);
		mo->bits_per_sample = cap.cap.valueINT;
		odm->codec->bytes_per_sec = mo->sample_rate * mo->num_channels * mo->bits_per_sample / 8;
	}
}

Bool MO_FetchFrame(MediaObject *mo, Bool resync, Bool *eos)
{
	Bool ret;
	u32 obj_time;
	LPCUBUFFER CU;
	ODManager *odm;
	*eos = 0;

	if (!mo) return 0;
	odm = (ODManager * )mo->od_man;

	if (!odm || !odm->codec || !odm->codec->CB) return 0;
	/*check if we need to play*/
	if (mo->num_open && !odm->is_open) {
		ODM_Start(odm);
		return 0;
	}

	ret = 0;
	CB_Lock(odm->codec->CB, 1);

	/*end of stream */
	*eos = CB_IsEndOfStream(odm->codec->CB);
	//not running
	if (!CB_IsRunning(odm->codec->CB)) goto exit;

	/*if frame locked return it*/
	if (mo->num_fetched) {
		*eos = 0;
		ret = 1;
		mo->num_fetched++;
		goto exit;
	}

	/*new frame to fetch, lock*/
	CU = CB_GetOutput(odm->codec->CB);
	/*no output*/
	if (!CU || (CU->RenderedLength == CU->dataLength)) goto exit;

	assert(CU->TS >= odm->codec->CB->LastRenderedTS);
	if (odm->codec->CB->UnitCount==1) resync = 0;

	/*resync*/
	if (resync) {
		obj_time = CK_GetTime(odm->codec->ck);
		while (CU->TS < obj_time) {
			if (!CU->next->dataLength) break;
			/*figure out closest time*/
			if (CU->next->TS > obj_time) {
				/*current frame is closer to object clock than next one, keep it*/
				if ((obj_time - CU->TS) < (CU->next->TS - obj_time) ) break;

			}
			/*discard*/
			CU->RenderedLength = CU->dataLength = 0;
			CB_DropOutput(odm->codec->CB);
			/*get next*/
			CU = CB_GetOutput(odm->codec->CB);
		}
	}
	mo->current_size = CU->dataLength - CU->RenderedLength;
	mo->current_frame = CU->data + CU->RenderedLength;
	mo->current_ts = CU->TS;
	/*also adjust CU time based on consummed bytes in input, since some codecs output very large audio chunks*/
	if (odm->codec->bytes_per_sec) mo->current_ts += CU->RenderedLength * 1000 / odm->codec->bytes_per_sec;

	mo->num_fetched++;
	ret = 1;

/*	
	obj_time = CK_GetTime(odm->codec->ck);
	fprintf(stdout, "At OTB %d fetch frame TS %d size %d - %d unit in CB\n", obj_time, mo->current_ts, mo->current_size, odm->codec->CB->UnitCount);
*/
	
exit:

	CB_Lock(odm->codec->CB, 0);
	return ret;
}

void MO_ReleaseFrame(MediaObject *mo, u32 nb_bytes, Bool forceDrop)
{
	u32 obj_time;
	ODManager * odm = (ODManager * )mo->od_man;
	if (!mo || !odm || !mo->num_fetched) return;
	mo->num_fetched--;
	if (mo->num_fetched) return;

	CB_Lock(odm->codec->CB, 1);

	/*perform a sanity check on TS since the CB may have changed status - this may happen in 
	temporal scalability only*/
	if (odm->codec->CB->output->dataLength ) {
		assert(odm->codec->CB->output->RenderedLength + nb_bytes <= odm->codec->CB->output->dataLength);
		odm->codec->CB->output->RenderedLength += nb_bytes;

		/*discard frame*/
		if ( odm->codec->CB->output->RenderedLength == odm->codec->CB->output->dataLength) {
			if (forceDrop) {
				CB_DropOutput(odm->codec->CB);
			} else {
				obj_time = CK_GetTime(odm->codec->ck);
				if (odm->codec->CB->output->next->dataLength) { 
					if (2*obj_time < mo->current_ts + odm->codec->CB->output->next->TS ) {
						odm->codec->CB->output->RenderedLength = 0;
					} else {
						CB_DropOutput(odm->codec->CB);
					}
				} else {
					CB_DropOutput(odm->codec->CB);
				}
			}
		}
	}
	CB_Lock(odm->codec->CB, 0);
}

void MO_GetObjectTime(MediaObject *mo, u32 *obj_time)
{
	ODManager * odm = (ODManager * )mo->od_man;
	if (!mo || !odm || !obj_time) return;

	/*regular media codec...*/
	if (odm->codec) {
		/*get absolute clock (without drift) for audio*/
		if (odm->codec->type==M4ST_AUDIO)  
			*obj_time = CK_GetRealTime(odm->codec->ck);
		else
			*obj_time = CK_GetTime(odm->codec->ck);
	}
	/*BIFS object */
	else if (odm->subscene && odm->subscene->bifs_codec) {
		*obj_time = CK_GetTime(odm->subscene->bifs_codec->ck);
	} 
	/*unknown / unsupported object*/
	else {
		*obj_time = 0;
	}
}


void MO_Play(MediaObject *mo)
{
	ODManager *odm;
	if (!mo) return;

	odm = (ODManager * )mo->od_man;
	if (!mo->num_open && odm) {
		ODM_Start(odm);
	} else {
		if (mo->num_to_restart) mo->num_restart--;
		if (!mo->num_restart && (mo->num_to_restart==mo->num_open+1) ) {
			MC_Restart(odm);
			mo->num_to_restart = mo->num_restart = 0;
		}
	}
	
	mo->num_open++;
}

void MO_Stop(MediaObject *mo)
{
	ODManager *odm;
	if (!mo) return;

	odm = (ODManager * )mo->od_man;
	assert(mo->num_open);
	mo->num_open--;
	if (!mo->num_open && odm) {
		ODM_Stop(odm, 0);
	} else {
		if (!mo->num_to_restart) {
			mo->num_restart = mo->num_to_restart = mo->num_open + 1;
		}
	}
}

void MO_Restart(MediaObject *mo)
{
	ODManager *odm;
#ifdef M4_DEF_MediaControl
	MediaControlStack *ctrl;
#else
	void *ctrl = NULL;
#endif
	if (!mo) return;

	odm = (ODManager * )mo->od_man;
	assert(mo->num_open);
	/*this should not happen (inline scene are restarted internally)*/
	assert(!odm->subscene);

#ifdef M4_DEF_MediaControl
	ctrl = ODM_GetMediaControl(odm);
#endif
	
	if (!ctrl) {
		Clock *ck = ODM_GetMediaClock(odm->parentscene->root_od);
		/*don't restart if sharing parent scene clock*/
		if (ODM_SharesClock(odm, ck)) return;
	}
	/*all other cases, call restart to take into account clock references*/
	MC_Restart(odm);
}

Bool MO_UrlChanged(MediaObject *mo, MFURL *url)
{
	if (!mo) return (url ? 1 : 0);
	if (URL_GetODID(url) != mo->OD_ID) return 1;
	return 0;
}


void MO_SetSpeed(MediaObject *mo, Float speed)
{
	ODManager *odm;
#ifdef M4_DEF_MediaControl
	MediaControlStack *ctrl;
#endif

	if (!mo) return;
	if (!mo->od_man) {
		mo->speed = speed;
		return;
	}
	odm = (ODManager *) mo->od_man;

#ifdef M4_DEF_MediaControl
	/*if media control forbidd that*/
	ctrl = ODM_GetMediaControl(odm);
	if (ctrl) return;
#endif

	ODM_SetSpeed(odm, speed);
}

Float MO_GetSpeed(MediaObject *mo, Float in_speed)
{
#ifdef M4_DEF_MediaControl
	ODManager *odm;
	MediaControlStack *ctrl;
	if (!mo || !mo->od_man) return in_speed;
	odm = (ODManager *) mo->od_man;
	/*get control*/
	ctrl = ODM_GetMediaControl(odm);
	return ctrl ? ctrl->control->mediaSpeed : in_speed;
#else
	return in_speed;
#endif
}

Bool MO_GetLoop(MediaObject *mo, Bool in_loop)
{
	Clock *ck;
	ODManager *odm;
#ifdef M4_DEF_MediaControl
	MediaControlStack *ctrl;
#endif

	if (!mo || !mo->od_man) return in_loop;
	odm = (ODManager *) mo->od_man;
	
	/*get control*/
#ifdef M4_DEF_MediaControl
	ctrl = ODM_GetMediaControl(odm);
	if (ctrl) in_loop = ctrl->control->loop;
#endif
	
	/*otherwise looping is only accepted if not sharing parent scene clock*/
	ck = ODM_GetMediaClock(odm->parentscene->root_od);
	if (ODM_SharesClock(odm, ck)) return 0;
	return in_loop;
}

Double MO_GetDuration(MediaObject *mo)
{
	ODManager *odm;
	if (!mo || !mo->od_man) return -1.0;

	odm = (ODManager *) mo->od_man;
	return ((Double)odm->duration)/1000.0;
}

Bool MO_ShouldDeactivate(MediaObject *mo)
{
	ODManager *odm;
#ifdef M4_DEF_MediaControl
	MediaControlStack *ctrl;
#endif
	
	if (!mo || !mo->od_man) return 0;
	odm = (ODManager *) mo->od_man;
	/*not running*/
	if (!odm->is_open) return 0;

#ifdef M4_DEF_MediaControl
	/*get media control and see if object owning control is running*/
	ctrl = ODM_GetMediaControl(odm);
	if (!ctrl) return 1;
	/*if ctrl and ctrl not ruling this mediaObject, deny deactivation*/
	if (ctrl->stream->od_man != odm) return 0;
	/*this is currently under discussion in MPEG. for now we deny deactivation as soon as a mediaControl is here*/
	odm = (ODManager *)ctrl->stream->od_man;
	if (odm->is_open) return 0;
#endif
	/*otherwise allow*/	
	return 1;
}

Bool MO_IsMuted(MediaObject *mo)
{
#ifdef M4_DEF_MediaControl
	if (!mo->od_man) return 1;
	return ((ODManager*)mo->od_man)->media_ctrl ? ((ODManager*)mo->od_man)->media_ctrl->control->mute : 0;
#else
	return 0;
#endif
}

Bool MO_IsFinished(MediaObject *mo)
{
	ODManager *odm;
	GenericCodec *codec;
	u32 dur;
	if (!mo || !mo->od_man) return 0;
	odm = (ODManager *) mo->od_man;

	/*for natural media use composition buffer*/
	if (odm->codec && odm->codec->CB) return (odm->codec->CB->Status==CB_STOP) ? 1 : 0;

	/*otherwise check EOS and time*/
	codec = odm->codec;
	dur = odm->duration;
	if (!odm->codec) {
		if (!odm->subscene) return 0;
		codec = odm->subscene->bifs_codec;
		dur = odm->subscene->duration;
	}
	if (codec->Status==CODEC_STOP) {
		/*codec is done, check by duration*/
		Clock *ck = ODM_GetMediaClock(odm);
		if (CK_GetTime(ck) > dur) return 1;
	}
	return 0;
}

/*resyncs clock - only audio objects are allowed to use this*/
void MO_AdjustClock(MediaObject *mo, s32 ms_drift)
{
	ODManager *odm;
	if (!mo || !mo->od_man) return;
	odm = (ODManager *) mo->od_man;
	if (!odm->codec || (odm->codec->type != M4ST_AUDIO) ) return;
	CK_AdjustDrift(odm->codec->ck, ms_drift);
}

