/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.ddgenerator.sun;


import java.io.IOException;
import java.io.OutputStream;
import java.util.Iterator;

import javax.enterprise.deploy.shared.ModuleType;

import com.sun.enterprise.deployment.Application;
import com.sun.enterprise.deployment.archivist.ApplicationArchivist;
import com.sun.enterprise.deployment.archivist.Archivist;
import com.sun.enterprise.deployment.archivist.ArchivistFactory;
import com.sun.enterprise.deployment.BundleDescriptor;
import com.sun.enterprise.deployment.deploy.shared.AbstractArchive;
import com.sun.enterprise.deployment.io.DeploymentDescriptorFile;
import com.sun.enterprise.deployment.util.ModuleDescriptor;
import com.sun.enterprise.deployment.WebBundleDescriptor;

/**
 * This class is responsible for writing deployment descriptors 
 * after a deployment action has occured to a abstract archive instance.
 *
 * @author  Jerome Dochez
 */
public class DescriptorArchivist {
    
    /** Creates a new instance of DescriptorArchivist */
    public DescriptorArchivist() {
    }
  
    /**
     * writes an application deployment descriptors
     * @param the application object
     * @param the abstract archive
     */
    public void write(Application application, AbstractArchive out) 
        throws IOException 
    {
        if (application.isVirtual()) {
            ModuleDescriptor aModule = (ModuleDescriptor) application.getModules().next();
            Archivist moduleArchivist = ArchivistFactory.getArchivistForType(aModule.getModuleType());
            write(aModule.getDescriptor(), moduleArchivist,  out);
        } else {
            // this is a real application.
            
            // let's start by writing out all submodules deployment descriptors
            for (Iterator modules = application.getModules();modules.hasNext();) {
                ModuleDescriptor aModule = (ModuleDescriptor) modules.next();
                Archivist moduleArchivist = ArchivistFactory.getArchivistForType(aModule.getModuleType());
                if (aModule.getAlternateDescriptor()!=null) {
                    // the application is using alternate deployment descriptors
                    String runtimeDDPath = "sun-" + aModule.getAlternateDescriptor();
                    DeploymentDescriptorFile ddFile = moduleArchivist.getConfigurationDDFile();
                    if (ddFile!=null) {
                        OutputStream os = out.putNextEntry(runtimeDDPath);
                        ddFile.write(aModule.getDescriptor(), os);
                        out.closeEntry();
                    }
                } else {
                    AbstractArchive moduleArchive = out.getEmbeddedArchive(aModule.getArchiveUri());
                    write(aModule.getDescriptor(),  moduleArchivist, moduleArchive);
                }
            }
            
            // now let's write the application runtime descriptor
            ApplicationArchivist archivist = new ApplicationArchivist();
            archivist.setDescriptor(application);
            archivist.writeRuntimeDeploymentDescriptors(out);                        
        }
    }
    
    /**
     * writes a bundle descriptor 
     * @param the bundle descriptor
     * @param the abstract archive
     */
    public void write(BundleDescriptor bundle, AbstractArchive out) 
        throws IOException
    {
        Archivist archivist = ArchivistFactory.getArchivistForArchive(out);
        write(bundle, archivist, out);
    }
    
    /**
     * writes a bundle descriptor
     * @param the bundle descriptor
     * @param the archivist responsible for writing such bundle type
     * @param the archive to write to
     */
    protected void write(BundleDescriptor bundle, Archivist archivist, AbstractArchive out) 
        throws IOException
    {
        
        archivist.setDescriptor(bundle);
        archivist.writeRuntimeDeploymentDescriptors(out);
        
        // now if this is a web bundle descriptor, we may need to rewrite the standard 
        // deployment descriptors if we have web services since the servlet implementation
        // has been switched
        if (bundle.getModuleType().equals(ModuleType.WAR)) {
            WebBundleDescriptor webBundle = (WebBundleDescriptor) bundle;
            if (webBundle.hasWebServices()) {
                archivist.writeStandardDeploymentDescriptors(out);
            }
        }
    }
}
