/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2005, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.ejb.cmp3.transaction.base;

import java.sql.Connection;
import java.sql.SQLException;
import javax.transaction.*;
import oracle.toplink.essentials.internal.ejb.cmp3.base.ExceptionFactory;
import oracle.toplink.essentials.internal.ejb.cmp3.jdbc.base.DataSourceImpl;

/**
 * Implementation of JTA Transaction manager class.
 *
 * Currently support is limited to enlisting a single tx data source
 */
public class TransactionManagerImpl implements TransactionManager, UserTransaction {
    // Not null when a transaction is active
    TransactionImpl tx;

    /************************/
    /***** Internal API *****/
    /************************/
    private void debug(String s) {
        System.out.println(s);
    }

    /*
     * Used to create the single instance
     */
    public TransactionManagerImpl() {
        this.tx = null;
    }

    /*
     * Return true if a transaction has been explicitly begun
     */
    public boolean isTransactionActive() {
        return tx != null;
    }

    /*
     * Return a Connection if a transaction is active, otherwise return null
     */
    public Connection getConnection(DataSourceImpl ds, String user, String password) throws SQLException {
        return (tx == null) ? null : tx.getConnection(ds, user, password);
    }

    /************************************************************/
    /***** Supported TransactionManager/UserTransaction API *****/
    /************************************************************/
    public void begin() throws NotSupportedException, SystemException {
        debug("Tx - begin");

        if (isTransactionActive()) {
            throw new ExceptionFactory().txActiveException();
        }

        // New transaction created by virtue of Transaction existence
        tx = new TransactionImpl();
    }

    public void commit() throws RollbackException, HeuristicMixedException, HeuristicRollbackException, SecurityException, IllegalStateException, SystemException {
        debug("Tx - commit");

        if (!isTransactionActive()) {
            throw new ExceptionFactory().txNotActiveException();
        }
        try{
            tx.commit();
        }finally{
            tx = null;
        }
    }

    public int getStatus() throws SystemException {
        return (!isTransactionActive()) ? Status.STATUS_NO_TRANSACTION : tx.getStatus();
    }

    public Transaction getTransaction() throws SystemException {
        return tx;

    }

    public void rollback() throws IllegalStateException, SecurityException, SystemException {
        debug("Tx - rollback");

        if (!isTransactionActive()) {
            throw new ExceptionFactory().txNotActiveException();
        }
        try{
            tx.rollback();
        }finally{
            tx = null;
        }
    }

    public void setRollbackOnly() throws IllegalStateException, SystemException {
        debug("Tx - rollback");

        if (!isTransactionActive()) {
            throw new ExceptionFactory().txNotActiveException();
        }
        tx.setRollbackOnly();
    }

    /****************************************************************/
    /***** NOT supported TransactionManager/UserTransaction API *****/
    /****************************************************************/
    public Transaction suspend() throws SystemException {
        return null;
    }

    public void resume(Transaction transaction) throws InvalidTransactionException, IllegalStateException, SystemException {
        // Do nothing
    }

    public void setTransactionTimeout(int i) throws SystemException {
        // Do nothing
    }
}
