/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.ejb.cmp3.metadata.accessors;

import java.util.List;
import java.util.ArrayList;

import javax.persistence.FetchType;
import javax.persistence.CascadeType;
import javax.persistence.JoinColumn;
import javax.persistence.JoinColumns;
import javax.persistence.PrimaryKeyJoinColumn;
import javax.persistence.PrimaryKeyJoinColumns;

import oracle.toplink.essentials.internal.helper.DatabaseField;

import oracle.toplink.essentials.mappings.DatabaseMapping;
import oracle.toplink.essentials.mappings.ForeignReferenceMapping;

import oracle.toplink.essentials.indirection.ValueHolderInterface;

import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataConstants;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataProcessor;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataDescriptor;

import oracle.toplink.essentials.internal.ejb.cmp3.metadata.columns.MetadataJoinColumn;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.columns.MetadataJoinColumns;

import oracle.toplink.essentials.internal.ejb.cmp3.metadata.accessors.ClassAccessor;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.accessors.objects.MetadataAccessibleObject;

/**
 * An relational accessor.
 * 
 * @author Guy Pelletier
 * @since TopLink EJB 3.0 Reference Implementation
 */
public abstract class RelationshipAccessor extends MetadataAccessor {
    protected String m_mappedBy;
    protected Class m_targetEntity;
    protected ArrayList<String> m_cascadeTypes;
    
    /**
     * INTERNAL:
     */
    protected RelationshipAccessor(MetadataAccessibleObject accessibleObject, ClassAccessor classAccessor) {
        super(accessibleObject, classAccessor);
        
        // These are the defaults. Any subclass should override these settings
        // where necessary.
        m_mappedBy = "";
        setIsOptional(false);
        m_targetEntity = void.class;
        setFetchType(MetadataConstants.LAZY);
        m_cascadeTypes = new ArrayList<String>();
    }
    
    /**
     * INTERNAL:
     */
    public void addCascadeType(String cascadeType) {
        m_cascadeTypes.add(cascadeType);
    }
    
    /**
     * INTERNAL:
     * Return the cascade types for this accessor. This method is supported by 
     * all relational accessors.
     */
    public ArrayList<String> getCascadeTypes() {
        return m_cascadeTypes;
    }
    
    /**
     * INTERNAL: (Overridden in XMLOneToOneAccessor, XMLManyToManyAccessor and XMLOneToManyAccessor)
     * Process the @JoinColumns and @JoinColumn.
     */    
    protected MetadataJoinColumns getJoinColumns() {
        JoinColumn joinColumn = getAnnotation(JoinColumn.class);
        JoinColumns joinColumns = getAnnotation(JoinColumns.class);
        
        return new MetadataJoinColumns(joinColumns, joinColumn);
    }
    
    /**
     * INTERNAL:
     * Return the mapped by attribute for this accessor. This method is
     * supported by all relational accessors except for many-to-one.
     */
    public String getMappedBy() {
        return m_mappedBy;
    }
    
    /**
     * INTERNAL:
     * Method to return an owner mapping. It will tell the owner class to
     * process itself if it hasn't already done so.
     */
    protected DatabaseMapping getOwningMapping() {
        String ownerAttributeName = m_mappedBy;
        MetadataDescriptor ownerDescriptor = getReferenceDescriptor();
        DatabaseMapping mapping = ownerDescriptor.getMappingForAttributeName(ownerAttributeName);
        
        // If no mapping was found, there is an error in the mappedBy field, 
        // therefore, throw an exception.
        if (mapping == null) {
            m_validator.throwNoMappedByAttributeFound(ownerDescriptor.getJavaClass(), ownerAttributeName, getJavaClass(), getAttributeName());
        }
        
        return mapping;
    }
    
    /**
     * INTERNAL:
     * Return the target entity for this accessor. This method is supported by 
     * all relational accessors.
     */
    public Class getTargetEntity() {
        return m_targetEntity;
    }
    
    /**
     * INTERNAL:
	 * Method to check if an annotated element has a @JoinColumn.
     */
	public boolean hasJoinColumn() {
		return isAnnotationPresent(JoinColumn.class);
    }
    
    /**
     * INTERNAL:
	 * Method to check if an annotated element has a @JoinColumns.
     */
	public boolean hasJoinColumns() {
		return isAnnotationPresent(JoinColumns.class);
    }
    
    /**
     * INTERNAL: (Overridden in XMLOneToOneAccessor)
	 * Method to check if an annotated element has a @PrimaryKeyJoinColumns or
     * at the very least a @PrimaryKeyJoinColumn.
     */
	public boolean hasPrimaryKeyJoinColumns() {
		return isAnnotationPresent(PrimaryKeyJoinColumns.class) || isAnnotationPresent(PrimaryKeyJoinColumn.class);
    }
    
    /**
     * INTERNAL:
     * Return true if this accessor represents a 1-1 primary key relationship.
     */
	public boolean isOneToOnePrimaryKeyRelationship() {
        return isOneToOne() && hasPrimaryKeyJoinColumns();
    }
    
    /**
     * INTERNAL:
     */
    protected void processCascadeTypes(ForeignReferenceMapping mapping) {
        for (String cascadeType : getCascadeTypes()) {
			setCascadeType(cascadeType, mapping);
		}
        
        // Apply the persistence unit default cascade-persist if necessary.
        if (m_descriptor.isCascadePersist() && ! mapping.isCascadePersist()) {
        	setCascadeType(CascadeType.PERSIST.name(), mapping);
        }
    }
    
    /**
     * INTERNAL:
     * Process a @JoinColumns or @JoinColumn. Will look for association
     * overrides.
     */	
    protected List<MetadataJoinColumn> processJoinColumns() { 
        if (m_descriptor.hasAssociationOverrideFor(getAttributeName())) {
            return processJoinColumns(m_descriptor.getAssociationOverrideFor(getAttributeName()), getReferenceDescriptor());
        } else {
            return processJoinColumns(getJoinColumns(), getReferenceDescriptor());
        }
    }
    
    /**
     * INTERNAL:
     * Process MetadataJoinColumns.
     */	
    protected List<MetadataJoinColumn> processJoinColumns(MetadataJoinColumns joinColumns, MetadataDescriptor descriptor) { 
        if (descriptor.hasCompositePrimaryKey()) {
            // The number of join columns should equal the number of primary key fields.
            if (joinColumns.size() != descriptor.getPrimaryKeyFields().size()) {
                m_validator.throwIncompleteJoinColumnsSpecified(getJavaClass(), getAnnotatedElement());
            }
            
            // All the primary and foreign key field names should be specified.
            for (MetadataJoinColumn joinColumn : joinColumns.values()) {
                if (joinColumn.isPrimaryKeyFieldNotSpecified() || joinColumn.isForeignKeyFieldNotSpecified()) {
                    m_validator.throwIncompleteJoinColumnsSpecified(getJavaClass(), getAnnotatedElement());
                }
            }
        } else {
            if (joinColumns.size() > 1) {
                m_validator.throwExcessiveJoinColumnsSpecified(getJavaClass(), getAnnotatedElement());
            }
        }
        
        return joinColumns.values();
    }
    
    /**
     * INTERNAL:
     * Front end validation before actually processing the relationship 
     * accessor. The process() method should not be called directly.
     */
    public void processRelationship() {
        // The processing of this accessor may have been fast tracked through a 
        // non-owning relationship. If so, no processing is required.
        if (! isProcessed()) {
            if (m_descriptor.hasMappingForAttributeName(getAttributeName())) {
                // Only true if there is one that came from Project.xml
                m_logger.logWarningMessage(m_logger.IGNORE_MAPPING, this);
            } else {
                // If a @Column is specified then throw an exception.
                if (hasColumn()) {
                    m_validator.throwRelationshipHasColumnSpecified(getJavaClass(), getAttributeName());
                }
                
                // Process the relationship accessor only if the target entity
                // is not a ValueHolderInterface.
                if (getTargetEntity() == ValueHolderInterface.class || (getTargetEntity() == void.class && getReferenceClass() == ValueHolderInterface.class)) {
                    // do nothing ... I'm too lazy (or too stupid) to do the negation of this expression :-)
                } else { 
                    process();
                }
            }
            
            // Set its processing completed flag to avoid double processing.
            setIsProcessed();
        }
    }

    /**
     * INTERNAL:
     * Set the cascade type on a mapping.
     */
    protected void setCascadeType(String type, ForeignReferenceMapping mapping) {
        if (type.equals(MetadataConstants.CASCADE_ALL) || type.equals(CascadeType.ALL.name())) {
            mapping.setCascadeAll(true);
        } else if (type.equals(MetadataConstants.CASCADE_MERGE) || type.equals(CascadeType.MERGE.name())) {
            mapping.setCascadeMerge(true);
        } else if (type.equals(MetadataConstants.CASCADE_PERSIST) || type.equals(CascadeType.PERSIST.name())) {
            mapping.setCascadePersist(true);
        }  else if (type.equals(MetadataConstants.CASCADE_REFRESH) || type.equals(CascadeType.REFRESH.name())) {
            mapping.setCascadeRefresh(true);
        } else if (type.equals(MetadataConstants.CASCADE_REMOVE) || type.equals(CascadeType.REMOVE.name())) {
            mapping.setCascadeRemove(true);
        }
    }
    
    /**
     * INTERNAL:
     */
    public void setCascadeTypes(CascadeType[] cascadeTypes) {
        for (CascadeType cascadeType : cascadeTypes) {
			addCascadeType(cascadeType.name());
		}
    }

    /**
     * INTERNAL:
     */
    public void setMappedBy(String mappedBy) {
        m_mappedBy = mappedBy;
    }
    
    /**
     * INTERNAL:
	 * Set the reference class for this accessor.
     */
    public void setReferenceClass(Class potentialReferenceClass, String context) {
        if (potentialReferenceClass == void.class) {
            // Log the defaulting column name based on the given context.
            getLogger().logConfigMessage(context, getAnnotatedElement(), getReferenceClass());
        } else {
            m_accessibleObject.setReferenceClass(potentialReferenceClass);
        }
	}
    
    /**
     * INTERNAL:
     */
    public void setTargetEntity(Class targetEntity) {
        m_targetEntity = targetEntity;
    }
}
