

/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the "License").  You may not use this file except
 * in compliance with the License.
 *
 * You can obtain a copy of the license at
 * glassfish/bootstrap/legal/CDDLv1.0.txt or
 * https://glassfish.dev.java.net/public/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * HEADER in each file and include the License file at
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable,
 * add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your
 * own identifying information: Portions Copyright [yyyy]
 * [name of copyright owner]
 *
 * Copyright 2005 Sun Microsystems, Inc. All rights reserved.
 *
 * Portions Copyright Apache Software Foundation.
 */

package org.apache.catalina.ssi;

import java.io.IOException;
import java.io.PrintWriter;
import java.text.DecimalFormat;

/**
 * Implements the Server-side #fsize command
 *
 * @author Bip Thelin
 * @author Dan Sandberg
 * @version $Revision: 1.2 $, $Date: 2005/12/08 01:28:03 $
 */
public final class SSIFsize implements SSICommand {
    protected final static int ONE_KILOBYTE = 1024;
    protected final static int ONE_MEGABYTE = 1024 * 1024;

    /**
     * @see SSICommand
     */
    public void process(SSIMediator ssiMediator,
			  String[] paramNames,
			  String[] paramValues,
			  PrintWriter writer) {

	String configErrMsg = ssiMediator.getConfigErrMsg();
        for(int i=0;i<paramNames.length;i++) {
	    String paramName = paramNames[i];
	    String paramValue = paramValues[i];

	    try {
		if ( paramName.equalsIgnoreCase("file") ||
		     paramName.equalsIgnoreCase("virtual") ) {
		    boolean virtual = paramName.equalsIgnoreCase("virtual");
		    long size = ssiMediator.getFileSize( paramValue,  virtual );
		    String configSizeFmt = ssiMediator.getConfigSizeFmt();
		    writer.write( formatSize(size, configSizeFmt ) );
		} else {
		    ssiMediator.log("#fsize--Invalid attribute: " + paramName );
		    writer.write( configErrMsg );
		}
	    } catch ( IOException e ) {
		ssiMediator.log("#fsize--Couldn't get size for file: " + paramValue, e );
		writer.write( configErrMsg );
	    }
	}
    }

    public String repeat( char aChar, int numChars ) {
	if ( numChars < 0 ) {
	    throw new IllegalArgumentException("Num chars can't be negative");
	}
	StringBuffer buf = new StringBuffer();
	for ( int i=0; i < numChars; i++ ) {
	    buf.append( aChar );
	}
	return buf.toString();
    }

    public String padLeft( String str, int maxChars ) {
	String result = str;
	int charsToAdd = maxChars - str.length();
	if ( charsToAdd > 0 ) {
	    result = repeat( ' ', charsToAdd ) + str;
	}
	return result;
    }




    //We try to mimick Apache here, as we do everywhere
    //All the 'magic' numbers are from the util_script.c Apache source file.
    protected String formatSize(long size, String format) {
        String retString = "";

        if ( format.equalsIgnoreCase("bytes") ) {
	    DecimalFormat decimalFormat = new DecimalFormat("#,##0");
	    retString = decimalFormat.format( size );
        } else {
	    if ( size == 0 ) {
		retString = "0k";
            } else if ( size < ONE_KILOBYTE ) {
		retString = "1k";
	    } else if ( size < ONE_MEGABYTE ) {
		retString = Long.toString( (size + 512) / ONE_KILOBYTE );
		retString += "k";
            } else if ( size  < 99 * ONE_MEGABYTE ) {
		DecimalFormat decimalFormat = new DecimalFormat("0.0M");
		retString = decimalFormat.format( size  / (double) ONE_MEGABYTE );
	    } else {
		retString = Long.toString( (size + ( 529 * ONE_KILOBYTE) ) / ONE_MEGABYTE );
		retString += "M";
	    }
	    retString = padLeft( retString, 5 );
        }

        return retString;
    }
}

