/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.security.auth.realm;

import com.sun.enterprise.admin.event.AdminEventListenerException;
import com.sun.enterprise.admin.event.UserMgmtEvent;
import com.sun.enterprise.admin.event.UserMgmtEventListener;
import com.sun.enterprise.security.SecurityUtil;
import com.sun.enterprise.security.auth.realm.file.FileRealm;
import com.sun.enterprise.server.pluggable.SecuritySupport;


/**
 * Implements interface UserMgmtEventListener.
 * So that users can be dynamically created/updated/deleted.
 * @author Shing Wai Chan
 */
public class UserMgmtEventListenerImpl implements UserMgmtEventListener {

    /**
     * user added.
     * It is called whenever a UserMgmtEvent with action of
     * UserMgmtEvent.ACTION_USERADD is received.
     * @throws AdminEventListenerException when the listener is unable to
     *         process the event.
     */
    public void userAdded(UserMgmtEvent event)
             throws AdminEventListenerException {
        reloadRealm(event);
    }

    /**
     * user deleted.
     * It is called whenever a UserMgmtEvent with action of
     * UserMgmtEvent.ACTION_USERREMOVE is received.
     * @throws AdminEventListenerException when the listener is unable to
     *         process the event.
     */
    public void userRemoved(UserMgmtEvent event)
             throws AdminEventListenerException {
        reloadRealm(event);
    }

    /**
     * user updated (attributes change).
     * It is called whenever a UserMgmtEvent with action of
     * UserMgmtEvent.ACTION_USERUPDATE is received.
     * @throws AdminEventListenerException when the listener is unable to
     *         process the event.
     */
    public void userUpdated(UserMgmtEvent event)
             throws AdminEventListenerException {
        reloadRealm(event);
    }

    /**
     * In this moment, user management is supported only in FileRealm.
     * The FileRealm will be refreshed in this case.
     * Other realms will throw Exception.
     * Administrative API may be factored out during JSR 196 implementation.
     * @param event the UserMgmtEvent
     * @exception AdminEventListenerException
     */
    private void reloadRealm(UserMgmtEvent event)
            throws AdminEventListenerException {
        try {
            String realmName = event.getAuthRealmName();
            Realm realm = Realm.getInstance(realmName);

            // should always true in this moment
            if (realm instanceof FileRealm) {
                SecuritySupport secSupp = SecurityUtil.getSecuritySupport();
                secSupp.synchronizeKeyFile(event.getConfigContext(), realmName);
            }

            realm.refresh();
        } catch(Exception ex) {
            throw new AdminEventListenerException(ex);
        }
    }
}
