/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.security.auth.realm;

import java.util.Properties;

import com.sun.enterprise.config.ConfigContext;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.config.serverbeans.AuthRealm;
import com.sun.enterprise.config.serverbeans.ElementProperty;
import com.sun.enterprise.config.serverbeans.SecurityService;
import com.sun.enterprise.config.serverbeans.ServerBeansFactory;
import com.sun.enterprise.admin.event.AdminEventListenerException;
import com.sun.enterprise.admin.event.AuthRealmEvent;
import com.sun.enterprise.admin.event.AuthRealmEventListener;
import com.sun.enterprise.security.SecurityUtil;
import com.sun.enterprise.server.pluggable.SecuritySupport;

/**
 * Implements interface AuthRealmEventListener.
 * So that realms can be dynamically created/updated/deleted.
 * @author Shing Wai Chan
 */
public class AuthRealmEventListenerImpl implements AuthRealmEventListener {

    /**
     * New auth realm created.
     * It is called whenever a AuthRealmEvent with action of
     * AuthRealmEvent.ACTION_CREATE is received.
     * @throws AdminEventListenerException when the listener is unable to
     *         process the event.
     */
    public void authRealmCreated(AuthRealmEvent event)
             throws AdminEventListenerException {
        try {
            createRealm(event);
        } catch(Exception ex) {
            throw new AdminEventListenerException(ex);
        }
    } 

    /**
     * Auth realm deleted.
     * It is called whenever a AuthRealmEvent with action of
     * AuthRealmEvent.ACTION_DELETE is received.
     * @throws AdminEventListenerException when the listener is unable to
     *         process the event.
     */
    public void authRealmDeleted(AuthRealmEvent event)
             throws AdminEventListenerException {
        try {
            //only unload the realm, keep any auxiliary file for sanity
            Realm.unloadInstance(event.getAuthRealmName());
        } catch(Exception ex) {
            throw new AdminEventListenerException(ex);
        }
    }

    /**
     * Auth realm updated (attributes change).
     * It is called whenever a AuthRealmEvent with action of
     * AuthRealmEvent.ACTION_UPDATE is received.
     * @throws AdminEventListenerException when the listener is unable to
     *         process the event.
     */
    public void authRealmUpdated(AuthRealmEvent event)
             throws AdminEventListenerException {
        try {
            //XXX replace with a new realm, need to revisit in JSR 196
            createRealm(event);
        } catch(Exception ex) {
            throw new AdminEventListenerException(ex);
        }
    }

    /**
     * This method will create or replace existing realm with a new one
     * in cache.
     * @param event
     * @exception for instance, BadRealmException, ConfigException,
     *            SynchronizationException
     */
    private void createRealm(AuthRealmEvent event) throws Exception {
        ConfigContext configContext = event.getConfigContext();
        String realmName = event.getAuthRealmName();
        SecurityService security = 
            ServerBeansFactory.getSecurityServiceBean(configContext);
        AuthRealm authRealm = security.getAuthRealmByName(realmName);
        //authRealm cannot be null here
        String className = authRealm.getClassname();
        ElementProperty[] elementProps = authRealm.getElementProperty();
        int size = (elementProps != null) ? elementProps.length : 0;
        Properties props = new Properties();
        for (int i = 0; i < size; i++) {
            props.setProperty(elementProps[i].getName(),
                    elementProps[i].getValue());
        }

        if ("com.sun.enterprise.security.auth.realm.file.FileRealm".equals(className)) {
            SecuritySupport secSupp = SecurityUtil.getSecuritySupport();
            secSupp.synchronizeKeyFile(configContext, realmName);
        }
        Realm.instantiate(realmName, className, props);
    }
}
