/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.security.auth.login;

import java.util.*;
import java.util.logging.Logger;
import java.util.logging.Level;
import com.sun.logging.LogDomains;

import javax.security.auth.*;
import javax.security.auth.callback.*;
import javax.security.auth.login.*;
import javax.security.auth.spi.*;

import javax.naming.*;
import javax.naming.directory.*;

import com.sun.enterprise.security.auth.realm.ldap.LDAPRealm;
// the following required for dynamic group support
import java.security.acl.Group;
import java.security.Principal;
import javax.security.auth.x500.X500Principal;
// imported from the ldap booster pack
import com.sun.jndi.ldap.obj.GroupOfURLs;
import javax.security.auth.login.LoginException;

/**
 * iAS JAAS LoginModule for an LDAP Realm.
 *
 * <P>Refer to the LDAPRealm documentation for necessary and optional
 * configuration parameters for the iAS LDAP login support.
 *
 * <P>There are various ways in which a user can be authenticated using
 * an LDAP directory. Currently this login module only supports one mode,
 * 'find and bind'. Other modes may be added as schedules permit.
 *
 * <P>Mode: <i>find-bind</i>
 * <ol>
 *  <li>An LDAP search is issued on the directory starting at base-dn
 *      with the given search-filter (having substituted the user name
 *      in place of %s). If no entries match this search, login fails
 *      and authentication is over.
 *  <li>The DN of the entry which matched the search as the DN
 *      of the user in the directory. If the search-filter
 *      is properly set there should always be a single match; if there are
 *      multiple matches, the first one found is used.
 *  <li>Next an LDAP bind is attempted using the above DN and the provided
 *      password. If this fails, login is considered to have failed and
 *      authentication is over.
 *  <li>Then an LDAP search is issued on the directory starting at
 *      group-base-dn with the given group-search-filter (having
 *      substituted %d for the user DN previously found). From the
 *      matched entry(ies) all the values of group-target are taken
 *      as group names in which the user has membership. If no entries
 *      are found, the group membership is empty.
 * </ol>
 *
 *
 */
public class LDAPLoginModule extends PasswordLoginModule
{
    private String _userDNbase;
    private String _searchFilter;
    private String _grpDNbase;
    private String _grpSearchFilter;
    private String _grpTarget;
    private LDAPRealm _ldapRealm;
    private String[] _dnOnly = {"dn"};

    /**
     * Performs authentication for the current user.
     *
     */
    protected void authenticate ()
        throws LoginException
    {
        if (!(_currentRealm instanceof LDAPRealm)) {
            String msg = sm.getString("ldaplm.badrealm");
            throw new LoginException(msg);
        }
        _ldapRealm = (LDAPRealm)_currentRealm;
        
                                // enforce that password cannot be empty.
                                // ldap may grant login on empty password!
        if (_password == null || _password.length() == 0) {
            String msg = sm.getString("ldaplm.emptypassword", _username);
            throw new LoginException(msg);
        }
        
                                // load configuration attributes   
        _userDNbase = _currentRealm.getProperty(LDAPRealm.PARAM_USERDN);
        _searchFilter =
            _currentRealm.getProperty(LDAPRealm.PARAM_SEARCH_FILTER);
        _grpDNbase = _currentRealm.getProperty(LDAPRealm.PARAM_GRPDN);
        _grpSearchFilter =
            _currentRealm.getProperty(LDAPRealm.PARAM_GRP_SEARCH_FILTER);
        _grpTarget = 
            _currentRealm.getProperty(LDAPRealm.PARAM_GRP_TARGET);
        
        String mode = _currentRealm.getProperty(LDAPRealm.PARAM_MODE);

        if (LDAPRealm.MODE_FIND_BIND.equals(mode)) {
            findAndBind();

        } else {
            String msg = sm.getString("ldaplm.badmode", mode);
            throw new LoginException(msg);
        }
    }


    /**
     * Supports mode=find-bind. See class documentation.
     *
     */
    private void findAndBind()
        throws LoginException
    {
        // do search for user, substituting %s for username
        StringBuffer sb = new StringBuffer(_searchFilter);
        substitute(sb, LDAPRealm.SUBST_SUBJECT_NAME, _username);
        String userid = sb.toString();

        // attempt to bind as the user
        DirContext ctx = null;
        String srcFilter = null;
        String[] grpList = null;
        try {
            ctx = new InitialDirContext(_ldapRealm.getLdapBindProps());
            String realUserDN = userSearch(ctx,_userDNbase, userid);
            if (realUserDN == null) {
                String msg = sm.getString("ldaplm.usernotfound", _username);
                throw new LoginException(msg);
            }

            boolean bindSuccessful = bindAsUser(realUserDN, _password);
            if (bindSuccessful == false) {
                String msg = sm.getString("ldaplm.bindfailed", realUserDN);
                throw new LoginException(msg);
            }

            // search groups using above connection, substituting %d (and %s)
            sb = new StringBuffer(_grpSearchFilter);
            substitute(sb, LDAPRealm.SUBST_SUBJECT_NAME, _username);
            substitute(sb, LDAPRealm.SUBST_SUBJECT_DN, realUserDN);

            srcFilter = sb.toString();
            ArrayList groupsList = new ArrayList();
            groupsList.addAll(groupSearch(ctx, _grpDNbase, srcFilter, _grpTarget));
            // search filter is constructed internally as
            // as a groupofURLS
            groupsList.addAll(dynamicGroupSearch(ctx, _grpDNbase, _grpTarget,
                realUserDN));          
            grpList = new String[groupsList.size()];
            groupsList.toArray(grpList);
        } catch (Exception e) {
            throw new LoginException(e.toString() );
        } finally {
            if (ctx != null) {
                try {
                    ctx.close();
                } catch (NamingException e) {};
            }
        }

        if (_logger.isLoggable(Level.FINE)) {
            _logger.log(Level.FINE, "LDAP:Group search filter: " + srcFilter);
            StringBuffer gb = new StringBuffer();
            gb.append("Group memberships found: ");
            if (grpList != null) {
                for (int i=0; i<grpList.length; i++) {
                    gb.append(" "+grpList[i]);
                }
            } else {
                gb.append("(null)");
            }
            _logger.log(Level.FINE, "LDAP: "+ gb.toString());
        }
        _ldapRealm.setGroupNames(_username, grpList);

        if(_logger.isLoggable(Level.FINE)){
             _logger.log(Level.FINE, "LDAP: login succeeded for: " + _username);
        }

        // note that we don't need to copy the grpList here as groupSearch
        // return a new String array for each call
        commitAuthentication(_username, _password,
                             _currentRealm, grpList);
    }


    /**
     * Do anonymous search for the user. Should be unique if exists.
     *
     */
    private String userSearch(DirContext ctx, String baseDN, String filter)
    {
        if (_logger.isLoggable(Level.FINEST)) {
            _logger.log(Level.FINE, "search: baseDN: "+ baseDN +
                           "  filter: " + filter);
        }
            
        String foundDN = null;
        NamingEnumeration namingEnum = null;

        SearchControls ctls = new SearchControls();
        ctls.setReturningAttributes(_dnOnly);
        ctls.setSearchScope(SearchControls.SUBTREE_SCOPE);
        ctls.setCountLimit(1);

        try {
            namingEnum = ctx.search(baseDN, filter, ctls);
            if (namingEnum.hasMore()) {
                SearchResult res = (SearchResult)namingEnum.next();

                StringBuffer sb = new StringBuffer();
                //for dn name with '/'
                CompositeName compDN = new CompositeName(res.getName());
                String ldapDN = compDN.get(0);
                sb.append(ldapDN);
                
                if (res.isRelative()) {
                    sb.append(",");
                    sb.append(baseDN);
                }
                foundDN = sb.toString();
                if (_logger.isLoggable(Level.FINEST)) {
                    _logger.log(Level.FINE, "Found user DN: " + foundDN);
                }
            }
        } catch (Exception e) {
            _logger.log(Level.WARNING, "ldaplm.searcherror", filter);
            _logger.log(Level.WARNING, "security.exception", e);
        } finally {
            if (namingEnum != null) {
                try {
                    namingEnum.close();
                } catch(Exception ex) {
                }
            }
        }

        return foundDN;
    }


    /**
     * Attempt to bind as a specific DN.
     *
     */
    private boolean bindAsUser(String bindDN, String password)
    {
        boolean bindSuccessful=false;

        Properties p = _ldapRealm.getLdapBindProps();
        
        p.put(Context.SECURITY_PRINCIPAL, bindDN);
        p.put(Context.SECURITY_CREDENTIALS, password);
        
        DirContext ctx = null;
        try {
            ctx = new InitialDirContext(p);
            bindSuccessful = true;
        } catch (Exception e) {
            if (_logger.isLoggable(Level.FINEST)) {
                _logger.finest("Error binding to directory as: " + bindDN);
                _logger.finest("Exception from JNDI: " + e.toString());
            }
        } finally {
            if (ctx != null) {
                try {
                    ctx.close();
                } catch (NamingException e) {};
            }
        }
        return bindSuccessful;
    }

    /**
     * Search for group membership using the given connection.
     *
     */
    private List dynamicGroupSearch(DirContext ctx, String baseDN, 
            String target, String userDN)
    {        
        List groupList = new ArrayList();
        String filter = LDAPRealm.DYNAMIC_GROUP_FILTER;
        
        String[] targets = new String[] { target, "memberUrl" };

        try {
            SearchControls ctls = new SearchControls();
            ctls.setReturningAttributes(targets);
            ctls.setSearchScope(SearchControls.SUBTREE_SCOPE);
            ctls.setReturningObjFlag(true);
            
            NamingEnumeration e = ctx.search(baseDN, filter, ctls);
            
            while(e.hasMore()) {
                SearchResult res = (SearchResult)e.next();
                Object searchedObject = res.getObject();
                if(searchedObject instanceof com.sun.jndi.ldap.obj.GroupOfURLs){ // dynamic group
                    com.sun.jndi.ldap.obj.GroupOfURLs gurls = (com.sun.jndi.ldap.obj.GroupOfURLs) searchedObject;
                    Principal x500principal = new X500Principal(userDN);
                    if (gurls.isMember(x500principal)) {
                        
                        Attribute grpAttr = res.getAttributes().get(target);
                        int sz = grpAttr.size();
                        for (int i=0; i<sz; i++) {
                            String s = (String)grpAttr.get(i);
                            groupList.add(s);
                        }
                    }
                }
                // recommended by Jaya Hangal from JDK team
                if (searchedObject instanceof Context) {
                    ((Context)searchedObject).close();
                }
            }
        } catch (Exception e) {
            _logger.log(Level.WARNING, "ldaplm.searcherror", filter);
            _logger.log(Level.WARNING, "security.exception", e);
        }
        return groupList;
    }
    
    /**
     * Search for group membership using the given connection.
     *
     */
    private List groupSearch(DirContext ctx, String baseDN,
                                 String filter, String target)
    {        
        List groupList = new ArrayList();
        
        try {
            String[] targets = new String[1];
            targets[0] = target;
            
            SearchControls ctls = new SearchControls();
            ctls.setReturningAttributes(targets);
            ctls.setSearchScope(SearchControls.SUBTREE_SCOPE);
            
            NamingEnumeration e = ctx.search(baseDN, filter, ctls);
            
            while(e.hasMore()) {
                SearchResult res = (SearchResult)e.next();
                Attribute grpAttr = res.getAttributes().get(target);
                int sz = grpAttr.size();
                for (int i=0; i<sz; i++) {
                    String s = (String)grpAttr.get(i);
                    groupList.add(s);
                }
            }
                
        } catch (Exception e) {
            _logger.log(Level.WARNING, "ldaplm.searcherror", filter);
            _logger.log(Level.WARNING, "security.exception", e);
        }

        return groupList;
    }

    /**
     * Do string substitution. target is replaced by value for all
     * occurences.
     *
     */
    private static void substitute(StringBuffer sb,
                                   String target, String value)
    {
        int i = sb.indexOf(target);
        while (i >= 0) {
            sb.replace(i, i+target.length(), value);
            i = sb.indexOf(target);
        }
    }

}
