/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.security;

import java.util.HashMap;

/**
 * This class represents the information associated to ciphers.
 * It also maintains a HashMap from configName to CipherInfo.
 * @author Shing Wai Chan
 */
public class CipherInfo {
    private static final short SSL2 = 0x1;
    private static final short SSL3 = 0x2;
    private static final short TLS = 0x4;

    // The old names mapped to the standard names as existed in the http path
    private static final String[][] OLD_HTTP_CIPHER_MAPPING = {
            {"rsa_rc4_128_md5", "SSL_RSA_WITH_RC4_128_MD5"},
            {"rsa_3des_sha"   , "SSL_RSA_WITH_3DES_EDE_CBC_SHA"},
            {"rsa_des_sha"    , "SSL_RSA_WITH_DES_CBC_SHA"},
            {"rsa_rc4_40_md5" , "SSL_RSA_EXPORT_WITH_RC4_40_MD5"},
            {"rsa_null_md5"   , "SSL_RSA_WITH_NULL_MD5"},
            {"rsa_des_56_sha" , "SSL_RSA_WITH_DES_CBC_SHA"},
            {"rsa_rc4_56_sha" , "SSL_RSA_WITH_RC4_128_SHA"}
        };

    // The old names mapped to the standard names as existed in this class before
    // The ones that have been commented out are covered in OLD_HTTP_CIPHER_MAPPING
    private static final String[][] OLD_CIPHER_MAPPING = {
            //{"rsa_rc4_128_md5", "SSL_RSA_WITH_RC4_128_MD5"},
            //{"rsa_rc4_40_md5" , "SSL_RSA_EXPORT_WITH_RC4_40_MD5"},
            //{"rsa_null_md5"   , "SSL_RSA_WITH_NULL_MD5"},
            //{"rsa_3des_sha"   , "SSL_RSA_WITH_3DES_EDE_CBC_SHA"},
            {"rsa_rc4_128_sha", "SSL_RSA_WITH_RC4_128_SHA"},
            {"fips_des_sha"   , "SSL_RSA_WITH_DES_CBC_SHA"},
            {"rsa_null_sha"   , "SSL_RSA_WITH_NULL_SHA"}
        };

    // The current set of cipher names that can be configured by the
    // user. Eventually we'd like to expand this list to something
    // like1. SSL_RSA_WITH_RC4_128_MD5 SSL_RSA_WITH_RC4_128_SHA
    // TLS_RSA_WITH_AES_128_CBC_SHA TLS_DHE_RSA_WITH_AES_128_CBC_SHA
    // SSL_RSA_WITH_3DES_EDE_CBC_SHA SSL_DHE_RSA_WITH_3DES_EDE_CBC_SHA
    // SSL_RSA_WITH_DES_CBC_SHA SSL_DHE_RSA_WITH_DES_CBC_SHA
    // SSL_RSA_EXPORT_WITH_RC4_40_MD5 SSL_RSA_EXPORT_WITH_DES40_CBC_SHA
    // SSL_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA TLS_DHE_DSS_WITH_AES_128_CBC_SHA
    // SSL_DHE_DSS_WITH_3DES_EDE_CBC_SHA SSL_DHE_DSS_WITH_DES_CBC_SHA
    // SSL_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA
    private static final String[] USER_CONFIGURABLE_CIPHERS = {
        "SSL_RSA_WITH_RC4_128_MD5",
        "SSL_RSA_WITH_3DES_EDE_CBC_SHA",
        "SSL_RSA_WITH_DES_CBC_SHA",
        "SSL_RSA_EXPORT_WITH_RC4_40_MD5",
        "SSL_RSA_WITH_NULL_MD5",
        "SSL_RSA_WITH_RC4_128_SHA",
        "SSL_RSA_WITH_NULL_SHA" };

    private static HashMap ciphers = new HashMap();

    private String configName;
    private String cipherName;
    private short protocolVersion;


    static {
        //The following ciphers name are from JSSE in JDK 1.4.
//Commented out ciphers will be uncommented once their configNames are finalized.

        int len = USER_CONFIGURABLE_CIPHERS.length;
        for(int i=0; i<len; i++) {
            String s = USER_CONFIGURABLE_CIPHERS[i];
            ciphers.put(s, new CipherInfo(s, s, (short)(SSL3|TLS)) );
        }

        len = OLD_HTTP_CIPHER_MAPPING.length;
        for(int i=0; i<len; i++) {
            String nonStdName = OLD_HTTP_CIPHER_MAPPING[i][0];
            String stdName    = OLD_HTTP_CIPHER_MAPPING[i][1];
            ciphers.put(nonStdName, 
                new CipherInfo(nonStdName, stdName, (short)(SSL3|TLS)) );
        }

        len = OLD_CIPHER_MAPPING.length;
        for(int i=0; i<len; i++) {
            String nonStdName = OLD_CIPHER_MAPPING[i][0];
            String stdName    = OLD_CIPHER_MAPPING[i][1];
            ciphers.put(nonStdName, 
                new CipherInfo(nonStdName, stdName, (short)(SSL3|TLS)) );
        }
    }

    /**
     * @param configName  name used in domain.xml, sun-acc.xml
     * @param cipherName  name that may depends on backend
     * @param protocolVersion
     */
    private CipherInfo(String configName, String cipherName, 
            short protocolVersion) {
        this.configName = configName;
        this.cipherName = cipherName;
        this.protocolVersion = protocolVersion;
    }

    public static CipherInfo getCipherInfo(String configName) {
        return (CipherInfo)ciphers.get(configName);
    }

    public String getConfigName() {
        return configName;
    }

    public String getCipherName() {
        return cipherName;
    }

    public boolean isSSL2() {
        return (protocolVersion & SSL2) == SSL2;
    } 

    public boolean isSSL3() {
        return (protocolVersion & SSL3) == SSL3;
    } 

    public boolean isTLS() {
        return (protocolVersion & TLS) == TLS;
    } 
}
