/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.resource.monitor;

import com.sun.enterprise.admin.monitor.registry.MonitoringLevel;
import com.sun.enterprise.admin.monitor.registry.MonitoredObjectType;
import com.sun.enterprise.admin.monitor.registry.MonitoringLevelListener;
import com.sun.enterprise.admin.monitor.registry.MonitoringRegistry;
import com.sun.enterprise.admin.monitor.registry.MonitoringRegistrationException;
import com.sun.enterprise.admin.monitor.stats.JDBCConnectionPoolStats;
import com.sun.enterprise.config.serverbeans.JdbcConnectionPool;
import com.sun.enterprise.PoolManager;
import com.sun.enterprise.resource.MonitorableResourcePool;
import com.sun.enterprise.server.ApplicationServer;
import com.sun.enterprise.server.ResourcesUtil;
import com.sun.enterprise.server.ServerContext;
import com.sun.enterprise.Switch;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.concurrent.ConcurrentHashMap;
import javax.management.j2ee.statistics.Stats;
import com.sun.logging.LogDomains;
import java.util.logging.Logger;
import java.util.logging.Level;
import java.security.AccessController;
import java.security.PrivilegedAction;

import com.sun.enterprise.resource.MonitorableResourcePool;
import com.sun.enterprise.resource.ResourcePool;

/**
 * Provides an implementation of the MonitoringLevelListener interface to
 * receive callbacks from admin regarding change in the monitoring level.
 * Though there are 3 monitoring levels defined by JSR77, we support
 * only 2 levels - OFF and ON (HIGH/LOW). So essentially, HIGH and LOW
 * for us is only ON
 *
 * @author Aditya Gore
 * @since s1aspe 8.0
 */
public class JDBCPoolMonitoringLevelListener implements MonitoringLevelListener {
    private PoolManager poolManager_;
    private MonitoringRegistry registry_;

    private static final MonitoringLevel OFF = MonitoringLevel.OFF;
    private static final MonitoringLevel HIGH = MonitoringLevel.HIGH;
    private static final MonitoringLevel LOW = MonitoringLevel.LOW;
    
    private static Logger _logger = LogDomains.getLogger( 
        LogDomains.RSR_LOGGER );

    public JDBCPoolMonitoringLevelListener() {

    }
    /**
     * This is the callback for change in monitoring levels.
     * This shall go off once the admin team removes this method
     */
    public void setLevel( MonitoringLevel level ) {}
    
    /**
     * This is the callback method called when the monitoring level
     * changes from HIGH/LOW <-> OFF 
     * <p>Here since the MonitoringLevel is a "type-safe enum", we
     * can directly check equality using ==
     * @param from - the old level
     * @param to - the new level
     * @param handback - the stats implementation object which was
     *                 registered with the MonitoringRegistry
     */
    public void changeLevel( MonitoringLevel from, MonitoringLevel to,
        Stats handback ) {
    }

    public void changeLevel(MonitoringLevel from, MonitoringLevel to, 
        MonitoredObjectType type) 
    {
        //If we were called, the type has to be JDBC_CONN_POOL so
	//we can safely ignore the type

	if ( from == to ) {
	    //Its a no-op, so return
	    return;
	}
        AccessController.doPrivileged(new PrivilegedAction() {
            public Object run() {
	        ServerContext ctxt = ApplicationServer.getServerContext();
		if (ctxt != null) {
                    registry_ = ctxt.getMonitoringRegistry();
		}
                return null;
            }
        });


	if (from == OFF || from == LOW)  {
	    if (to == HIGH) {
		logFine("Changing level from " + from +"  to HIGH");	
                transitionToHigh();
	    }
	}

	if (from == HIGH  || from == LOW ) {
	    if ( to == OFF ) {
                logFine("Switching level from " + from + " to OFF");
                switchOffMonitoring();
            }
	}

    	if (from == OFF || from == HIGH )  {
	    if ( to == LOW  ) {
                logFine("Changing level from " + from + " to LOW");	
                transitionToLow();
	    }
	}
    }

    /*
     * Query the resources util and get a list of jdbc pools
     */
    private MonitorableResourcePool[] getPoolList() {
        ResourcesUtil resUtil = ResourcesUtil.getInstance();
	JdbcConnectionPool[] jp = resUtil.getJdbcConnectionPools();

	if (jp == null) {
	    return null;
	}
	
        MonitorableResourcePool[] pools = new MonitorableResourcePool[ jp.length ];
        ConcurrentHashMap poolTable = getPoolManager().getPoolTable();
        
	for( int i = 0 ; i < jp.length; i++ ) {
            ResourcePool p = (ResourcePool) poolTable.get( jp[i].getName() );
            if (p != null && (p instanceof MonitorableResourcePool ) ) {
	        pools[i] = (MonitorableResourcePool)p;
            }
	}
	
	return pools;
    }

    private PoolManager getPoolManager() {
	return Switch.getSwitch().getPoolManager();
    }

    private void transitionToHigh() {
        AccessController.doPrivileged( new PrivilegedAction() {    
            public Object run() {
                //we should create a new pool object everytime since
                //the stats have to be collected afresh
                MonitorableResourcePool[] pools = getPoolList();
                if (pools == null) {
                    return null;
                }
        
                for (int i = 0; i < pools.length; i++ ) {
                    if ( pools[i] != null ) {
                        try {   
                            JDBCConnectionPoolStatsImpl stats = new 
                                JDBCConnectionPoolStatsImpl(pools[i]);
                            getPoolManager().setMonitoringEnabledHigh(
                                pools[i].getPoolName() );
                            registry_.registerJDBCConnectionPoolStats( 
                                stats, pools[i].getPoolName(), null);
                        } catch(Exception mre) {
                            try {
                                _logger.log( Level.INFO, "poolmon.cannot_reg: "
                                    +pools[i].getPoolName(), mre.getMessage() );
                                getPoolManager().disableMonitoring( 
                                    pools[i].getPoolName());	
                            } catch (Exception ex) {
                                //FIXME: ignore?
                            }
                        }
                    }    
                }
                return null;
            }
        });
    }

    private void switchOffMonitoring() {
        //deregister
        AccessController.doPrivileged( new PrivilegedAction() {
            public Object run() {
                MonitorableResourcePool[] pools = getPoolList();
                if (pools == null) {
                    return null;
                }
                for( int i = 0; i < pools.length; i++) {
                    if (pools[i] != null ) {
                        try {   
                            registry_.unregisterJDBCConnectionPoolStats( 
                                pools[i].getPoolName() );
                            getPoolManager().disableMonitoring( 
                                pools[i].getPoolName());		
                        } catch( Exception mre ) {
                            _logger.log( Level.WARNING, "poolmon.cannot_unreg: "
                                + pools[i].getPoolName(), mre.getMessage() );
                        }
                    }
                }
                return null;
            }
            
        });
    }

    private void transitionToLow() {
        AccessController.doPrivileged( new PrivilegedAction() {    
            public Object run() {
                //we should create a new pool object everytime since
                //the stats have to be collected afresh
                MonitorableResourcePool[] pools = getPoolList();
                if (pools == null) {
                    return null;
                }
        
                for (int i = 0; i < pools.length; i++ ) {
                    if ( pools[i] != null ) {
                        try {   
                            JDBCConnectionPoolStatsImpl stats = new 
                                JDBCConnectionPoolStatsImpl(pools[i]);
                            getPoolManager().setMonitoringEnabledLow( 
                                pools[i].getPoolName() );	
                            registry_.registerJDBCConnectionPoolStats( 
                                stats, pools[i].getPoolName(), null);
                        } catch(Exception mre) {
                            try {
                                _logger.log( Level.INFO, "poolmon.cannot_reg: "
                                    +pools[i].getPoolName(), mre.getMessage() );
                                getPoolManager().disableMonitoring( 
                                    pools[i].getPoolName() );	
                            } catch (Exception ex) {
                                //FIXME: ignore?
                            }
                        }
                    }    
                }
                return null;
            }
        });
    }
    
    private void logFine( String msg ) {
        if ( msg != null && _logger.isLoggable( Level.FINE ) ) {
            _logger.fine( msg );
        }
    }
}    
