/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.appclient.jws;

import com.sun.enterprise.deployment.Application;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * Represents the origin of some content to be served to Java Web Start.
 * @author tjquinn
 */
public abstract class ContentOrigin {

    protected static String lineSep = System.getProperty("line.separator");
    
    /**
     *Map from URL path to the content for each file this app or
     *app client supports.
     */
    protected Map<String,Content> pathToContent;

    /** boolean indicating whether adhoc path is registered with container **/
    private boolean adhocPathRegistered = false;

    /**
     *Creates a new instance of the Entry class to record an app client
     *or J2EE application.
     *@param the application (descriptor) for the app client or J2EE app
     */
    public ContentOrigin() {
        pathToContent = new HashMap<String,Content>();
    }

    /**
     *Returns whether the administrator has enabled this content origin (either
     *a Java EE application or a stand-alone app client) for Java Web Start
     *access.
     *@return boolean indicating whether the module's app clients are enabled for JWS access
     */
    public abstract boolean isEnabled();
    
    protected Content getContent(String contentKey) {
        return pathToContent.get(contentKey);
    }

    protected DynamicContent addDynamicContent(String path, String template, Properties tokenValues, String mimeType) throws IOException {
        return addDynamicContent(path, template, tokenValues, mimeType, false /* requiresElevatedPrivs */);
    }
    
    protected DynamicContent addDynamicContent(
            String path,
            String template,
            Properties tokenValues,
            String mimeType,
            boolean requiresElevatedPrivs) {
        // merge - future
        String docText = Util.replaceTokens(template, tokenValues);
        String contentKey = getContentKeyPrefix() + path;
        DynamicContent result = new DynamicContent(this, contentKey, path, docText, mimeType, requiresElevatedPrivs);
        pathToContent.put(result.getContentKey(), result);
        return result;
    }

    protected DynamicContent addDynamicContent(DynamicContent content) {
        pathToContent.put(content.getContentKey(), content);
        return content;
    }
    
    protected StaticContent addStaticContent(String path, URI installRootURI, File file) throws URISyntaxException {
        String contentKey = getContentKeyPrefix() + path;
        StaticContent result = new StaticContent(this, contentKey, path, file, installRootURI, false /* isMainJar */);
        return addStaticContent(result);
    }
    
    protected StaticContent addStaticContent(StaticContent content) {
        pathToContent.put(content.getContentKey(), content);
        return content;
    }

    /**
     *Returns the prefix for content keys for content from this origin.
     *@return the content key prefix for this origin
     */
    protected abstract String getContentKeyPrefix();
    
    /**
     *Returns a collection of this origin's contents.
     *@param map from content keys to contents
     */
    public Collection<Content> getContents() {
        return pathToContent.values();
    }

    /**
     *Mark the origin as registered with webcontainer.
     */
    void adhocPathRegistered() {
        adhocPathRegistered=true;
    }

    /**
     *Return a boolean indicating whether this
     *is registerd with the webcontainer.
     */
    boolean isAdhocPathRegistered() {
        return adhocPathRegistered;
    }
    
    /**
     *Returns a longer display of information about this origin
     *@return detailed summary of this instance
     */
    public String toLongString() {
        return toString() + pathToContent.toString();
    }
}
