/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.ejb.portable;

import java.util.Properties;
import java.io.FileInputStream;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.naming.NameNotFoundException;

import javax.ejb.spi.HandleDelegate;
import javax.ejb.EJBException;

/**
 * Common code for looking up the java:comp/HandleDelegate.
 *
 * This class can potentially be instantiated in another vendor's container
 * so it must not refer to any non-portable RI-specific classes.
 *
 * @author Kenneth Saks
 */
public class HandleDelegateUtil 
{

    private static final String JNDI_PROPERTY_FILE_NAME = 
        "com.sun.ejb.portable.jndi.propertyfilename";

    // Class-level state for jndi properties override.  Flag is used
    // so that we only check once for jndi properties file override.
    private static boolean checkedJndiProperties = false;

    // contents of file referred to by jndi properties override
    private static Properties jndiProperties = null;

    static HandleDelegate getHandleDelegate() 
        throws NamingException
    {
        HandleDelegate handleDelegate;
        try {
            InitialContext ctx = new InitialContext();
            handleDelegate = (HandleDelegate) 
                ctx.lookup("java:comp/HandleDelegate");
        } catch(NamingException ne) {
      
            // If the lookup fails, it's probably because the default 
            // InitialContext settings needed to access the correct
            // java:comp/HandleDelegate have been overridden in this VM.  
            // In that case, check if the system value class override 
            // property file is available and if so use it.
            Properties props = null;
            try {
                props = getJndiProperties();
            } catch(Exception e) {
                // Exception while attempting to access jndi property override.
                // Create new NamingException that describes the error.
                NamingException ne2 = new NamingException
                    ("Error while accessing " + JNDI_PROPERTY_FILE_NAME +
                     " : " + e.getMessage());
                ne2.initCause(e);
                throw ne2;
            }
        
            if( props == null ) {                    
                // There was no property override set.
                NamingException ne3 = new NamingException
                    ("java:comp/HandleDelegate not found. Unable to " +
                     " use jndi property file override since " +
                     JNDI_PROPERTY_FILE_NAME + " has NOT been set");
                ne3.initCause(ne);
                throw ne3;
            }
        
            try {
                InitialContext ctx = new InitialContext(props);
                handleDelegate = (HandleDelegate) 
                    ctx.lookup("java:comp/HandleDelegate");
            } catch(NamingException ne4) {
                NamingException overrideEx = 
                    new NamingException("Unable to lookup HandleDelegate " +
                                        "with override properties = " + 
                                        props.toString());
                overrideEx.initCause(ne4);
                throw overrideEx;
            }
        }

        return handleDelegate;
    }

    /**
     * Internal method for accessing jndi properties override.  We only
     * look for properties file at most once, whether it is present or not.
     * 
     */
    private static Properties getJndiProperties() 
        throws Exception
    {

        synchronized(HandleDelegateUtil.class) {
            if( !checkedJndiProperties ) {
                try {
                    String jndiPropertyFileName = 
                        System.getProperty(JNDI_PROPERTY_FILE_NAME);
                    
                    if( jndiPropertyFileName != null ) {
                        FileInputStream fis = 
                            new FileInputStream(jndiPropertyFileName);
                        jndiProperties = new Properties();
                        jndiProperties.load(fis);
                        // Let an exception encountered here bubble up, so
                        // we can include its info in the exception propagated
                        // to the application.
                    }
                } finally {
                    // Always set to true so we don't keep doing this 
                    // system property and file access multiple times
                    checkedJndiProperties = true;
                }
            }
        }

        return jndiProperties;
    }


}
