/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.deployment.node.web;

import java.util.Map;
import java.util.Enumeration;
import java.util.logging.Level;
import org.w3c.dom.Node;

import com.sun.enterprise.deployment.xml.WebTagNames;
import com.sun.enterprise.deployment.node.DisplayableComponentNode;
import com.sun.enterprise.deployment.node.DescriptorFactory;
import com.sun.enterprise.deployment.node.XMLElement;
import com.sun.enterprise.deployment.node.SecurityRoleRefNode;
import com.sun.enterprise.deployment.node.RunAsNode;

import com.sun.enterprise.deployment.Descriptor;
import com.sun.enterprise.deployment.RoleReference;
import com.sun.enterprise.deployment.RunAsIdentityDescriptor;
import com.sun.enterprise.deployment.web.EnvironmentEntry;
import com.sun.enterprise.deployment.web.InitializationParameter;
import com.sun.enterprise.deployment.WebComponentDescriptor;

import com.sun.enterprise.deployment.util.DOLUtils;

/**
 * This node is responsible for handling the servlet xml sub tree
 *
 * @author  Jerome Dochez
 * @version 
 */
public class ServletNode extends DisplayableComponentNode {

    private final static XMLElement tag = 
        new XMLElement(WebTagNames.SERVLET);
    
    private WebComponentDescriptor descriptor;
    
    /** Creates new ServletNode */
    public ServletNode() {
        super();
        registerElementHandler(new XMLElement(WebTagNames.ROLE_REFERENCE), SecurityRoleRefNode.class);        
        registerElementHandler(new XMLElement(WebTagNames.INIT_PARAM), InitParamNode.class);                
        registerElementHandler(new XMLElement(WebTagNames.RUNAS_SPECIFIED_IDENTITY), 
                                                             RunAsNode.class, "setRunAsIdentity");                
        
    }
    
    /**
     * @return the XML tag associated with this XMLNode
     */
    protected XMLElement getXMLRootTag() {
        return tag;
    }
    
   /**
    * @return the descriptor instance to associate with this XMLNode
    */
    public Object getDescriptor() {
        
        if (descriptor==null) {
            descriptor = (WebComponentDescriptor) DescriptorFactory.getDescriptor(getXMLPath());
        }
        return descriptor;
    }     
    
    /**
     * Adds  a new DOL descriptor instance to the descriptor instance associated with 
     * this XMLNode
     *
     * @param descriptor the new descriptor
     */    
    public void addDescriptor(Object  newDescriptor) {       
        if (newDescriptor instanceof RoleReference) {  
            if (DOLUtils.getDefaultLogger().isLoggable(Level.FINE)) {            
                DOLUtils.getDefaultLogger().fine("Adding security role ref " + newDescriptor);
            }
            descriptor.addSecurityRoleReference(
                        (RoleReference) newDescriptor);    
        } else if (newDescriptor instanceof EnvironmentEntry) {            
            if (DOLUtils.getDefaultLogger().isLoggable(Level.FINE)) {            
                DOLUtils.getDefaultLogger().fine("Adding init-param " + newDescriptor);
            }
            descriptor.addInitializationParameter(
                        (InitializationParameter) newDescriptor);    
        } else super.addDescriptor(newDescriptor);
    }
    
    /**
     * all sub-implementation of this class can use a dispatch table to map xml element to
     * method name on the descriptor class for setting the element value. 
     *  
     * @return the map with the element name as a key, the setter method as a value
     */    
    protected Map getDispatchTable() {
        // no need to be synchronized for now
        Map table = super.getDispatchTable();
        table.put(WebTagNames.NAME, "setName");        
        table.put(WebTagNames.SERVLET_NAME, "setCanonicalName");
        table.put(WebTagNames.LOAD_ON_STARTUP, "setLoadOnStartUp");
        return table;
    }        

    /**
     * receives notiification of the value for a particular tag
     * 
     * @param element the xml element
     * @param value it's associated value
     */
    public void setElementValue(XMLElement element, String value) {
        if (WebTagNames.SERVLET_CLASS.equals(element.getQName())) {
            descriptor.setServlet(true);            
            descriptor.setWebComponentImplementation(value);
        } else if (WebTagNames.JSP_FILENAME.equals(element.getQName())) {
            descriptor.setServlet(false);
            descriptor.setWebComponentImplementation(value);
        } else {
            super.setElementValue(element, value);
        }
    }    
    
    /**
     * write the descriptor class to a DOM tree and return it
     *
     * @param parent node in the DOM tree 
     * @param node name for the root element of this xml fragment      
     * @param the descriptor to write
     * @return the DOM tree top node
     */
    public Node writeDescriptor(Node parent, WebComponentDescriptor descriptor) {    

        Node myNode = super.writeDescriptor(parent, descriptor);
        appendTextChild(myNode, WebTagNames.SERVLET_NAME, descriptor.getCanonicalName());         
        if (descriptor.isServlet()) {
            appendTextChild(myNode, WebTagNames.SERVLET_CLASS, descriptor.getWebComponentImplementation());
        } else {
            appendTextChild(myNode, WebTagNames.JSP_FILENAME, descriptor.getWebComponentImplementation());
        }
        
        // init-param*
        WebBundleNode.addInitParam(myNode, WebTagNames.INIT_PARAM, descriptor.getInitializationParameters());
        
        if (descriptor.getLoadOnStartUp()!=-1) {
            appendTextChild(myNode, WebTagNames.LOAD_ON_STARTUP, String.valueOf(descriptor.getLoadOnStartUp()));
        }
        
        // run-as
        RunAsIdentityDescriptor runAs = descriptor.getRunAsIdentity();
        if (runAs!=null) {
            RunAsNode runAsNode = new RunAsNode();
            runAsNode.writeDescriptor(myNode, WebTagNames.RUNAS_SPECIFIED_IDENTITY, runAs);
        }
        
        // sercurity-role-ref*
        Enumeration roleRefs = descriptor.getSecurityRoleReferences();
        SecurityRoleRefNode roleRefNode = new SecurityRoleRefNode();
        while (roleRefs.hasMoreElements()) {
            roleRefNode.writeDescriptor(myNode, WebTagNames.ROLE_REFERENCE, 
                            (RoleReference) roleRefs.nextElement());            
        }
        
        return myNode;
    }
}
