/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 * ServiceReferenceNode.java
 *
 * Created on March 21, 2002, 2:38 PM
 */

package com.sun.enterprise.deployment.node;

import java.util.*;
import java.util.logging.Level;
import org.w3c.dom.Node;

import com.sun.enterprise.deployment.xml.WebServicesTagNames;
import com.sun.enterprise.deployment.xml.TagNames;
import com.sun.enterprise.deployment.ServiceReferenceDescriptor;
import com.sun.enterprise.deployment.ServiceRefPortInfo;
import com.sun.enterprise.deployment.InjectionTarget;
import com.sun.enterprise.deployment.util.DOLUtils;
import javax.xml.namespace.QName;

/** 
 * This node is responsible for loading web services
 * reference information
 *
 * @author  Kenneth Saks
 */
public class ServiceReferenceNode extends DisplayableComponentNode {
    // used for temporary storage during element processing
    private String serviceEndpointInterface = null;
    private ServiceRefPortInfo portInfo = null;

    /** Creates a new instance of ServiceReferenceNode */
    public ServiceReferenceNode() {
        super();
        registerElementHandler
            (new XMLElement(WebServicesTagNames.HANDLER),
             WebServiceHandlerNode.class, "addHandler");
        registerElementHandler
            (new XMLElement(WebServicesTagNames.HANDLER_CHAIN),
             WebServiceHandlerChainNode.class, "addHandlerChain");
        registerElementHandler
            (new XMLElement(TagNames.INJECTION_TARGET), 
             InjectionTargetNode.class, "addInjectionTarget");         
    }
    
    /**
     * all sub-implementation of this class can use a dispatch table 
     * to map xml element to method name on the descriptor class for 
     * setting the element value. 
     *  
     * @return map with the element name as a key, the setter method as a value
     */
    protected Map getDispatchTable() {
        Map table = super.getDispatchTable();
        table.put(WebServicesTagNames.SERVICE_REF_NAME, "setName");
        table.put(WebServicesTagNames.SERVICE_INTERFACE, "setServiceInterface");
        table.put(WebServicesTagNames.WSDL_FILE, "setWsdlFileUri");
        table.put(WebServicesTagNames.JAXRPC_MAPPING_FILE, "setMappingFileUri");
        table.put(WebServicesTagNames.MAPPED_NAME, "setMappedName");
        table.put(WebServicesTagNames.SERVICE_REF_TYPE, "setInjectionTargetType");
        return table;
    }

    private ServiceReferenceDescriptor getServiceReferenceDescriptor() {
        return (ServiceReferenceDescriptor) getDescriptor();
    }

    /**
     * receives notiification of the value for a particular tag
     * 
     * @param element the xml element
     * @param value it's associated value
     */
    public void setElementValue(XMLElement element, String value) {    
        String qname = element.getQName();
        if (WebServicesTagNames.SERVICE_ENDPOINT_INTERFACE.equals(qname)) {
            portInfo = getServiceReferenceDescriptor().getPortInfoBySEI(value);
            if( portInfo == null  ) {
                portInfo = getServiceReferenceDescriptor().
                    addContainerManagedPort(value);
            }
        } else if( WebServicesTagNames.SERVICE_QNAME.equals(qname) ) {
            String prefix = getPrefixFromQName(value);
            String localPart = getLocalPartFromQName(value);
            String namespaceUri = resolvePrefix(element, prefix);
            if( namespaceUri == null) {
                DOLUtils.getDefaultLogger().log(Level.SEVERE, "enterprise.deployment.backend.invalidDescriptorMappingFailure",
                    new Object[] { prefix , getServiceReferenceDescriptor().getName()});
            } else {
                QName serviceName = new QName(namespaceUri, localPart);
                getServiceReferenceDescriptor().setServiceName
                    (serviceName, prefix);
            }
        } else if(WebServicesTagNames.ENABLE_MTOM.equals(qname)) {
            portInfo.setMtomEnabled(value);
        } else if( WebServicesTagNames.PORT_COMPONENT_LINK.equals(qname) ) {
            // set link name.  link resolution will be performed during
            // validation stage.
            portInfo.setPortComponentLinkName(value);
            portInfo = null;
        } else {
            super.setElementValue(element, value);
        }
    }    
    
    /**
     * write the method descriptor class to a query-method DOM tree and 
     * return it
     *
     * @param parent node in the DOM tree 
     * @param node name for the root element of this xml fragment      
     * @param the descriptor to write
     * @return the DOM tree top node
     */
    public Node writeDescriptor(Node parent, String nodeName, 
                                ServiceReferenceDescriptor descriptor) {        
        Node serviceRefNode = 
            super.writeDescriptor(parent, nodeName, descriptor);

        writeDisplayableComponentInfo(serviceRefNode, descriptor);
        appendTextChild(serviceRefNode, WebServicesTagNames.SERVICE_REF_NAME, 
                        descriptor.getName());
        appendTextChild(serviceRefNode, WebServicesTagNames.SERVICE_INTERFACE, 
                        descriptor.getServiceInterface());
        appendTextChild(serviceRefNode, WebServicesTagNames.SERVICE_REF_TYPE, 
                descriptor.getInjectionTargetType());        
        appendTextChild(serviceRefNode, WebServicesTagNames.WSDL_FILE,
                        descriptor.getWsdlFileUri());
        appendTextChild(serviceRefNode, WebServicesTagNames.JAXRPC_MAPPING_FILE,
                        descriptor.getMappingFileUri());

        if( descriptor.hasServiceName() ) {
            QName serviceName = descriptor.getServiceName();
            appendQNameChild(WebServicesTagNames.SERVICE_QNAME, serviceRefNode,
                             serviceName.getNamespaceURI(),
                             serviceName.getLocalPart(), 
                             descriptor.getServiceNameNamespacePrefix());
        }

        for(Iterator iter = descriptor.getPortsInfo().iterator(); 
            iter.hasNext();) {
            ServiceRefPortInfo next = (ServiceRefPortInfo) iter.next();
            if( next.isContainerManaged() ) {
                String serviceEndpointInterface = 
                    next.getServiceEndpointInterface();
                Node portComponentRefNode = appendChild
                    (serviceRefNode, WebServicesTagNames.PORT_COMPONENT_REF);
                appendTextChild(portComponentRefNode, 
                                WebServicesTagNames.SERVICE_ENDPOINT_INTERFACE,
                                serviceEndpointInterface);
                appendTextChild(portComponentRefNode,
                                WebServicesTagNames.ENABLE_MTOM,
                                next.getMtomEnabled());
                appendTextChild(portComponentRefNode,
                                WebServicesTagNames.PORT_COMPONENT_LINK,
                                next.getPortComponentLinkName());
            }
        }

        WebServiceHandlerNode handlerNode = new WebServiceHandlerNode();
        handlerNode.writeWebServiceHandlers(serviceRefNode, 
                                            descriptor.getHandlers());

        WebServiceHandlerChainNode handlerChainNode = 
                new WebServiceHandlerChainNode();
        handlerChainNode.writeWebServiceHandlerChains(serviceRefNode, 
                                            descriptor.getHandlerChain());
                
        appendTextChild(serviceRefNode, WebServicesTagNames.MAPPED_NAME,
                descriptor.getMappedName());
        
        if (descriptor.isInjectable()) {
            InjectionTargetNode ijNode = new InjectionTargetNode();
            for (InjectionTarget target : descriptor.getInjectionTargets()) {
                ijNode.writeDescriptor(serviceRefNode, TagNames.INJECTION_TARGET, target);
            }
        }

        return serviceRefNode;
    }    
    
}
