/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.deployment.node;

import java.util.Map;
import org.w3c.dom.Node;

import com.sun.enterprise.deployment.InjectionTarget;
import com.sun.enterprise.deployment.xml.TagNames;
import com.sun.enterprise.deployment.Descriptor;
import com.sun.enterprise.deployment.EnvironmentProperty;

/**
 * This node is responsible for handling all env-entry related xml tags
 *
 * @author  Jerome Dochez
 * @version 
 */
public class EnvEntryNode extends DeploymentDescriptorNode {
    
    private boolean setValueCalled = false;

    public EnvEntryNode() {
        super();
        registerElementHandler(new XMLElement(TagNames.INJECTION_TARGET), 
                                InjectionTargetNode.class, "addInjectionTarget");                          
    }
    
   /**
     * all sub-implementation of this class can use a dispatch table to map xml element to
     * method name on the descriptor class for setting the element value. 
     *  
     * @return the map with the element name as a key, the setter method as a value
     */    
    protected Map getDispatchTable() {    
        Map table = super.getDispatchTable();
        table.put(TagNames.ENVIRONMENT_PROPERTY_NAME, "setName");
        table.put(TagNames.ENVIRONMENT_PROPERTY_VALUE, "setValue");
        table.put(TagNames.ENVIRONMENT_PROPERTY_TYPE, "setType");        
        table.put(TagNames.MAPPED_NAME, "setMappedName");
        return table;
    }    
    
    public boolean endElement(XMLElement element) {

        if (TagNames.ENVIRONMENT_PROPERTY_NAME.equals(element.getQName())) {
            // name element is always right before value, so initialize
            // setValueCalled to false when it is processed.
            setValueCalled = false;
        } else if( TagNames.ENVIRONMENT_PROPERTY_VALUE.equals
                   (element.getQName()) ) {
            setValueCalled = true;
        }

        return super.endElement(element);
    } 

    public void addDescriptor(Object newDescriptor) {
        if( setValueCalled ) {
            super.addDescriptor(newDescriptor);
        } else {
            // Don't add it to DOL.  The env-entry only exists
            // at runtime if it has been assigned a value.
        }
    }

    /**
     * write the descriptor class to a DOM tree and return it
     *
     * @param parent node in the DOM tree 
     * @param node name for the root element of this xml fragment      
     * @param the descriptor to write
     * @return the DOM tree top node
     */
    public Node writeDescriptor(Node parent, String nodeName, Descriptor descriptor) {    
        if (!(descriptor instanceof EnvironmentProperty)) {
            throw new IllegalArgumentException(getClass() + " cannot handles descriptors of type " + descriptor.getClass());
        }            
        EnvironmentProperty envProp = (EnvironmentProperty) descriptor;
        Node envEntryNode = super.writeDescriptor(parent, nodeName, descriptor);
        
        writeLocalizedDescriptions(envEntryNode, envProp);
        appendTextChild(envEntryNode, TagNames.ENVIRONMENT_PROPERTY_NAME, envProp.getName());
        appendTextChild(envEntryNode, TagNames.ENVIRONMENT_PROPERTY_TYPE, envProp.getType());
        appendTextChild(envEntryNode, TagNames.ENVIRONMENT_PROPERTY_VALUE, envProp.getValue());
        appendTextChild(envEntryNode, TagNames.MAPPED_NAME, envProp.getMappedName());
        if( envProp.isInjectable() ) {
            InjectionTargetNode ijNode = new InjectionTargetNode();
            for (InjectionTarget target : envProp.getInjectionTargets()) {
                ijNode.writeDescriptor(envEntryNode, TagNames.INJECTION_TARGET, target);
            }
        }

        return envEntryNode;
    }
}
