/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.admin.wsmgmt.repository.spi;

import com.sun.enterprise.admin.wsmgmt.repository.impl.*;

/**
 * This SPI mechanism allows RepositoryProvider instance. This repority provider
 * can be queried for existence of web services.  
 * <br>
 * Because the methods in the RepositoryFactory are all static, there is only 
 * one RepositoryFactory per Java VM at any one time. This ensures that there is
 * a single source from which RepositoryProvider implementation is obtained. 
 * <br>
 * Registering RepositoryProvider
 * <br>
 * The name of the provider is supplied on the command line. If no provider is
 * specified, default implementation is assumed. Default provider class is
 * com.sun.enterprise.admin.repository.impl.ApplicationServerRepositoryProvider.
 * For example the following 
 * -Drepository.provider.classname=com.sun.web.WebServerRepositoryProvider
 * would change the provider class to com.sun.web.WebServerRepositoryProvider.
 */
public class RepositoryFactory {

    /**
     * Make RepositoryFactory private, only one sigleton object is available.
     */
    private RepositoryFactory () {
    }

    /**
     * Returns the RepositoryFactory singleton.
     *
     * @return the RepositoryFactory instance
     */
    public static RepositoryFactory  getRepositoryFactory() {
        return new RepositoryFactory();
    }

    /**
     * Returns the RepositoryProvider instance. If 
     * -Drepository.provider.classname is defined, that class is loaded and
     * returned as the repository provider. If there is an error finding or 
     * loading the class, the default provider class is returned.
     *
     * @return RepositoryProvider implementation
     * @throws IllegalAccessException - if the class or its nullary constructor 
     *                                  is not accessible. 
     *         InstantiationException - if this Class represents an abstract 
     *                                  class,an interface, an array class, a 
     *                                  primitive type, or void; or if the class     *                                  has no nullary constructor; or if the 
     *                                  instantiation fails for some other 
     *                                  reason. 
     *         ClassCastException     - if the provider implementation specified
     *                                  by -D does not implement the com.sun.
     *                                  enterprise.admin.wsmgmt.repository.spi.
     *                                  RepositoryProvider interface.
     *         ClassNotFoundException - if the provider implementation specified     *                                  by -D does could not be found by the 
     *                                  class loader.
     */
    public static RepositoryProvider getRepositoryProvider() 
      throws InstantiationException, IllegalAccessException, ClassCastException,
        ClassNotFoundException {
       String implName = System.getProperty(REPOSITORY_PROVIDER_NAME);
       if ( implName == null ) {
            return new AppServRepositoryProvider();
       } else {
            Class repClass = Class.forName(implName);
            Object o = repClass.newInstance();
            return (RepositoryProvider)o;

       }
    }

    /**
     * Returns the WebServiceInfoProvider instance. If 
     * -Dwebserviceinfo.provider.classname is defined, that class is loaded and
     * returned as the repository provider. If there is an error finding or 
     * loading the class, the default provider class is returned.
     *
     * @return WebServiceInfoProvider implementation
     * @throws IllegalAccessException - if the class or its nullary constructor 
     *                                  is not accessible. 
     *         InstantiationException - if this Class represents an abstract 
     *                                  class,an interface, an array class, a 
     *                                  primitive type, or void; or if the class     *                                  has no nullary constructor; or if the 
     *                                  instantiation fails for some other 
     *                                  reason. 
     *         ClassCastException     - if the provider implementation specified
     *                                  by -D does not implement the com.sun.
     *                                  enterprise.admin.wsmgmt.repository.spi.
     *                                  RepositoryProvider interface.
     *         ClassNotFoundException - if the provider implementation specified     *                                  by -D does could not be found by the 
     *                                  class loader.
     */
    public static WebServiceInfoProvider getWebServiceInfoProvider() 
     throws InstantiationException, IllegalAccessException, ClassCastException,
     ClassNotFoundException {
       String implName = System.getProperty(WEBSERVICE_INFO_PROVIDER_NAME);
       if ( implName == null ) {
            //return new AppServWebServiceInfoProvider();
            Class repClass = Class.forName(AS_DEFAULT_PROVIDER);
            return (WebServiceInfoProvider) repClass.newInstance();
       } else {
            Class repClass = Class.forName(implName);
            return (WebServiceInfoProvider) repClass.newInstance();
       }
    }

    /** Environment property name to customize Repository Provider */
    public static final String REPOSITORY_PROVIDER_NAME = 
        "repository.provider.classname";

    /** Environment property name to customize Web Service Info Provider */
    public static final String WEBSERVICE_INFO_PROVIDER_NAME = 
        "webservice_info.provider.classname";
    
    private static final String AS_DEFAULT_PROVIDER = 
        "com.sun.enterprise.tools.common.AppServWebServiceInfoProvider";
}
