/*
Copyright (C) 2000  Steffen Zschaler

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * PropertyPageContainer.java
 *
 * Created on 13. September 2000, 12:10
 */
 
package tudresden.ocl.injection.reverseeng.propertypages;

import tudresden.ocl.injection.reverseeng.propertypages.events.*;

import java.util.*;

import javax.swing.event.*;

/** 
  * A container for property pages.
  * 
  * @author  sz9 (Steffen Zschaler)
  * @version 1.0
  */
public class PropertyPageContainer extends javax.swing.JPanel implements PropertyPageListener {

  /**
    * Property page displayed, when no properties are available.
    */
  protected PropertyPage m_ppNoProperties = new DefaultPropertyPage();
  
  /**
    * List of property pages currently displayed.
    * 
    * @element-type PropertyPage
    */
  protected List m_lppPropertyPages = new LinkedList();
  
  /**
    * True, if property pages other than {@link #m_ppNoProperties} are
    * being displayed.
    */
  protected boolean m_fPropertiesAvailable = true;
  
  /**
    * The listeners registered to receive events from this container.
    */
  protected EventListenerList m_ellListeners = new EventListenerList();
  
  /** 
    * Create new PropertyPageContainer 
    */
  public PropertyPageContainer() {
    initComponents ();
    
    removeAllPropertyPages();
  }

  /** This method is called from within the constructor to
   * initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is
   * always regenerated by the FormEditor.
   */
  private void initComponents () {//GEN-BEGIN:initComponents
    m_jtpPropertyPages = new javax.swing.JTabbedPane ();
    setLayout (new java.awt.GridBagLayout ());
    java.awt.GridBagConstraints gridBagConstraints1;

    m_jtpPropertyPages.setTabPlacement (3);


    gridBagConstraints1 = new java.awt.GridBagConstraints ();
    gridBagConstraints1.gridwidth = 0;
    gridBagConstraints1.gridheight = 0;
    gridBagConstraints1.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints1.anchor = java.awt.GridBagConstraints.NORTHWEST;
    gridBagConstraints1.weightx = 1.0;
    gridBagConstraints1.weighty = 1.0;
    add (m_jtpPropertyPages, gridBagConstraints1);

  }//GEN-END:initComponents


  // Variables declaration - do not modify//GEN-BEGIN:variables
  private javax.swing.JTabbedPane m_jtpPropertyPages;
  // End of variables declaration//GEN-END:variables

  /**
    * Removes all property pages and sets {@link #m_ppNoProperties}.
    */
  public void removeAllPropertyPages() {
    if (m_fPropertiesAvailable) {
      while (m_fPropertiesAvailable &&
              (m_lppPropertyPages.size() > 0)) {
        removePropertyPage ((PropertyPage) m_lppPropertyPages.get (0));
      }
      
      if (m_fPropertiesAvailable) {
        // Should only happen with freshly constructed PropertyPageContainers.
        addPropertyPage (m_ppNoProperties);
        
        m_fPropertiesAvailable = false;
      }
    }
  }

  public void removePropertyPage (PropertyPage pp) {
    if (m_lppPropertyPages.indexOf (pp) > -1) {
      // Remove property page
      m_jtpPropertyPages.remove (pp.getComponent());
      m_lppPropertyPages.remove (pp);
 
      pp.onPropertyPageRemoved (this);
      firePropertyPageRemoved (pp);
      
      // If no property pages left, and we did not just remove 
      // m_ppNoProperties, set it and mark that no properties are 
      // available.
      if (m_fPropertiesAvailable &&
          (m_lppPropertyPages.size() == 0)) {
        addPropertyPage (m_ppNoProperties);
        
        m_fPropertiesAvailable = false;
      }
    }
  }
  
  public void addPropertyPage (PropertyPage pp) {
    if (! m_fPropertiesAvailable) {
      removePropertyPage ((PropertyPage) m_lppPropertyPages.get (0));
      
      m_fPropertiesAvailable = true;
    }
    
    m_jtpPropertyPages.addTab (pp.getTitle(), pp.getIcon(), pp.getComponent(), pp.getToolTip());
    m_jtpPropertyPages.setEnabledAt (m_jtpPropertyPages.getTabCount() - 1, pp.isEnabled());
    pp.getComponent().setEnabled (pp.isEnabled());
    
    m_lppPropertyPages.add (pp);
    
    firePropertyPageAdded (pp);
  }
  
  /**
    * Replace the complete set of property pages with the specified set.
    *
    * <p>This method attempts to maintain the selected page.</p>
    *
    * @param appNewPageSet the set of pages to be displayed
    */
  public void replacePropertyPages(PropertyPage[] appNewPageSet) {
    replacePropertyPages (Arrays.asList (appNewPageSet));
  }

  /**
    * Replace the complete set of property pages with the specified set.
    *
    * <p>This method attempts to maintain the selected page.</p>
    *
    * @param lppNewPageSet the set of pages to be displayed
    */
  public void replacePropertyPages(List lppNewPageSet) {
    int nSelectedPage = m_jtpPropertyPages.getSelectedIndex();
    
    removeAllPropertyPages();
    
    for (Iterator i = lppNewPageSet.iterator(); i.hasNext();) {
      addPropertyPage (((PropertyPage) i.next()));
    }
    
    if (m_jtpPropertyPages.getTabCount() > nSelectedPage) {
      m_jtpPropertyPages.setSelectedIndex (nSelectedPage);
    }
  }

  /** 
    * Invoked when the icon of a property page changes.
    * @param ppe {@link EventObject} describing the {@link PropertyPage} that changed.
    */
  public void onIconChanged (PropertyPageEvent ppe) {
    int nIdx = m_lppPropertyPages.indexOf (ppe.getSourcePage());
    
    if (nIdx > -1) {
      m_jtpPropertyPages.setIconAt (nIdx, ppe.getSourcePage().getIcon());
    }
  }
  
  /** 
    * Invoked when the title of a property page changes.
    * @param ppe {@link EventObject} describing the {@link PropertyPage} that changed.
    */
  public void onTitleChanged(PropertyPageEvent ppe) {
    int nIdx = m_lppPropertyPages.indexOf (ppe.getSourcePage());
    
    if (nIdx > -1) {
      m_jtpPropertyPages.setTitleAt (nIdx, ppe.getSourcePage().getTitle());
    }
  }
  
  /** 
    * Invoked when the tool tip associated with a property page changes.
    * @param ppe {@link EventObject} describing the {@link PropertyPage} that changed.
    */
  public void onToolTipChanged(PropertyPageEvent ppe) {
    // A little more tricky, because they forgot setToolTipTextAt!    
    PropertyPage pp = ppe.getSourcePage();
    int nIdx = m_lppPropertyPages.indexOf (pp);

    if (nIdx > -1) {
      int nSelectedPage = m_jtpPropertyPages.getSelectedIndex();

      // Remove old page
      m_jtpPropertyPages.removeTabAt (nIdx);

      // Insert updated page
      m_jtpPropertyPages.insertTab (pp.getTitle(), pp.getIcon(), pp.getComponent(), pp.getToolTip(), nIdx);

      // Restore selection
      m_jtpPropertyPages.setSelectedIndex (nSelectedPage);
    }
  }
  
  /** 
    * Invoked when the component that represents a property page has changed.
    *
    * <p>This event will not be invoked, when merely the data displayed in
    * the component or subcomponents of it have changed.</p>
    * @param ppe {@link EventObject} describing the {@link PropertyPage} that changed.
    */
  public void onComponentChanged(PropertyPageEvent ppe) {
    PropertyPage pp = ppe.getSourcePage();
    int nIdx = m_lppPropertyPages.indexOf (pp);
    
    if (nIdx > -1) {
      m_jtpPropertyPages.setComponentAt (nIdx, pp.getComponent());
      pp.getComponent().setEnabled (pp.isEnabled());
    }
  }
  
  /**
    * Invoked when the enabled state of a property page has changed.
    * @param ppe {@link EventObject} describing the {@link PropertyPage} that changed.
    */
  public void onEnabledChanged(PropertyPageEvent ppe) {
    PropertyPage pp = ppe.getSourcePage();
    int nIdx = m_lppPropertyPages.indexOf (pp);
    
    if (nIdx > -1) {
      m_jtpPropertyPages.setEnabledAt (nIdx, pp.isEnabled());
      pp.getComponent().setEnabled (pp.isEnabled());
    }
  }
 
  /**
    * Start notifying the specified listener of events regarding this property page container.
    *
    * @param ppcl the listener to be notified of events.
    */
  public void addPropertyPageContainerListener (PropertyPageContainerListener ppcl) {
    m_ellListeners.add (PropertyPageContainerListener.class, ppcl);
  }

  /**
    * Stop notifying the specified listener of events regarding this property page container.
    *
    * @param ppcl the listener to be notified of events.
    */
  public void removePropertyPageContainerListener (PropertyPageContainerListener ppcl) {
    m_ellListeners.remove (PropertyPageContainerListener.class, ppcl);
  }
  
  protected void firePropertyPageAdded (PropertyPage pp) {
    Object[] listeners = m_ellListeners.getListenerList();
    PropertyPageContainerEvent ppce = null;
    
    // Process the listeners last to first, notifying
    // those that are interested in this event
    for (int i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == PropertyPageContainerListener.class) {
        if (ppce == null) {
          ppce = new PropertyPageContainerEvent (this, pp);
        }
        
        try {
          ((PropertyPageContainerListener) listeners[i + 1]).onPropertyPageAdded (ppce);
        }
        catch (Throwable t) {
          System.err.println ("Exception occurred while dispatching PropertyPageContainerEvent: \n");
          t.printStackTrace();
        }
      }
    }    
  }
  
  protected void firePropertyPageRemoved (PropertyPage pp) {
    Object[] listeners = m_ellListeners.getListenerList();
    PropertyPageContainerEvent ppce = null;
    
    // Process the listeners last to first, notifying
    // those that are interested in this event
    for (int i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == PropertyPageContainerListener.class) {
        if (ppce == null) {
          ppce = new PropertyPageContainerEvent (this, pp);
        }
        
        try {
          ((PropertyPageContainerListener) listeners[i + 1]).onPropertyPageRemoved (ppce);
        }
        catch (Throwable t) {
          System.err.println ("Exception occurred while dispatching PropertyPageContainerEvent: \n");
          t.printStackTrace();
        }
      }
    }    
  }
}