#!/bin/bash

# Copyright (c) 2002-2007 Userful Corporation. All rights reserved.
# http://www.userful.com/

# install-license.sh - get some info from the system, submit it via URL, and 
# if we get a vailid license: install it, if not: display a meaningful
# error message

function usage() {
    echo "
This script attempts to install a license key from store.userful.com.

Options:
   -a code      supply an Activation code
   -f file      supply a File that contains an activation code
   -d           Delete the activation file after using it (happens even
                if there is an error)
   -D           only Delete the file if a license was successfully installed
                if there is an error)
   -n           do Not use the graphical display 
   -h           display this Help message and exit
"
}

ACTIVATION_CODE=""
ACTIVATION_FILE=""
ACTIVATION_CODE_FROM_FILE=""
DELETE_ACTIVATION_FILE=false
ACTIVATION_CODE=""
SUPPLIED_ACTIVATION_FILE=false
NON_GRAPHICAL=false
DELETE_ACTIVATION_FILE_ON_SUCCESS=false

# parse arguments:
while getopts "a:df:hn" flag ; do
    case "$flag" in
        a)
            ACTIVATION_CODE="$OPTARG"
            ;;
        d)
            DELETE_ACTIVATION_FILE=true
            ;;
        D)
            DELETE_ACTIVATION_FILE_ON_SUCCESS=true
            ;;
        f)
            SUPPLIED_ACTIVATION_FILE=true
            ACTIVATION_FILE="$OPTARG"
            ;;
        h)
            usage
            exit 0
            ;;
        n)
            NON_GRAPHICAL=true
            ;;
        *)
            usage
            exit 1
            ;;
    esac
done

echo "remaining arguments: $@ OPTIND: $OPTIND"

if [ -z "$ACTIVATION_CODE" ] ; then
    ACTIVATION_CODE="`echo $* | awk "{ print \$$OPTIND }" `"
fi

if [ $NON_GRAPHICAL == true ] ; then
    unset DISPLAY
fi

if [ $DELETE_ACTIVATION_FILE == true ] && [ $SUPPLIED_ACTIVATION_FILE == false ] ; then
    echo "ERROR: specfied activation code file deletion, but did not specify an activation code file"
    usage
    exit 1
fi

# Deal with the activation code file
if [ $SUPPLIED_ACTIVATION_FILE == true ] ; then
    if [ -n "$ACTIVATION_FILE" ] && [ -f "$ACTIVATION_FILE" ] ; then
        ACTIVATION_CODE_FROM_FILE=`cat "$ACTIVATION_FILE"` || exit 1
        if [ -z "$ACTIVATION_CODE_FROM_FILE" ] ; then
            echo "ERROR: activation file \"$ACTIVATION_FILE\" is empty"
            usage
            exit 1
        fi
    else
        echo "ERROR: \"$ACTIVATION_FILE\" does not exist"
        usage
        exit 1
    fi

    if [ "$DELETE_ACTIVATION_FILE" == true ] ; then
        if [ -n "$ACTIVATION_FILE" ] && [ -f $ACTIVATION_FILE ] ; then
            rm -f $ACTIVATION_FILE || exit 1
        fi
    fi
fi

# at this point the non-getopt style arguments should remain - this is backward compatable with the way it used to work

# check to see is local versions of the programs exists, 
# if they do, use them, if not, use the default install locations
# if those don't exists use 'echo'
ERR_DLG=/usr/share/1box-utils/error-dialog.py
if [ ! -x $ERR_DLG ] ; then
    ERR_DLG=./error-dialog.py
fi 
if [ ! -x $ERR_DLG ] ; then
    ERR_DLG=echo
fi

ACTIVATION_DLG=./activation-dialog.py
if [ ! -x $ACTIVATION_DLG ] ; then
    ACTIVATION_DLG=/usr/share/1box-utils/activation-dialog.py
fi
if [ ! -x $ACTIVATION_DLG ] ; then
    ACTIVATION_DLG=$ERR_DLG
fi

PROGRESS_DLG=./progress-dialog.py
if [ ! -x $PROGRESS_DLG ] ; then
    PROGRESS_DLG=/usr/share/1box-utils/progress-dialog.py
fi
if [ ! -x $PROGRESS_DLG ] ; then
    PROGRESS_DLG=$ERR_DLG
fi

if [ -z "$DISPLAY" ] ; then
    ERR_DLG=echo
    ACTIVATION_DLG=echo
    PROGRESS_DLG=echo
fi

$PROGRESS_DLG &
PROGRESS_DLG_PID=$!
KILL_PROGRESS="kill -SIGHUP $PROGRESS_DLG_PID"

URL="http://store.userful.com/license/control/GetLicense?"

# We always include an activation code even if it is empty because the 
# server requires it
if [ -n "$ACTIVATION_CODE_FROM_FILE" ] ; then
    if [ -n "$ACTIVATION_CODE" ] ; then
        echo "WARNING: both an activation code and activation code file were supplied, using activation code from file."
    fi
    ACTIVATION_CODE="$ACTIVATION_CODE_FROM_FILE"
fi
URL="${URL}activation=$ACTIVATION_CODE"

ETH_NUM=0
MAC=`LANG=C /sbin/ifconfig -a eth$ETH_NUM | grep "HWaddr" | awk '{ print $5 }'`
if [ -z "$MAC" ] || [ ${#MAC} != 17 ] ; then 
    # MAC is no good
    MAC=nomac
fi

# license file order:
# http://wiki.userful.ca/wiki/index.php?title=License_File
# order of preffered MAC addr formats: : - _

MAC_COLONS=$MAC
MAC_DASHES=${MAC//:/-}
MAC_UNDERS=${MAC//:/_}

FILENAME=
if [ -e /etc/X11/$MAC_COLONS.1Box.info ]; then
    FILENAME="/etc/X11/$MAC_COLONS.1Box.info"
elif [ -e /etc/X11/$MAC_DASHES.1Box.info ]; then
    FILENAME=/etc/X11/$MAC_DASHES.1Box.info
elif [ -e /etc/X11/$MAC_UNDERS.1Box.info ]; then
    FILENAME=/etc/X11/$MAC_UNDERS.1Box.info
elif [ -e /etc/X11/1Box.info ] ; then
    FILENAME="/etc/X11/1Box.info"
elif [ -e /etc/X11/1Box.info.trial ]; then
    FILENAME="/etc/X11/1Box.info.trial"
fi

LICENSE_KEY=
if [ -n "$FILENAME" ] ; then
    echo "Found existing license: $FILENAME"
    LICENSE_KEY="`/usr/share/1box-utils/b64encode.py $FILENAME`"
    URL="${URL}&"
    URL="${URL}lickey=$LICENSE_KEY"
else
    URL="${URL}&lickey="
fi

if [ $MAC == nomac ] ; then
    exec $ERR_DLG "text=No valid MAC address found. Please contact Userful: http://store.userful.com/ or 1-(866)-USERFUL."
else
    URL="${URL}&"
    URL="${URL}mac=${MAC//:/}"
fi

# if an oem file exists, get the hash and send that too
OEMFILE=/etc/userful/oem
if [ -f $OEMFILE ] ; then
    which sha1sum &> /dev/null || OEMHASH=none
    if [ "$OEMHASH" != none ] ; then
        SHA=`sha1sum $OEMFILE`
        OEMHASH=${SHA// *}
        URL="${URL}&oem=$OEMHASH"
    fi
fi

# add the language to the url
if [ -n "$LANG" ] ; then
    URL="${URL}&"
	URL="${URL}lang=$LANG"
else
    # if it's empty, send it anyway
    URL="${URL}&"
	URL="${URL}lang="
fi

# add the version to the url
FILE=/etc/userful/dm-release
VERSION=unknown
if [ -f $FILE ] ; then
    OS_VERSION=`/usr/share/1box-utils/b64encode.py $FILE`
fi
URL="${URL}&"
URL="${URL}version=$OS_VERSION"

# add the OS version to the URL
FILE=/etc/userful/os-info
OS_VERSION=unknown
if [ -f $FILE ] ; then
    OS_VERSION=`/usr/share/1box-utils/b64encode.py $FILE`
fi
URL="${URL}&"
URL="${URL}osversion=$OS_VERSION"

TMP_DIR=/tmp/install-license-$$
mkdir -p $TMP_DIR

SRC_FILE=$TMP_DIR/1Box.info

echo "Attempting to get license file from $URL"

# Get license file

C=`which curl`
W=`which wget`

RET=1
# we are no longer installing "get-url.py" because the required libraries are rarely installed
if [ -x /usr/share/1box-utils/get-url.py ] ; then
	/usr/share/1box-utils/get-url.py "$URL" >  $SRC_FILE 2> /dev/null
	RET=$?
fi
if [ $RET != 0 ] ; then
    if [ -n "$C" ] && [ -x $C ] ; then 
        curl "$URL" > $SRC_FILE 2> /dev/null
    elif [ -n "$W" ] && [ -x $W ] ; then 
        wget -O $SRC_FILE "$URL"
    else
        if [ -d /proc/$PROGRESS_DLG_PID ] ; then
            $KILL_PROGRESS
        fi
        exec $ERR_DLG "Could not use python urllib, find 'wget', or 'curl' to download license file. Go to http://store.userful.com/ to get your license."
    fi
fi

# make the command line version look a little nicer
if [ -z "$DISPLAY" ] ; then
    echo
fi

# verify license file
SIZE=`stat -c "%s" $SRC_FILE`

if [ -d /proc/$PROGRESS_DLG_PID ] ; then
	$KILL_PROGRESS
fi

if [ $SIZE != 256 ] ; then
	# license is bad: it's actually an error message - show the message (or do something else)
    if [ -n "$ACTIVATION_CODE" ] ; then
        exec $ACTIVATION_DLG key="$ACTIVATION_CODE" error_text="`cat $SRC_FILE`" 
    else
        if [ -z "$DISPLAY" ] ; then
            echo "We did not recive a license file from the server."
            echo "Please use go to http://store.userful.com/ to register. After registering, run this script again."
            echo "The text returned from the server was: "
        fi
        exec $ACTIVATION_DLG "error_text=`cat $SRC_FILE`" 
    fi
else
	:
	# license is good
fi

# license is good: install license file
INSTALL_DIR=/etc/X11
if [ ! -w $INSTALL_DIR ] ; then
    mv -f $SRC_FILE /tmp/1Box.info &> /dev/null && SRC_FILE=/tmp/1Box.info
    exec $ERR_DLG "License key downloaded but not installed. You must be root for automatic installation. The file has been saved to $SRC_FILE. Ask your system administrator to copy this file to $INSTALL_DIR or re-run this application as root."
fi

BACKUP_DIR=$INSTALL_DIR/1Box-info-backup
mkdir -p $BACKUP_DIR 

if ls $INSTALL_DIR/*1Box.info* &> /dev/null ; then
    mv -f --backup=numbered $INSTALL_DIR/*1Box.info* $BACKUP_DIR
fi

mv $SRC_FILE $INSTALL_DIR/1Box.info 

if [ "$DELETE_ACTIVATION_FILE_ON_SUCCESS" == true ] ; then
    if [ -n "$ACTIVATION_FILE" ] && [ -f $ACTIVATION_FILE ] ; then
        rm -f $ACTIVATION_FILE || exit 1
    fi
fi

exec $ERR_DLG type=sucess text="Successfully installed license file."
