//
// File:        StubSource.java
// Package:     gov.llnl.babel.backend.c
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: StubSource.java,v 1.29 2003/09/18 14:46:41 epperly Exp $
// Description: generate C backend stub source to a pretty writer stream
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.c;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * Class <code>StubSource</code> generates an C Backend stub source file
 * to a language writer output stream.  The constructor takes a language
 * writer stream and method <code>generateCode</code> generates intermediate
 * object representation for the specified symbol to the output stream.  The
 * language writer output stream is not closed by this object.
 */
public class StubSource {
  private LanguageWriterForC d_writer;

  /**
   * This is a convenience utility function that writes the C client
   * stub source information into the provided language writer output
   * stream.  The output stream is not closed on exit.  A code
   * generation exception is thrown if an error is detected.
   *
   * @param sym the <code>Symbol</code> whose stub source is being written.
   *
   * @param writer the output writer to which the stub source will
   *               be written. This will not be closed.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  public static void generateCode(Symbol             symbol,
                                  LanguageWriterForC writer)
    throws CodeGenerationException
  {
    StubSource source = new StubSource(writer);
    source.generateCode(symbol);
  }

  /**
   * Create a <code>StubSource</code> object that will write symbol
   * information to the provided output writer stream.
   *
   * @param writer the output writer to which the header will be written.
   *               This will not be closed.
   */
  public StubSource(LanguageWriterForC writer) {
    d_writer = writer;
  }

  /**
   * Write C stub source information for the provided symbol to the
   * language writer output stream provided in the constructor.  This
   * method does not close the language writer output stream and may
   * be called for more than one symbol (although the generated source
   * may not be valid input for the C compiler).  A code generation
   * exception is generated if an error is detected.  No code is
   * generated for enumerated and package symbols.
   *
   * @param sym the <code>Symbol</code> whose stub source is being written.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  public void generateCode(Symbol symbol)
    throws CodeGenerationException
  {
    if (symbol != null) {
      switch (symbol.getSymbolType()) {
        case Symbol.CLASS:
          generateExtendable((Extendable) symbol);
          break;
        case Symbol.ENUM:
          generateEnum(symbol);
          break;
        case Symbol.INTERFACE:
          generateExtendable((Extendable) symbol);
          break;
        case Symbol.PACKAGE:
          break;
        default:
          throw new CodeGenerationException("Unsupported symbol type.");
      }
    } else {
      throw new CodeGenerationException("Unexpected null Symbol.");
    }
  }

  /**
   * Generate the Stub source for a SIDL class or interface.  The source
   * file begins with a banner and include files followed by the stub
   * implementations for the methods.
   *
   * @param ext the <code>Extendable</code> whose stub source is being written.
   */
  private void generateExtendable(Extendable ext)
    throws CodeGenerationException
  {
    /*
     * Generate the file banner and include files.
     */
    SymbolID id        = ext.getSymbolID();
    String   source    = C.getStubFile(id);
    String   header    = C.getHeaderFile(id);
    String   iorHeader = IOR.getHeaderFile(id);

    d_writer.writeBanner(ext, source, false, 
      CodeConstants.C_DESC_STUB_PREFIX + id.getFullName());

    d_writer.generateInclude(header, false);
    d_writer.generateInclude(iorHeader, false);
    d_writer.generateInclude("SIDL_interface_IOR.h", true);
    d_writer.printlnUnformatted("#include <stddef.h>");
    if (!iorHeader.equals("SIDL_BaseInterface_IOR.h")) {
      d_writer.generateInclude("SIDL_BaseInterface_IOR.h", false);
    }
    if (!BabelConfiguration.isSIDLBaseClass(id)) {
      d_writer.printlnUnformatted("#include \"babel_config.h\"");
      d_writer.printlnUnformatted("#ifdef SIDL_DYNAMIC_LIBRARY");
      d_writer.printlnUnformatted("#include <stdio.h>");
      d_writer.printlnUnformatted("#include <stdlib.h>");
      d_writer.printlnUnformatted("#include \"SIDL_Loader.h\"");
      d_writer.printlnUnformatted("#endif");
    }
    d_writer.println();

    /*
     * Output standard function stubs for methods in the object.
     */

    if (!ext.isInterface()) {
      generateGetIOR(id);
    }
    if (ext.hasStaticMethod(true)) {
      generateGetStaticEPV(id);
    }
    generateMethodStubs(ext);
    ArrayMethods ar = new ArrayMethods(id, false);
    ar.generateStub(d_writer);
  }

  /**
   * Generate the Stub source for a SIDL enum.  The source
   * file begins with a banner and include files followed by the stub
   * implementations for the methods.
   *
   * @param enm the <code>Enumeration</code> whose stub source is being
   * written .
   */
  private void generateEnum(Symbol enm)
    throws CodeGenerationException
  {
    /*
     * Generate the file banner and include files.
     */
    SymbolID id        = enm.getSymbolID();
    String   source    = C.getStubFile(id);
    String   header    = C.getHeaderFile(id);

    d_writer.writeBanner(enm, source, false, 
      CodeConstants.C_DESC_STUB_PREFIX + id.getFullName());

    d_writer.generateInclude(header, false);
    d_writer.generateInclude("SIDL_int_IOR.h", true);
    d_writer.printlnUnformatted("#include <stddef.h>");
    d_writer.println();

    ArrayMethods ar = new ArrayMethods(id, true);
    ar.generateStub(d_writer);
  }

  /**
   * Generate method implementations for the methods in the interface
   * or class.  This method will also generate the constructor and cast
   * functions.
   */
  private void generateMethodStubs(Extendable ext)
    throws CodeGenerationException
  {
    /*
     * Generate the name for this entry point vector as well as a pointer
     * to "self" for this structure.  For classes, self will be the object
     * structure whereas for interfaces it is void*.
     */
    SymbolID id   = ext.getSymbolID();
    String   self = C.getSymbolName(id) + " self";

    /*
     * Output the constructor if this class or interface is not abstract.
     */

    if (!ext.isAbstract()) {
      d_writer.writeComment("Constructor function for the class.", false);
      d_writer.println(C.getSymbolName(id));
      d_writer.println(C.getFullMethodName(id, "_create") + "()");
      d_writer.println("{");
      d_writer.increaseTabLevel();
      d_writer.println("return (*(_getIOR()->createObject))();");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();
    }

    /*
     * Output each of the method implementations from the interface or class.
     */
    Collection methods = ext.getMethods(true);
    for (Iterator m = methods.iterator(); m.hasNext(); ) {
      Method method = (Method) m.next();
      generateMethodSignature(self, ext.isInterface(), id, method);
      d_writer.println("{");
      d_writer.increaseTabLevel();
      if (C.getReturnString(method.getReturnType()) != "void") {
        d_writer.print("return ");
      }
      if (Method.STATIC == method.getDefinitionModifier()) {
        d_writer.print("(_getSEPV()");
      } else {
        d_writer.print("(*self->d_epv");
      }
      d_writer.print("->");
      d_writer.print(IOR.getVectorEntry(method.getLongMethodName()));
      d_writer.println(")(");
      d_writer.increaseTabLevel();
      generateArgumentList(self, ext.isInterface(), id, method, false);
      d_writer.println(");");
      d_writer.decreaseTabLevel();
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();
    }

    /*
     * Output the cast methods for the class or interface.
     */

    final Method castMethod = 
      IOR.getBuiltinMethod(IOR.CAST, id);
    d_writer.writeComment(
      "Cast method for interface and class type conversions.", false);
    d_writer.println(C.getSymbolName(id));
    d_writer.println(C.getFullMethodName(id, "_cast") + "(");
    d_writer.increaseTabLevel();
    d_writer.println("void* obj)");
    d_writer.decreaseTabLevel();
    d_writer.println("{");
    d_writer.increaseTabLevel();
    d_writer.println(C.getSymbolName(id) + " cast = NULL;");
    d_writer.println();
    d_writer.println("if (obj != NULL) {");
    d_writer.increaseTabLevel();
    d_writer.println("SIDL_BaseInterface base = (SIDL_BaseInterface) obj;");
    d_writer.println("cast = ("
                   + C.getSymbolName(id)
                   + ") (*base->d_epv->"
                   + IOR.getVectorEntry(castMethod.getLongMethodName())
                   + ")(");
    d_writer.increaseTabLevel();
    d_writer.println("base->d_object,");
    d_writer.println("\"" + id.getFullName() + "\");");
    d_writer.decreaseTabLevel();
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
    d_writer.println("return cast;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();

    d_writer.writeComment(
      "String cast method for interface and class type conversions.", false);
    d_writer.println("void*");
    d_writer.println(C.getFullMethodName(id, "_cast2") + "(");
    d_writer.increaseTabLevel();
    d_writer.println("void* obj,");
    d_writer.println("const char* type)");
    d_writer.decreaseTabLevel();
    d_writer.println("{");
    d_writer.increaseTabLevel();
    d_writer.println("void* cast = NULL;");
    d_writer.println();
    d_writer.println("if (obj != NULL) {");
    d_writer.increaseTabLevel();
    d_writer.println("SIDL_BaseInterface base = (SIDL_BaseInterface) obj;");
    d_writer.println("cast = (*base->d_epv->" +
                     IOR.getVectorEntry(castMethod.getLongMethodName()) +
                     ")(base->d_object, type);");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
    d_writer.println("return cast;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
  }

  /**
   * Generate the method's signature.
   *
   * @param self the String representing the method's self argument.
   *
   * @param is_interface the boolean indicating whether working with a class
   *                     or an interface
   *
   * @param id the <code>SymbolID</code> of the <code>Extendable</code> whose
   *   stub source is being written.
   *
   * @param method the <code>Method</code> whose signature is being output.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void generateMethodSignature(String   self,
                                       boolean  is_interface,
                                       SymbolID id,
                                       Method   method)
    throws CodeGenerationException
  {
    d_writer.writeComment(method, false);
    d_writer.println(C.getReturnString(method.getReturnType()));
    d_writer.print(C.getFullMethodName(id, method));
    d_writer.println("(");
    d_writer.increaseTabLevel();
    generateArgumentList(self, is_interface, id, method, true);
    d_writer.println(")");
    d_writer.decreaseTabLevel();
  }

  /**
   * Generate the method's argument list.
   *
   * @param self the String representing the method's self argument as it
   *             should appear in the method's signature.
   *
   * @param is_interface the boolean indicating whether working with a class
   *                     or an interface
   *
   * @param id the <code>SymbolID</code> of the <code>Extendable</code> whose
   *           stub source is being written.
   *
   * @param method the <code>Method</code> whose list is being output.
   *
   * @param in_signature the boolean indicating whether or not the arguments
   *                     are being output as part of the method's signature.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void generateArgumentList(String   self,
                                    boolean  is_interface,
                                    SymbolID id,
                                    Method   method,
                                    boolean  in_signature)
    throws CodeGenerationException
  {
    /*
     * If the method is not static, then it will begin with an
     * object reference.
     */
    boolean has_throws = !method.getThrows().isEmpty();
    List    args       = method.getArgumentList();

    if (!method.isStatic()) {
      if (in_signature) {
        d_writer.print(self);
      } else if (is_interface) {
        d_writer.print("self->d_object");
      } else {
        d_writer.print("self");
      }
      if ((args.size() > 0) || has_throws) {
        d_writer.println(",");
      }
    } else {
      if ((args.size() == 0) && !has_throws) {
        if (in_signature) {
          d_writer.print("void");
        }
      }
    }

    /*
     * Output each argument in turn.
     */
    for (Iterator a = args.iterator(); a.hasNext(); ) {
      Argument arg = (Argument) a.next();
      if (in_signature) {
        d_writer.print(C.getArgumentWithFormal(arg));
      } else {
        d_writer.print(arg.getFormalName());
      }
      if (a.hasNext() || (has_throws)) {
        d_writer.println(",");
      }
    }

    /*
     * If there is a throws clause, then return the exception type
     * as the last in the argument list associated with the signature.
     */
    if (has_throws) {
      if (in_signature) {
        d_writer.print("SIDL_BaseException *_ex");
      } else {
        d_writer.print("_ex");
      }
    }
  }

  /**
   * Generate the _getIOR function that provides access to the IOR functions
   * either through static linking or dynamic loading.
   */
  private void generateGetIOR(SymbolID id) {
    final String ext_name = IOR.getExternalName(id);
    d_writer.writeComment("Hold pointer to IOR functions.", false);
    d_writer.println("static const " + ext_name + " *_ior = NULL;");
    d_writer.writeComment("Lookup the symbol to get the IOR functions.",
                          false);
    d_writer.println("static const " + ext_name + "* _loadIOR(void)");
    
    d_writer.writeComment("Return pointer to internal IOR functions.", false);
    d_writer.println("{");
    d_writer.increaseTabLevel();
    if (BabelConfiguration.isSIDLBaseClass(id)) {
      d_writer.println("_ior = " + IOR.getExternalFunc(id) + "();");
    } else {
      d_writer.printlnUnformatted("#ifdef SIDL_STATIC_LIBRARY");
      d_writer.println("_ior = " + IOR.getExternalFunc(id) + "();");
      d_writer.printlnUnformatted("#else");
      d_writer.println("const " + ext_name + "*(*dll_f)(void) =");
      d_writer.increaseTabLevel();
      d_writer.print("(const " + ext_name + "*(*)(void)) ");
      d_writer.println("SIDL_Loader_lookupSymbol(");
      d_writer.increaseTabLevel();
      d_writer.println("\"" + IOR.getExternalFunc(id) + "\");");
      d_writer.decreaseTabLevel();
      d_writer.decreaseTabLevel();
      d_writer.println("_ior = (dll_f ? (*dll_f)() : NULL);");
      d_writer.println("if (!_ior) {");
      d_writer.increaseTabLevel();
      d_writer.disableLineBreak();
      d_writer.println(
        "fputs(\"Babel: unable to load the implementation for "
        + id.getFullName() + "; please set SIDL_DLL_PATH\\n\", stderr);");
      d_writer.enableLineBreak();
      d_writer.println("exit(-1);");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.printlnUnformatted("#endif");
    }
    d_writer.println("return _ior;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");

    d_writer.println();
    d_writer.printlnUnformatted("#define _getIOR() (_ior ? _ior : _loadIOR())");

    d_writer.println();
  }


  /**
   * Return the pointer that provides access to the static EPV.
   */
  private void generateGetStaticEPV(SymbolID id) {
    String sepv_name = IOR.getSEPVName(id);
    d_writer.writeComment("Hold pointer to static entry point vector",false);
    d_writer.println("static const " + sepv_name + " *_sepv = NULL;");
    d_writer.writeComment("Return pointer to static functions.",false);
    d_writer.printlnUnformatted("#define _getSEPV() (_sepv ? _sepv : (_sepv = (*(_getIOR()->getStaticEPV))()))");
    d_writer.println();
  }
}
