//
// File:        Configuration.java
// Package:     gov.llnl.babel.backend
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: Configuration.java,v 1.3 2002/08/20 16:55:30 epperly Exp $
// Description: singleton configuration file for the backend code generators
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend;

import gov.llnl.babel.symbols.SymbolUtilities;
import gov.llnl.babel.BabelConfiguration;

/**
 * <code>Configuration</code> is a singleton class that represents the
 * generic backend configuration options such as the maximum array size
 * in a SIDL array or whether a symbol is part of the standard library.
 */
public class Configuration {
   private static Configuration s_instance = null;

   /**
    * Create a new instance of the backend configuration object.  Although
    * singleton classes do not typically define a public constructor, this
    * implementation does so to support multiple configurations in the same
    * application.  Most implementations, however, will not directly create
    * a configuration through the constructor and will instead use the
    * singleton methods <code>getInstance</code> and <code>setInstance</code>.
    */
   public Configuration() {
   }

   /**
    * Return the singleton instance of the configuration.  If the instance
    * has not yet been created, then it will be created by this call.
    */
   public static Configuration getInstance() {
      if (s_instance == null) {
         s_instance = new Configuration();
      }
      return s_instance;
   }

   /**
    * Set the singleton instance of the configuration.  Use this function
    * only if you want to change the default configuration implementation
    * or use multiple configurations in the same application.  Otherwise,
    * <code>getInstance</code> will automatically create the configuration
    * for you.
    */
   public static void setInstance(Configuration instance) {
      s_instance = instance;
   }

   /**
    * Return the maximum array dimension supported during code generation.
    */
   public int getMaximumArrayDimension() {
     return BabelConfiguration.getMaximumArray();
   }

   /**
    * Check whether the symbol name represents one of the standard library
    * classes (e.g., SIDL).
    */
   public boolean isBuiltinSymbol(String fqn) {
      boolean is_builtin = false;
      if (fqn != null) {
         String p = SymbolUtilities.getOutermostPackage(fqn);
         if ("SIDL".equals(p)) {
            is_builtin = true;
         }
      }
      return is_builtin;
   }
}
