//
// File:        CodeSplicer.java
// Package:     gov.llnl.babel.backend
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: CodeSplicer.java,v 1.14 2003/09/15 23:32:35 kumfert Exp $
// Description: splice code from an old user file into a new generated file
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend;

import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.backend.writers.LineRedirector;
import gov.llnl.babel.backend.writers.LineCountingFilterWriter;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * Class <code>CodeSplicer</code> splices code segments from an existing
 * user file into a new automatically generated file.  The intended use of
 * the code splicer is to preserve user edits to generated code.  The new
 * file replaces the original one; however, the code splicer preserves areas
 * delineated with the pair of strings:
 * <code>
 * DO-NOT-DELETE splicer.begin(symbol)
 * DO-NOT-DELETE splicer.end(symbol)
 * </code>
 * embedded in comments in both the edited and generated files.  The
 * code splicer also tracks which symbols were not used in the newly
 * generated file, and these symbols may be output at the end of the
 * new file to preserve code for the user.
 */
public class CodeSplicer {
   private final static String s_target = "DO-NOT-DELETE splicer";
   private final static String s_begin  = s_target + ".begin(";
   private final static String s_end    = s_target + ".end(";

   private Map d_symbol_edits;
   private Set d_used_symbols;
   private String d_vpath;
   private String d_path; 
   private Map d_splice_bounding_lines;
   private LineRedirector d_lr;

   /**
    * Return the code splicer start string for the beginning of a code
    * splice region.  The symbol string must be unique within this file.
    * The symbol may not contain a close parenthesis.
    */
   public static String getBeginString(String symbol) {
      return s_begin + symbol + ")";
   }

   /**
    * Return the code splicer end string for the closing of a code splice
    * region.  The symbol string must be unique within this file and must
    * match the string used to open the splice region.  The symbol may not
    * contain a close parenthesis.
    */
   public static String getEndString(String symbol) {
      return s_end + symbol + ")";
   }

   /**
    * Create a new instance of the code splicer class.  The constructor
    * will read data from the buffered reader stream and throw an IO exception
    * if any error condition is encountered while reading from the file.
    * The code splices are stored in a map with keyvalues that are the
    * region symbols and corresponding symbols that are the lines of code
    * retrieved from the file.
    */
   public CodeSplicer(BufferedReader reader, String vpath, String path)
       throws IOException {
      d_symbol_edits = new HashMap();
      d_used_symbols = new HashSet();
      if (vpath==null||vpath.equals("")||vpath.equals(".") ) { 
         d_vpath=".";
      } else { 
         d_vpath = vpath;
      }
      d_path = path;
      if (d_path.equals(d_vpath)) { 
	  d_vpath=".";
      }
      d_splice_bounding_lines = new HashMap();
      d_lr = null;
      populateDatabase(reader);
   }
  
   /**
    * Create an empty CodeSplicer.  Query Symbol will return false
    * in every instance with an empty CodeSplicer.
    */
   public CodeSplicer() { 
      d_symbol_edits = new HashMap();
      d_used_symbols = new HashSet();
      d_vpath = ".";
      d_splice_bounding_lines = new HashMap();
      d_lr = null;
   }    


   public void setLineRedirector( LineRedirector lr ) { 
      d_lr = lr;
   }

   /** query the current vpath setting
    *  "." indicates no vpath
    */
   public String getVPath() { 
      return d_vpath;
   }

   /** explicitly set the vpath.  
    *  Note that vpath=null, "" or "."
    *  are all handled internally as "."
    */
   public void setVPath( String vpath ) { 
      if (vpath==null||vpath.equals("")||vpath.equals(".") ) { 
         d_vpath=".";
      } else { 
         d_vpath = vpath;
      }
   }  

   /**
    * Query whether the specified symbol exists in the symbol database.
    * The symbol may not contain a close parenthesis and it may not start
    * or end with white space.
    */
   public boolean hasSymbol(String symbol) {
      return d_symbol_edits.containsKey(symbol);
   }

   /**
    * Output symbol information to the specified output print writer.  If
    * the symbol does not exist in the database, then nothing is output to
    * the print stream.  If the symbol exists, then the symbol begin and
    * end comments are output as well as any text between those comments.
    * The symbol may not contain a close parenthesis and it may not start
    * or end with white space.
    */
  public void outputSymbolEdits(String symbol, PrintWriter writer) { 
      String edits = (String) d_symbol_edits.get(symbol);
      if (edits != null) {
         if ((!d_vpath.equals(".")) && d_lr != null ) { 
            int[] bounds = (int[]) d_splice_bounding_lines.get(symbol);
            d_lr.redirectBegin(d_vpath, bounds[0]);	    
            int start=d_lr.getLineCount();
            writer.print(edits);
            writer.flush();
            int end=d_lr.getLineCount();
 	    d_lr.redirectEnd(d_path, end);
	 } else { 
           writer.print(edits);
         }
         d_used_symbols.add(symbol);
      }
   }

   /**
    * Retrieve the edit string associated with the specified symbol.  If
    * no symbol exists in the database, then return null.  The symbol may
    * not contain a close parenthesis and it may not start or end with
    * white space.
    */
   public String getEditString(String symbol) {
      return (String) d_symbol_edits.get(symbol);
   }

   /**
    * Returns TRUE if there is at least one unused edit in the symbol
    * edit database.  See outputUnusedSymbolEdits() for more information.
    */
   public boolean hasUnusedSymbolEdits() {
     return d_used_symbols.size() < d_symbol_edits.size();
   }

   /**
    * Output the unused edits in the symbol edit database.  These symbols
    * were read from the input file but were not used in the output file.
    * The symbols will be output to the specified output print writer in
    * no particular order.
    */
   public void outputUnusedSymbolEdits(PrintWriter writer) {
      for (Iterator i = d_symbol_edits.entrySet().iterator(); i.hasNext(); ) {
         Map.Entry entry  = (Map.Entry) i.next();
         String    symbol = (String) entry.getKey();
         String    edits  = (String) entry.getValue();
         if (!d_used_symbols.contains(symbol)) {
            writer.print(edits);
         }
      }
   }

   /**
    * Retrieve a <code>Set</code> of the symbols in the symbol edit
    * database.  Each entry in the set is a string representing a symbol.
    */
   public Set getSymbols() {
      return d_symbol_edits.keySet();
   }

   /**
    * Populate the symbol edits database from the specified file.  Any
    * input errors will return a <code>IOException</code>.
    */
   private void populateDatabase(BufferedReader reader) throws IOException {
      /*
       * Search for beginning tag lines until we reach the end of the file.
       */
      boolean eof = false;
      int line=0;
      while (!eof) {
         String s = reader.readLine();
	 line++;
         if (s == null) {
            eof = true;
         } else {

            if (s.indexOf(s_end) >=0 ) {
               // bug #548: end(a) without leading start(a) is an error
               throw new IOException("found \'splicer.end()\' outside of a splicer block\n" + 
                                     line + ": " + s );
            }
            /*
             * If not EOF check whether this line contains the start string.
             */
            String symbol = extractSymbol(s);
            if (symbol != null) {
		int[] bounds = new int[2];
		bounds[0]=line;

               /*
                * Add lines to the writer until the end string is detected.
                */
               StringWriter sw = new StringWriter();
               PrintWriter  pw = new PrintWriter(sw);
               pw.print(s + '\n');

               String  end = getEndString(symbol);
               boolean eob = false;
               while (!eob) {
                  String b = reader.readLine();
		  line++;
                  if (b == null) {
                     eob = true;
                     eof = true;
                  } else if (b.indexOf(s_begin) >=0 ) {
                     // bug #548: begin(a),begin(b),end(a) is an error
                     throw new IOException("encountered \'splicer.begin()\' inside of a splicer block \"" + symbol + "\"\n" + 
                                           line + ": " + b);
                  } else if (b.indexOf(end) >= 0) {
                     eob = true;
                  } else if (b.indexOf(s_end) >=0) { 
                     // bug #548: begin(a),end(b),end(a) is an error
                     throw new IOException("encountered \'splicer.end()\' inside of a splicer block \"" + symbol + "\"\n" + 
                                           line + ": " + b);
                  }
                  pw.print(b + '\n');
               }
               pw.flush();
               d_symbol_edits.put(symbol, sw.toString());
	       bounds[1]=line;
	       d_splice_bounding_lines.put(symbol,bounds);
            }
         }
      }
   }

   /**
    * Extract a symbol from the specified string.  If the specified string
    * is not formatted correctly, then return null.
    */
   private String extractSymbol(String s) {
      String symbol = null;

      if (s != null) {
         int front = s.indexOf(s_begin);
         if (front >= 0) {
            front += s_begin.length();
            int back = s.indexOf(")", front);
            if (back >= front) {
               symbol = s.substring(front, back).trim();
            }
         }
      }

      return symbol;
   }

  /**
   * The easiest method to envoke to create a spliced region 
   * during code generation.
   * @param symbol The string symbol to match.  Should be unique in a file
   * @param writer The LanguageWriter to embed the symbols in appropriate
   * comments
   * @param alt_msg An alternate message to embed in comments only if no
   * information was found by the splicer.
   */
  public void splice( String symbol, LanguageWriter writer, String alt_msg) {
    if ( hasSymbol( symbol ) ) { 
      outputSymbolEdits( symbol, writer.getPrintWriter() );
    } else { 
      writer.disableLineBreak();
      writer.writeCommentLine( getBeginString( symbol ) );
      if ( alt_msg != null && alt_msg.length()>0 ) { 
        writer.writeCommentLine( alt_msg );
      }
      writer.writeCommentLine( getEndString( symbol ) );
      writer.enableLineBreak();
    }
  }

  /**
   * The easiest method to envoke to create a spliced region 
   * during code generation.
   * @param symbol The string symbol to match.  Should be unique in a file
   * @param writer The LanguageWriter to embed the symbols in appropriate
   * comments
   * @param alt_msg An alternate message to embed in comments only if no
   * @param alt_code A default code chunk (used for languages that REQUIRE a return value (or some such)
   * information was found by the splicer.
   */
  public void splice( String symbol, LanguageWriter writer, String alt_msg, String alt_code) {
    if ( hasSymbol( symbol ) ) { 
      outputSymbolEdits( symbol, writer.getPrintWriter() );
    } else { 
      writer.disableLineBreak();
      writer.writeCommentLine( getBeginString( symbol ) );
      if ( alt_msg != null && alt_msg.length()>0 ) { 
        writer.writeCommentLine( alt_msg );
      }
      if ( alt_code != null && alt_code.length()>0) { 
	  writer.println(alt_code);
      }
      writer.writeCommentLine( getEndString( symbol ) );
      writer.enableLineBreak();
    }
  }

   /**
    * Test out the code splicer class.  The single argument is the edit input
    * file from which to extract code splices.  The splices are then output
    * to the standard system output.
    */
   public static void main(String[] args) {
      if (args.length != 1) {
         System.err.println(
            "usage: gov.llnl.babel.backend.CodeSplicer input-file");
         System.exit(1);
      }

      try {
         BufferedReader reader = new BufferedReader(new FileReader(args[0]));
         CodeSplicer cs = new CodeSplicer(reader,".",args[0]);
         reader.close();
         PrintWriter pw = new PrintWriter(System.out);
         cs.outputUnusedSymbolEdits(pw);
         pw.flush();
      } catch (IOException ex) {
         System.out.println(ex.getMessage());
         System.exit(1);
      }
   }
}
