/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Ant", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.aspectj.tools.ant.taskdefs.compilers;

import org.apache.tools.ant.*;
import org.apache.tools.ant.types.*;
import org.apache.tools.ant.types.Path;
import org.apache.tools.ant.taskdefs.*;
import org.apache.tools.ant.taskdefs.compilers.*;
import java.io.*;
import java.util.Iterator;
import java.util.List;
import java.util.ArrayList;
import java.lang.reflect.*;

//XXX
import java.util.*;

/**
 * Ajc uses this as the CompilerAdapter.
 *
 * This task was developed by the <a href="http://aspectj.org">AspectJ Project</a>
 *
 * @author <a href="mailto:palm@parc.xerox.com">Jeffrey Palm</a>
 * @see    org.aspectj.tools.ant.taskdefs.Ajc
 */
public class Ajc extends DefaultCompilerAdapter {

    /** The value of a compiler success. */
    public final static int AJC_COMPILER_SUCCESS = 0;

    /** The name of the compiler's main class. */
    private final static String MAIN_CLASS_NAME = "org.aspectj.tools.ajc.Main";
    
    /**
     * List of arguments allowed only by javac and <b>not</b> ajc.
     */    
    final static List javacOnlyFlags
        = finalList(new String[] { "-g:none", "-g:lines",
        "-g:vars", "-g:source", "-nowarn"});
    final static List javacOnlyArgs  
        = finalList(new String[] { "-sourcepath",
        "-encoding", "-target" });

    private static List finalList(String[] args) {
        List result = new ArrayList();
        result.addAll(Arrays.asList(args));
        return Collections.unmodifiableList(result);
    }

    /**
     * Checks the command line for arguments allowed only in AJC and
     * disallowed by AJC and then calls the <code>compile()<code> method.
     *
     * @return true if a good compile, false otherwise.
     * @throws org.apache.tools.ant.BuildException
     */
    public boolean execute() throws BuildException {
        attributes.log("Using AJC", Project.MSG_VERBOSE);
        return compile(addAjcOptions(setupJavacCommand()));
    }

    /**
     * Invokes the AJC compiler using reflection.
     *
     * @param cline the command line to pass the compiler
     * @return      true for a good compile (0), false otherwise
     * @throws      org.apache.tools.ant.BuildException
     */
    private boolean compile(Commandline cline) throws BuildException {
        PrintStream err = System.err;
        PrintStream out = System.out;
        try {
            Class main = Class.forName(MAIN_CLASS_NAME);
            if (main == null) {
                throw new ClassNotFoundException(MAIN_CLASS_NAME);
            }
            PrintStream logstr =
                new PrintStream(new LogOutputStream(attributes,
                                                    Project.MSG_WARN));
            System.setOut(logstr);
            System.setErr(logstr);
            return ((Integer)main.getMethod
                    ("compile", new Class[]{String[].class}).invoke
                    (main.newInstance(), new Object[]{
                        removeUnsupported(cline, logstr)
                    })).intValue() == AJC_COMPILER_SUCCESS;
        } catch (Exception e) {
            if (e instanceof BuildException) {
                throw (BuildException)e;
            } else {
                throw new BuildException("Error starting AJC compiler",
                                         e, location);
            }
        } finally {
            System.setErr(err);
            System.setOut(out);
        }
    }

    
    /**
     * Removes unsupported arguments from <code>cline</code>
     * issuing warnings for each using <code>log</code>.
     *
     * @param cline the <code>org.apache.tools.ant.types.Commandline</code> from
     *              which the argument is removed.
     * @return      a new <code>java.lang.String</code> array containing all the
     *              supported arguments found in <code>cline</code>.
     * @throws      org.apache.tools.ant.BuildException
     */
    private String[] removeUnsupported(Commandline cline, PrintStream log) {
        if (null == log) log = System.err;
        String[] args = cline.getCommandline();
        List argsList = new ArrayList();
        for (int i = 0; i < args.length; i++) {
            String arg = args[i];
            if (javacOnlyFlags.contains(arg)) {
              log.println("ignored by ajc " + arg);
            } else if (javacOnlyArgs.contains(arg)) {
              i++;
              if (i < args.length) {
                arg = arg + " " + args[i];
              }
              log.println("ignored by ajc " + arg);
            } else {
                argsList.add(args[i]);
            }
        }
        return (String[])argsList.toArray(new String[argsList.size()]);
    }
    
    /**
     * Adds arguments that setupJavacCommand() doesn't pick up.
     *
     * @param cline <code>org.apache.tools.ant.types.Commandline</code> to
     *              which arguments are added.
     * @throws      org.apache.tools.ant.BuildException
     * @see         AjcCompiler#ajcOnlyArgs
     * @see         org.apache.tools.ant.taskdefs.compilers#DefaultCompilerAdapter.setupJavacCommand()
     */
    private Commandline addAjcOptions(Commandline cline) throws BuildException {
        Javac javac = getJavac();
                               
        org.aspectj.tools.ant.taskdefs.Ajc2 ajc = null;

        try {
            ajc = (org.aspectj.tools.ant.taskdefs.Ajc2)javac;
        } catch (ClassCastException cce) {
            throw new BuildException(cce+"");
        }
        
        if (ajc.getThreads() != null) {
            cline.createArgument().setValue("-threads");
            cline.createArgument().setValue(ajc.getThreads() + "");
        }
        if (ajc.getNocomments()) {
            cline.createArgument().setValue("-nocomments");
        }
        if (ajc.getNosymbols()) {
            cline.createArgument().setValue("-nosymbols");
        }
        if (ajc.getPreprocess()) {
            cline.createArgument().setValue("-preprocess");
        }
        if (ajc.getWorkingdir() != null) {
            cline.createArgument().setValue("-workingdir");
            cline.createArgument().setFile(ajc.getWorkingdir());
        }

        return cline;
    }

    /**
     * Logs the compilation parameters, adds the files to compile and logs the 
     * &qout;niceSourceList&quot;
     */
    protected void logAndAddFilesToCompile(Commandline cmd) {

        // Same behavior as DefaultCompilerAdapter.logAndAddFilesToCompile
        attributes.log("Compilation args: " + cmd.toString(), Project.MSG_VERBOSE);
        StringBuffer niceSourceList = new StringBuffer("File");
        if (compileList.length != 1) {
            niceSourceList.append("s");
        }
        niceSourceList.append(" to be compiled:");
        niceSourceList.append(lSep);

        for (int i=0; i < compileList.length; i++) {

            // DefaultCompilerAdapter only expects .java files but we must deal
            // with .lst files also
            File file = compileList[i];

            if (file == null) continue;

            String arg = file.getAbsolutePath();
            String rest = "";
            String name = file.getName();

            // For .java files take the default behavior and add that
            // file to the command line
            if (name.endsWith(".java")) {
                cmd.createArgument().setValue(arg);
            }
            niceSourceList.append("   " + arg + rest + lSep);
        }
        attributes.log(niceSourceList.toString(), Project.MSG_VERBOSE);
    }    
}
