/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Ant", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.aspectj.tools.ant.taskdefs;

import org.apache.tools.ant.*;
import org.apache.tools.ant.types.*;
import org.apache.tools.ant.types.Path;
import org.apache.tools.ant.taskdefs.*;
import org.apache.tools.ant.taskdefs.compilers.*;
import java.io.*;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.Vector;
import java.util.Iterator;
import java.util.ArrayList;
import java.util.StringTokenizer;
import java.lang.reflect.*;
import java.net.*;

/**
 * Main taskdef class for the AspectJ compiler, <code>ajc</code>.
 * See the README and examples/build.xml for more information.
 */
public class Ajc extends MatchingTask {

    public final static String AJC_CLASSNAME = "org.aspectj.tools.ajc.Main";
    public final static String FALSE_MAIN = "org.aspectj.tools.ant.taskdefs.Ajc";

    protected boolean version;
    protected boolean source14;
    protected Set ignoredOptions;
    protected Commandline cmd;
    protected Commandline vmcmd;
    private int threads = -1;
    private File destdir;
    private File workingdir;
    private Path internalclasspath;
    private Path classpath;
    private Path bootclasspath;
    private Path extdirs;
    private Path srcdir;
    private List argfiles;
    private boolean fork;
    private boolean failonerror;

    public Ajc() {
        reset();
    }

    // ------------------------- options 
    // find option types and whether ignored:
    //   sed -n '/void set/p' Ajc.java | sed 's/.*\/\/ //' | sort -u
    //   sed -n '/ignoredOptions/d;/ignored/p' Ajc.java 
    // each option may be "ignored" and is one of:
    //   ajc-old             used to be an ajc option
    //   ajc-only            only an option for ajc, not javac
    //   ajc-taskdef only    only an option for ajc taskdef, not ajc
    //   javac-also          also an option in javac

    // ------------------------- options in order per ajc output

    public void setVerbose(boolean verbose) {  // javac-also docDone
        setif(verbose, "-verbose");
    }

    public void setVersion(boolean version) {  // javac-also docDone
        this.version = version;
    }
    
    public void setNocomments(boolean nocomments) { // ajc-only docDone
        setif(nocomments, "-nocomments");
    }

    public void setEmacssym(boolean input) {  // ajc-only docDone
        setif(input, "-emacssym"); 
    }

    public void setUsejavac(boolean input) {  // ajc-only docDone
        setif(input, "-usejavac"); 
    }

    public void setPreprocess(boolean preprocess) { // ajc-only docDone
        setif(preprocess, "-preprocess");
    }

    public void setWorkingdir(String workingdir) { // ajc-only docDone
        this.workingdir = project.resolveFile(workingdir);
    }

    public void setDestdir(String destdir) { // javac-also  docDone
        this.destdir = project.resolveFile(destdir);
    }

    public void setOptimize(boolean optimize) { // javac-also ignored docDone
        setif(optimize, "-O");
    }
   
    public void setClasspath(Path classpath) {  // javac-also docDone
        if (this.classpath == null) {
            this.classpath = classpath;
        } else {
            this.classpath.append(classpath);
        }
    }
    public Path createClasspath() {  // javac-also docDone
        if (classpath == null) {
            classpath = new Path(project);
        }
        return classpath.createPath();
    }        
    
    public void setClasspathref(Reference classpathref) {  // javac-also docDone
        createClasspath().setRefid(classpathref);
    }
    
    public void setBootclasspath(Path bootclasspath) {  // javac-also docDone
        if (this.bootclasspath == null) {
            this.bootclasspath = bootclasspath;
        } else {
            this.bootclasspath.append(bootclasspath);
        }
    }
    public Path createBootclasspath() {  // javac-also docDone
        if (bootclasspath == null) {
            bootclasspath = new Path(project);
        }
        return bootclasspath.createPath();
    }    
    
    public void setBootclasspathref(Reference bootclasspathref) {  // javac-also docDone
        createBootclasspath().setRefid(bootclasspathref);
    }
    
    public void setExtdirs(Path extdirs) {  // javac-also docDone
        if (this.extdirs == null) {
            this.extdirs = extdirs;
        } else {
            this.extdirs.append(extdirs);
        }
    }

    public Path createExtdirs() {  // javac-also docDone
        if (extdirs == null) {
            extdirs = new Path(project);
        }
        return extdirs.createPath();
    }  

    public void setArgfile(File argfile) { // ajc-only docDone
        if (argfiles == null) {
            argfiles = new Vector();
        }
        argfiles.add(argfile);
    }

    public void setArgfiles(String argfiles) { // ajc-only docDone
        StringTokenizer tok = new StringTokenizer(argfiles, ", ", false);
        if (tok.hasMoreTokens() && this.argfiles == null) {
            this.argfiles = new Vector();
        }
        while (tok.hasMoreTokens()) {
            this.argfiles.add(project.resolveFile(tok.nextToken().trim()));
        }
    }

    public void setEncoding(String encoding) { // ignored docDone
        ignore("encoding");
    }

    public void setSource(String input) {    // javac-also (Ant 1.4) docDone
        source14 = "1.4".equals(input);
    }

    public void setLenient(boolean input) {  // ajc-only docDone
        setif(input, "-lenient"); 
    }

    public void setStrict(boolean input) {  // ajc-only docDone
        setif(input, "-strict"); 
    }

    public void setPorting(boolean input) {  // ajc-only docDone
        setif(input, "-porting"); 
    }

    public void setX(String input) {  // ajc-only docDone
        StringTokenizer tokens = new StringTokenizer(input, ",", false);
        while (tokens.hasMoreTokens()) {
            String token = tokens.nextToken().trim();
            if (1 < token.length()) {
                setif(true, "-X" + token); 
            }
        }
    }

    // ------------------------- vestigial
    public void setThreads(int threads) { // ajc-old docDone ignored
        this.threads = threads;
    }
    
    public void setDumpstack(boolean dumpstack) { // ajc-old
        setif(dumpstack, "-dumpstack");
    }

    // ------------------------- specific to taskdef

    public void setInternalclasspath(Path internalclasspath) { // ajc-taskdef only
        if (this.internalclasspath == null) {
            this.internalclasspath = internalclasspath;
        } else {
            this.internalclasspath.append(internalclasspath);
        }
    }
    public Path createInternalclasspath() { // ajc-taskdef only
        if (internalclasspath == null) {
            internalclasspath = new Path(project);
        }
        return internalclasspath.createPath();
    }        
    
    public void setInternalclasspathref(Reference internalclasspathref) { // ajc-taskdef only
        createInternalclasspath().setRefid(internalclasspathref);
    }

     public void setSrcdir(Path srcdir) { // javac-also docDone
        if (this.srcdir == null) {
            this.srcdir = srcdir;
        } else {
            this.srcdir.append(srcdir);
        }
    }

    public Path createSrc() { // javac-also docDone
        return createSrcdir();
    }

    public Path createSrcdir() { // javac-also docDone
        if (srcdir == null) {
            srcdir = new Path(project);
        }
        return srcdir.createPath();
    }

    public void setFork(boolean fork) { // ajc-only docDone
        this.fork = fork;
    }

    public void setFailonerror(boolean failonerror) {  // javac-also docDone
        this.failonerror = failonerror;
    }

    public void setMaxmemory(String max) { // ajc-taskdef only docDone
        // we do not run under 1.1 anyway...
        createJvmarg().setValue((Project.getJavaVersion().
                                 startsWith("1.1") ?
                                 "-mx" : "-Xmx") +max);
    }

    public void setJvmarg(String input) {  // ajc-taskdef only docDone
        vmcmd.createArgument().setValue(input);
    }

    public Commandline.Argument createJvmarg() { // ajc-taskdef only docDone
        return vmcmd.createArgument();
    }

    // ------------------------- javac task compatibility
    public void setNosymbols(boolean nosymbols) { // todo remove?
        setif(nosymbols, "-nosymbols");
    }
    
    public void setDebug(boolean debug) { // javac-also docDone
        setif(debug, "-g"); // todo: changed from -debug
    }
    
    public void setDeprecation(boolean deprecation) { // javac-also docDone
        setif(deprecation, "-deprecation");
    }

    // ------------------------- javac task compatibility - ignored
    public void setTarget(String target) {  // javac-also ignored docDone
        ignore("target"); // todo: ajc accepts but doesn't use - pass in?
    }
    public void setDepend(String depend) {  // javac-also ignored docDone
        ignore("depend");
    }
    public void setIncludeantruntime(boolean includeAntruntime) {  // javac-also ignored docDone
        ignore("includeantruntime");
    }
    public void setIncludejavaruntime(boolean includeJavaruntime ) {  // javac-also ignored docDone
        ignore("includeJavaruntime");
    }

    // ------------------------- other state methods
    protected final void ignore(String attribute) {
        ignoredOptions.add(attribute);
    }

    public void backdoorSetFile(File file) {
        if (null != file) {
            cmd.createArgument().setFile(file);
        }
    }
    
    /** reset variables to permit gc */
    public void reset() {
        cmd = new Commandline();
        vmcmd = new Commandline();
        threads = -1;
        destdir = null;
        workingdir = null;
        internalclasspath = null;
        classpath = null;
        bootclasspath = null;
        extdirs = null;
        srcdir = null;
        argfiles = null;
        fork = false;
        failonerror = true;
        version = false;
        ignoredOptions = new HashSet();
    }

    protected final void setif(boolean b, String flag) {
        if (b) cmd.createArgument().setValue(flag);
    }

    // ------------------------- operational methods

    public void execute() throws BuildException {
        if (srcdir == null && argfiles == null) {
            throw new BuildException("one of srcdir or argfiles must be set!",
                                     location);
        }
        if (threads != -1) {
            cmd.createArgument().setValue("-threads");
            cmd.createArgument().setValue(threads+"");
        }
        if (workingdir != null) {
            cmd.createArgument().setValue("-workingdir");
            cmd.createArgument().setFile(workingdir);
        }
        if (destdir != null) {
            cmd.createArgument().setValue("-d");
            cmd.createArgument().setFile(destdir);
        }
        if (classpath != null) {
            cmd.createArgument().setValue("-classpath");
            cmd.createArgument().setPath(classpath);
        }
        if (bootclasspath != null) {
            cmd.createArgument().setValue("-bootclasspath");
            cmd.createArgument().setPath(bootclasspath);
        }
        if (extdirs != null) {
            cmd.createArgument().setValue("-extdirs");
            cmd.createArgument().setPath(extdirs);
        }
        if (source14) {
            cmd.createArgument().setValue("-source");
            cmd.createArgument().setValue("1.4");
        }
        int numargfiles = 0;
        if (argfiles != null) {
            for (Iterator i = argfiles.iterator(); i.hasNext();) {
                String name = i.next()+"";
                File argfile = project.resolveFile(name);
                if (check(argfile, name, false, location)) {
                    cmd.createArgument().setValue("-argfile");
                    cmd.createArgument().setFile(argfile);
                    numargfiles++;    
                }
            }
        }
        int numfiles = 0;
        if (srcdir != null) {
            // todo: ignore any srcdir if any argfiles and no explicit includes
            String[] dirs = srcdir.list();
            for (int i = 0; i < dirs.length; i++) {
                File dir = project.resolveFile(dirs[i]);
                check(dir, dirs[i], true, location);
                String[] files = getDirectoryScanner(dir).getIncludedFiles();
                for (int j = 0; j < files.length; j++) {
                    File file = new File(dir, files[j]);
                    if (file.getName().endsWith(".java")
                        || file.getName().endsWith(".aj")) {
                        cmd.createArgument().setFile(file);
                        numfiles++;
                    }
                }
            }
        }
        if ((null != ignoredOptions) && (ignoredOptions.size() > 0)) {
            log("The following attributes were ignored " + ignoredOptions,
                Project.MSG_WARN);
        }
        log("Compiling " + numfiles + " source and " +
            + numargfiles + " arg files" 
            + (null == destdir ? "" : " to " + destdir.getPath()));

         // here is the actual invocation
        int result = (fork || (internalclasspath != null)) ? fork() : spoon();
        if (result != 0) {
            String msg = "Compilation error: " + result;
            if (failonerror) {
                reset();
                throw new BuildException(msg);
            } else {
                log(msg, Project.MSG_WARN);
            }
        }
        reset(); // see throw above
    }

    protected void version(Path classpath) {
        try {
            Class main = findClass("org.aspectj.tools.ajc.Main",
                                   classpath);
            Method printVersion = main.getDeclaredMethod("printVersion",
                                                         new Class[]{});
            printVersion.setAccessible(true);
            printVersion.invoke(main.newInstance(), new Object[]{});
        } catch (Exception e) {}
    }

    protected Class findClass(String classname, Path classpathPath) {
        String classpath = (classpathPath != null ?
                            classpathPath+"" : "").trim();
        if (classpath.length() == 0) {
            try {
                return Class.forName(classname);
            } catch (ClassNotFoundException e){}
        }
        List urls = new ArrayList();
        for (StringTokenizer t = new StringTokenizer(classpath,
                                                     File.pathSeparator);
             t.hasMoreTokens();) {
            File f = new File(t.nextToken().trim());
            try {
                if (f.exists()) {
                    URL url = f.toURL();
                    if (url != null) urls.add(url);
                }
            } catch (MalformedURLException e) {}
        }
        if (urls.size() == 0) return null;
        try {
            return new URLClassLoader((URL[])urls.toArray
                                      (new URL[urls.size()]),
                                      null).loadClass(classname);
        } catch (ClassNotFoundException e) {}
        return null;
    }

    protected int fork() throws BuildException {
        Java java = (Java)project.createTask("java");
        java.setTaskName(getTaskName());
        Path compileClasspath;
        if (internalclasspath != null) {
            compileClasspath = internalclasspath;
            compileClasspath.append(Path.systemClasspath);
        } else {
            compileClasspath = Path.systemClasspath;
        }
        if (version) version(compileClasspath);
        java.setClasspath(compileClasspath);
        java.setClassname(FALSE_MAIN);
        String[] args;
        args = cmd.getArguments();
        for (int i = 0; i < args.length; i++) {
            java.createArg().setValue(args[i]);
        }
        args = vmcmd.getArguments();
        for (int i = 0; i < args.length; i++) {
            java.createJvmarg().setValue(args[i]);
        }
        java.setFork(fork);
        // java handles its own verbose logging when forking
        return java.executeJava();
    }

    /** utility to render String[] for logging */
    public static String render(String[] args) {
        if (null == args) return "";
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < args.length; i++) {
            sb.append(args[i]);
            sb.append(" ");
        }
        return sb.toString();
    }

    protected int spoon() throws BuildException {
        if (version) version(null);
        try {
            log("Running  in-process " + AJC_CLASSNAME + " using " 
                + Ajc.render(cmd.getCommandline()), Project.MSG_VERBOSE);
            Object main =
                Class.forName(AJC_CLASSNAME).newInstance();
            return ((Integer)main.getClass().getMethod
                    ("compile", new Class[]{String[].class}).
                    invoke(main, new Object[]{cmd.getCommandline()})).intValue();
        } catch (Throwable t) {
            t.printStackTrace();
            throw new BuildException("Couldn't create compiler!",
                                     t, location);
        }
    }

    protected final boolean check(File file, String name,
                                  boolean isDir, Location loc) {
        loc = loc != null ? loc : location;
        if (file == null) {
            throw new BuildException(name + " is null!", loc);
        }
        if (!file.exists()) {
            throw new BuildException(file + "doesn't exist!", loc);
        }
        if (isDir ^ file.isDirectory()) {
            String e = file + " should" + (isDir ? "" : "n't")  +
                " be a directory!";
            throw new BuildException(e, loc);
        }
        return true;
    }

    public static void main(String[] args) {
        try {
            Class main = Class.forName(Ajc.AJC_CLASSNAME);
            main.getMethod
                ("compile", new Class[]{String[].class}).
                invoke(main.newInstance(), new Object[]{args});
        } catch (Throwable t) { t.printStackTrace(); }
    }
}
