/*
 * Copyright (C) 2009 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors:
 *    Cody Russell <crussell@canonical.com>
 */

#define _GNU_SOURCE
#include <unistd.h>

#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <pwd.h>
#include <dirent.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkx.h>

#include <X11/Xlib.h>
#include <X11/extensions/Xcomposite.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>
#include <dbus/dbus-glib-lowlevel.h>

#include "gtytimeline.h"
#include "xsplash.h"
#include "log.h"

typedef struct _AnimContext          AnimContext;
typedef struct _XsplashServerPrivate XsplashServerPrivate;

struct _AnimContext
{
  XsplashServer *server;
  GtkWindow     *window;
  GtyTimeline   *timeline;
  gboolean       starting_opaque;
  gboolean       exit_on_finish;
};

struct _XsplashServerPrivate
{
  GtkWidget       *window;
  GtkWidget       *logo;
  GtkWidget       *fixed;
  GtkWidget       *throbber;
  GtkWidget       *background;
  GdkPixbuf       *logo_pixbuf;
  GdkPixbuf       *throbber_pixbuf;

  gchar           *dbusobject;
  DBusGConnection *system_bus;
  DBusGProxy      *bus_proxy;

  GdkWindow       *cow;
  GdkScreen       *screen;
};

enum {
  PROP_0,
  PROP_DBUS_OBJECT,
  PROP_SYSTEM_BUS,
  PROP_DBUS_PROXY
};

static void               start_daemon_mode              (void);
static AnimContext *      anim_context_new               (XsplashServer  *server,
                                                          GtyTimeline    *timeline,
                                                          gpointer        id);
static void               anim_context_free              (AnimContext    *context);
static void               fade_frame_cb                  (GtyTimeline    *timeline,
                                                          gdouble         progress,
                                                          gpointer        user_data);
static void               fade_finished_cb               (GtyTimeline    *timeline,
                                                          gpointer        user_data);
static void               set_property                   (GObject        *obj,
                                                          guint           id,
                                                          const GValue   *value,
                                                          GParamSpec     *pspec);
static void               get_property                   (GObject        *obj,
                                                          guint           id,
                                                          GValue         *value,
                                                          GParamSpec     *pspec);


static void               window_realized                (GtkWidget      *window,
                                                          gpointer        user_data);
static gboolean           key_press_event                (GtkWidget      *widget,
                                                          GdkEventKey    *event,
                                                          gpointer        user_data);

static void               xsplash_server_dispose         (GObject        *object);

gboolean                  xsplash_server_add_wait_signal (XsplashServer  *server,
                                                          gchar          *waitfor,
                                                          GError        **error);
gboolean                  xsplash_server_signal_loaded   (XsplashServer  *server,
                                                          gchar          *app,
                                                          GError        **error);
static void               start_throbber                 (XsplashServer  *server);
static void               throbber_frame_cb              (GtyTimeline    *timeline,
                                                          gdouble         progress,
                                                          gpointer        user_data);

static void               add_signal                     (gchar *name);
static gboolean           xsplash_add_signal             (const gchar    *option_name,
                                                          const gchar    *value,
                                                          gpointer        data,
                                                          GError        **error);

#define XSPLASH_DBUS_NAME   "com.ubuntu.BootCurtain"
#define XSPLASH_DBUS_OBJECT "/com/ubuntu/BootCurtain"

static gboolean  gdm_session      = FALSE;
static gchar    *background_image = NULL;
static gchar    *logo_image       = NULL;
static gchar    *throbber_image   = NULL;
static guint     throbber_frames  = 50;
static gboolean  ping_pong        = FALSE;
static gboolean  have_xcomposite  = FALSE;
static gboolean  daemon_mode      = FALSE;
static gboolean  is_composited    = FALSE;
static gboolean  redirected       = FALSE;
static GSList   *signal_list      = NULL;
static guint     timeout          = 15;

static GOptionEntry entries[] = {
  {
    "gdm-session", 'g', 0,
    G_OPTION_ARG_NONE, &gdm_session,
    "Run in gdm session", NULL
  },
  {
    "background", 'b', 0,
    G_OPTION_ARG_FILENAME, &background_image,
    "Filename for background image", NULL
  },
  {
    "daemon", 'd', 0,
    G_OPTION_ARG_NONE, &daemon_mode,
    "Run in daemon mode", NULL
  },
  {
    "logo", 'l', 0,
    G_OPTION_ARG_FILENAME, &logo_image,
    "Filename for logo image", NULL
  },
  {
    "throbber", 't', 0,
    G_OPTION_ARG_FILENAME, &throbber_image,
    "Filename for throbber image", NULL
  },
  {
    "frames", 'f', 0,
    G_OPTION_ARG_INT, &throbber_frames,
    "Number of frames for the throbber (default 50)", NULL
  },
  {
    "pingpong", 'p', 0,
    G_OPTION_ARG_NONE, &ping_pong,
    "Whether to reverse throbber directions", NULL
  },
  {
    "timeout", 'x', 0,
    G_OPTION_ARG_INT, &timeout,
    "Timeout (in seconds - 15s by default) ", NULL
  },
  {
    "add-signal", 's', 0,
    G_OPTION_ARG_CALLBACK, (GOptionArgFunc) xsplash_add_signal,
    "Add a signal to listen for.", NULL
  },
  { NULL }
};

#include "dbus-xsplash-server.h"

G_DEFINE_TYPE (XsplashServer, xsplash_server, G_TYPE_OBJECT);
#define XSPLASH_SERVER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), XSPLASH_TYPE_SERVER, XsplashServerPrivate))

static gboolean fading = FALSE;

static void
xsplash_server_class_init (XsplashServerClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS (class);

  g_type_class_add_private (class, sizeof (XsplashServerPrivate));

  object_class->dispose      = xsplash_server_dispose;
  object_class->set_property = set_property;
  object_class->get_property = get_property;

  g_object_class_install_property (object_class, PROP_DBUS_OBJECT,
                                   g_param_spec_string ("dbus-object", "DBus object path",
                                                        "The object that represents this set of menus on DBus",
                                                        XSPLASH_DBUS_OBJECT,
                                                        G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_STRINGS));

  g_object_class_install_property (object_class, PROP_SYSTEM_BUS,
                                   g_param_spec_pointer ("system-bus", "DBus system bus",
                                                         "The system bus",
                                                         G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (object_class, PROP_DBUS_PROXY,
                                   g_param_spec_object ("dbus-proxy", "DBus proxy",
                                                        "The dbus proxy",
                                                        DBUS_TYPE_G_PROXY,
                                                        G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));

  dbus_g_object_type_install_info (XSPLASH_TYPE_SERVER,
                                   &dbus_glib_xsplash_server_object_info);
}

static void
xsplash_server_dispose (GObject *object)
{
  XsplashServerPrivate *priv = XSPLASH_SERVER_GET_PRIVATE (object);

  if (have_xcomposite && priv->cow)
    {
      XCompositeReleaseOverlayWindow (GDK_DISPLAY_XDISPLAY (gdk_display_get_default ()),
                                      GDK_DRAWABLE_XID (GDK_WINDOW_ROOT));
      g_object_unref (priv->cow);
    }

  if (priv->throbber_pixbuf)
    g_object_unref (priv->throbber_pixbuf);

  if (priv->window)
    g_object_unref (priv->window);
}

static void
set_property (GObject *obj, guint id, const GValue *value, GParamSpec *pspec)
{
  XsplashServerPrivate *priv = XSPLASH_SERVER_GET_PRIVATE (obj);

  switch (id) {
  case PROP_DBUS_OBJECT:
    g_return_if_fail (priv->dbusobject == NULL);

    priv->dbusobject = g_value_dup_string (value);

    DBusGConnection *connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, NULL);

    dbus_g_connection_register_g_object (connection,
                                         priv->dbusobject,
                                         obj);
    break;

  case PROP_SYSTEM_BUS:
    priv->system_bus = g_value_get_pointer (value);
    break;

  case PROP_DBUS_PROXY:
    priv->bus_proxy = g_value_get_object (value);
    break;

  default:
    g_return_if_reached ();
    break;
  }
}

static void
get_property (GObject *obj, guint id, GValue *value, GParamSpec *pspec)
{
  XsplashServerPrivate *priv = XSPLASH_SERVER_GET_PRIVATE (obj);

  switch (id) {
  case PROP_DBUS_OBJECT:
    g_value_set_string (value, priv->dbusobject);
    break;

  case PROP_SYSTEM_BUS:
    g_value_set_pointer (value, priv->system_bus);
    break;

  case PROP_DBUS_PROXY:
    g_value_set_object (value, priv->bus_proxy);
    break;

  default:
    g_return_if_reached ();
    break;
  }
}

static GdkPixbuf *
scale_to_min (GdkPixbuf *src, gint min_width, gint min_height)
{
  double factor;
  int src_width, src_height;
  int new_width, new_height;

  src_width = gdk_pixbuf_get_width (src);
  src_height = gdk_pixbuf_get_height (src);

  factor = MAX (min_width / (double)src_width, min_height / (double)src_height);

  new_width = floor (src_width * factor + 0.5);
  new_height = floor (src_height * factor + 0.5);

  return gdk_pixbuf_scale_simple (src, new_width, new_height, GDK_INTERP_BILINEAR);
}

static gint
get_monitor_width ()
{
  GdkScreen *screen;
  GdkRectangle rect;

  screen = gdk_screen_get_default ();

  gdk_screen_get_monitor_geometry (screen,
                                   0,
                                   &rect);

  return rect.width;
}

static gint
get_monitor_height ()
{
  GdkScreen *screen;
  GdkRectangle rect;

  screen = gdk_screen_get_default ();

  gdk_screen_get_monitor_geometry (screen,
                                   0,
                                   &rect);

  return rect.height;
}

static gchar *
get_background_filename (void)
{
  return g_strdup (DATADIR "/images/xsplash/bg_2560x1600.jpg");
}

static const gchar *
get_filename_size_modifier (gint width)
{
  if (width < 1280)
    return "small";
  else if (width < 1600)
    return "medium";
  else if (width < 2560)
    return "large";
  else
    return "xtra_large";
}

static gchar *
get_throbber_filename (void)
{
  gchar *ret;
  gint width;

  width = get_monitor_width ();

  if (throbber_image != NULL)
    {
      g_message ("get_throbber_filename(): user provided a throbber on the command line; using that");

      return g_strdup (throbber_image);
    }

  g_message ("get_throbber_filename(): looking for the best throbber for screen width...");

  ret = g_strdup_printf (DATADIR "/images/xsplash/throbber_%s.png",
                         get_filename_size_modifier (width));

  g_message (" ** Chose `%s'", get_filename_size_modifier (width));
  g_message (" ** throbber filename is: %s", ret);

  return ret;
}

static gchar *
get_logo_filename ()
{
  gchar *ret;
  gint width;

  width = get_monitor_width ();

  if (logo_image != NULL)
    {
      g_message ("get_logo_filename(): user provided a logo on the command line; using that");

      return g_strdup (logo_image);
    }

  g_message ("get_logo_filename(): looking for the best logo for screen width...");

  ret = g_strdup_printf (DATADIR "/images/xsplash/logo_%s.png",
                         get_filename_size_modifier (width));

  g_message (" ** Chose `%s'", get_filename_size_modifier (width));
  g_message (" ** logo filename is: %s", ret);

  return ret;
}

static GdkPixbuf *
get_pixbuf (gint width, gint height)
{
  GdkPixbuf *pixbuf;
  GdkPixbuf *scaled;
  gint x, y, w, h;

  pixbuf = gdk_pixbuf_new_from_file (background_image, NULL);
  scaled = scale_to_min (pixbuf,
                         width,
                         height);

  w = gdk_pixbuf_get_width (scaled);
  h = gdk_pixbuf_get_height (scaled);

  x = (width - w) / 2;
  y = (height - h) / 2;

  x = ABS(x);
  y = ABS(y);

  return gdk_pixbuf_new_subpixbuf (scaled,
                                   x,
                                   y,
                                   w - (2 * x),
                                   h - (2 * y));
}

static gboolean
focus_out_event (GtkWidget     *widget,
                 GdkEventFocus *event,
                 gpointer       user_data)
{
  gtk_window_present (GTK_WINDOW (widget));

  return FALSE;
}

static void
composited_changed (GdkScreen *screen,
                    gpointer data)
{
  XsplashServer *server = (XsplashServer *)data;
  XsplashServerPrivate *priv = XSPLASH_SERVER_GET_PRIVATE (server);

  is_composited = gdk_screen_is_composited (screen);

  if (GTK_WIDGET_REALIZED (priv->window))
    {
      if (is_composited)
        {
          if (redirected)
            {
              gdk_window_remove_redirection (priv->window->window);
            }

          gtk_widget_set_colormap (priv->window,
                                   gdk_screen_get_rgba_colormap (priv->screen));
        }
    }

  gtk_window_present (GTK_WINDOW (priv->window));
}

static void
setup_background_image (XsplashServer *server)
{
  XsplashServerPrivate *priv = XSPLASH_SERVER_GET_PRIVATE (server);
  GdkPixmap *pixmap;
  GdkPixbuf *pixbuf;

  pixbuf = get_pixbuf (get_monitor_width (),
                       get_monitor_height ());

  pixmap = gdk_pixmap_new (priv->cow,
                           gdk_pixbuf_get_width (pixbuf),
                           gdk_pixbuf_get_height (pixbuf),
                           -1);

  gdk_draw_pixbuf (pixmap,
                   NULL,
                   pixbuf,
                   0, 0,
                   0, 0,
                   -1, -1,
                   GDK_RGB_DITHER_MAX,
                   0, 0);

  if (!priv->background)
    {
      priv->background = gtk_image_new_from_pixmap (pixmap,
                                                    NULL);
    }
  else
    {
      gtk_image_set_from_pixmap (GTK_IMAGE (priv->background),
                                 pixmap, NULL);
    }
}

static void
setup_logo_image (XsplashServer *server)
{
  XsplashServerPrivate *priv = XSPLASH_SERVER_GET_PRIVATE (server);

  if (!priv->logo)
    {
      gchar *logo_filename = NULL;

      logo_filename = get_logo_filename (get_monitor_width ());
      priv->logo_pixbuf = gdk_pixbuf_new_from_file (logo_filename, NULL);

      priv->logo = gtk_image_new_from_pixbuf (priv->logo_pixbuf);

      gtk_fixed_put (GTK_FIXED (priv->fixed),
                     priv->logo,
                     get_monitor_width () / 2 - gdk_pixbuf_get_width (priv->logo_pixbuf) / 2,
                     get_monitor_height () / 3 - gdk_pixbuf_get_height (priv->logo_pixbuf) / 2);

      if (logo_filename != NULL)
        g_free (logo_filename);
    }
  else
    {
      gtk_fixed_move (GTK_FIXED (priv->fixed),
                      priv->logo,
                      get_monitor_width () / 2 - gdk_pixbuf_get_width (priv->logo_pixbuf) / 2,
                      get_monitor_height () / 3 - gdk_pixbuf_get_height (priv->logo_pixbuf) / 2);
    }
}

static void
setup_throbber_image (XsplashServer *server)
{
  XsplashServerPrivate *priv = XSPLASH_SERVER_GET_PRIVATE (server);

  if (!priv->throbber)
    {
      gchar *throbber_filename = NULL;

      throbber_filename = get_throbber_filename ();

      if (throbber_filename && throbber_frames)
        {
          priv->throbber_pixbuf = gdk_pixbuf_new_from_file (throbber_filename, NULL);

          if (priv->throbber_pixbuf != NULL)
            {
              priv->throbber = gtk_image_new ();
              gtk_widget_show (priv->throbber);

              gtk_fixed_put (GTK_FIXED (priv->fixed),
                             priv->throbber,
                             get_monitor_width () / 2 - gdk_pixbuf_get_width (priv->throbber_pixbuf) / 2,
                             get_monitor_height () / 3 + gdk_pixbuf_get_height (priv->logo_pixbuf) / 2 + gdk_pixbuf_get_height (priv->throbber_pixbuf) / ((throbber_frames - 1) * 2));

              start_throbber (server);
            }
          else
            {
              g_message ("couldn't load throbber image from file (%s); "
                         "disabling throbber", throbber_image);
            }
        }

      if (throbber_filename)
        g_free (throbber_filename);
    }
  else
    {
      gtk_fixed_move (GTK_FIXED (priv->fixed),
                      priv->throbber,
                      get_monitor_width () / 2 - gdk_pixbuf_get_width (priv->throbber_pixbuf) / 2,
                      get_monitor_height () / 3 + gdk_pixbuf_get_height (priv->logo_pixbuf) / 2 + gdk_pixbuf_get_height (priv->throbber_pixbuf) / ((throbber_frames - 1) * 2));
    }
}

static void
image_size_changed (GdkScreen *screen,
                  gpointer   user_data)
{
  XsplashServer *server = (XsplashServer *)user_data;

  setup_background_image (server);
  setup_logo_image (server);
  setup_throbber_image (server);

  g_message ("**** monitors changed");
}

static void
xsplash_server_init (XsplashServer *server)
{
  XsplashServerPrivate *priv = XSPLASH_SERVER_GET_PRIVATE (server);

  priv->dbusobject = NULL;
  priv->system_bus = NULL;
  priv->bus_proxy = NULL;

  if (gdk_display_supports_composite (gdk_display_get_default ()))
    {
      have_xcomposite = TRUE;
    }

  if (have_xcomposite)
    {
      Window window;
      GdkCursor *cursor;

      gdk_error_trap_push ();

      window = XCompositeGetOverlayWindow (GDK_DISPLAY_XDISPLAY (gdk_display_get_default ()),
                                           GDK_DRAWABLE_XID (gdk_get_default_root_window ()));
      priv->cow = gdk_window_foreign_new (window);

      cursor = gdk_cursor_new (GDK_BLANK_CURSOR);
      gdk_window_set_cursor (priv->cow,
                             cursor);
      gdk_cursor_unref (cursor);

      gdk_flush ();
      gdk_error_trap_pop ();
    }

  priv->window = g_object_new (gtk_window_get_type (),
                               "name", "xsplash-window",
                               "type-hint", GDK_WINDOW_TYPE_HINT_DOCK,
                               NULL);

  //gtk_window_fullscreen (GTK_WINDOW (priv->window));
  gtk_window_set_keep_above (GTK_WINDOW (priv->window), TRUE);
  gtk_window_set_decorated (GTK_WINDOW (priv->window), FALSE);

  priv->screen = gtk_widget_get_screen (priv->window);
  g_signal_connect (G_OBJECT (priv->screen),
                    "composited-changed",
                    G_CALLBACK (composited_changed),
                    server);
  g_signal_connect (G_OBJECT (priv->screen),
                    "monitors-changed",
                    G_CALLBACK (image_size_changed),
                    server);
  g_signal_connect (G_OBJECT (priv->screen),
                    "size-changed",
                    G_CALLBACK (image_size_changed),
                    server);

  g_signal_connect (priv->window,
                    "realize",
                    G_CALLBACK (window_realized),
                    server);
  g_signal_connect (priv->window,
                    "key-press-event",
                    G_CALLBACK (key_press_event),
                    server);
  g_signal_connect (priv->window,
                    "focus-out-event",
                    G_CALLBACK (focus_out_event),
                    server);

  priv->fixed = gtk_fixed_new ();

  setup_background_image (server);

  gtk_container_add (GTK_CONTAINER (priv->fixed),
                     priv->background);

  setup_logo_image (server);
  setup_throbber_image (server);

  gtk_container_add (GTK_CONTAINER (priv->window), priv->fixed);

  start_daemon_mode ();

  gtk_widget_show_all (priv->window);
  gtk_window_present (GTK_WINDOW (priv->window));
}

static void
throbber_frame_cb (GtyTimeline *timeline,
                   gdouble      progress,
                   gpointer     user_data)
{
  XsplashServer *server;
  XsplashServerPrivate *priv;
  GdkPixbuf *pixbuf;
  gint frame;
  gint height, y_offset;

  server = (XsplashServer *)user_data;
  priv = XSPLASH_SERVER_GET_PRIVATE (server);
  frame = MIN (throbber_frames * progress, throbber_frames - 1);

  height = gdk_pixbuf_get_height (priv->throbber_pixbuf) / throbber_frames;
  y_offset = height * (frame);

  pixbuf = gdk_pixbuf_new_subpixbuf (priv->throbber_pixbuf,
                                     0, y_offset,
                                     gdk_pixbuf_get_width (priv->throbber_pixbuf), height);

  gtk_image_set_from_pixbuf (GTK_IMAGE (priv->throbber),
                             pixbuf);

  if (ping_pong)
    {
      if (progress == 1.0 || progress == 0.0)
        {
          gty_timeline_set_direction (timeline,
                                      !gty_timeline_get_direction (timeline));
        }
    }

  gdk_window_raise (priv->window->window);
}

static void
start_throbber (XsplashServer *server)
{
  XsplashServerPrivate *priv;
  GtyTimeline *timeline;

  priv = XSPLASH_SERVER_GET_PRIVATE (server);
  timeline = gty_timeline_new (1000);
  gty_timeline_set_loop (timeline, TRUE);

  g_signal_connect (timeline,
                    "frame",
                    G_CALLBACK (throbber_frame_cb),
                    server);

  gty_timeline_start (timeline);

  g_message ("throbber started (%d frames)", throbber_frames);
}

static void
begin_fade (XsplashServer *server, gboolean exit_on_finish)
{
  XsplashServerPrivate *priv;
  GtyTimeline *timeline;
  AnimContext *context;

  if (!fading)
    {
      priv = XSPLASH_SERVER_GET_PRIVATE (server);

      /* The throbber can draw incorrectly while we're fading out under
       * some conditions, so let's just hide it first. */
      gtk_widget_hide (priv->throbber);

      timeline = gty_timeline_new (800);
      context = anim_context_new (server,
                                  timeline,
                                  "fade");
      context->exit_on_finish = exit_on_finish;

      g_signal_connect (timeline,
                        "frame",
                        G_CALLBACK (fade_frame_cb),
                        context);
      g_signal_connect (timeline,
                        "finished",
                        G_CALLBACK (fade_finished_cb),
                        context);
      gty_timeline_start (timeline);

      fading = TRUE;
    }
}

static gboolean
key_press_event (GtkWidget   *widget,
                 GdkEventKey *event,
                 gpointer     user_data)
{
  XsplashServer *server = (XsplashServer *)user_data;

  switch (event->keyval)
    {
    case GDK_Escape:
      g_message ("ESC interrupt");
      begin_fade (server, TRUE);

      /* else queue another fade? */
      break;

    default:
      break;
    }

  return FALSE;
}

static AnimContext *
anim_context_new (XsplashServer *server,
                  GtyTimeline   *timeline,
                  gpointer       id)
{
  AnimContext *context = g_slice_new (AnimContext);
  XsplashServerPrivate *priv = XSPLASH_SERVER_GET_PRIVATE (server);

  context->server          = server;
  context->window          = GTK_WINDOW (priv->window);
  context->timeline        = timeline;
  context->starting_opaque = (gtk_window_get_opacity (context->window) == 1.0);

  return context;
}

static void
anim_context_free (AnimContext *context)
{
  g_object_unref (context->timeline);

  g_slice_free (AnimContext, context);
}

static void
fade_frame_cb (GtyTimeline *timeline,
               gdouble      progress,
               gpointer     user_data)
{
  AnimContext *context = user_data;
  gdouble opacity;

  if (context->starting_opaque)
    {
      opacity = 1.0 - progress;
    }
  else
    {
      opacity = progress;
    }

  gtk_window_set_opacity (context->window, opacity);
}

static void
fade_finished_cb (GtyTimeline *timeline,
                  gpointer     user_data)
{
  AnimContext *context = user_data;

  anim_context_free (context);

  g_message ("fade finished");

  if (context->exit_on_finish)
    {
      gtk_main_quit ();
    }
}

static void
window_realized (GtkWidget *window,
                 gpointer   user_data)
{
  XsplashServer *server = (XsplashServer *)user_data;
  XsplashServerPrivate *priv = XSPLASH_SERVER_GET_PRIVATE (server);
  GdkCursor *cursor = gdk_cursor_new (GDK_BLANK_CURSOR);

  gdk_window_set_cursor (window->window,
                         cursor);
  gdk_cursor_unref (cursor);

  if (have_xcomposite && !is_composited)
    {
      gdk_window_redirect_to_drawable (window->window,
                                       priv->cow,
                                       0, 0,
                                       0, 0,
                                       window->allocation.width,
                                       window->allocation.height);
      redirected = TRUE;
    }
}

gboolean
temporary_hack_for_initial_fade (gpointer user_data)
{
  XsplashServer *server = (XsplashServer *)user_data;

  g_message ("** timeout **");

  begin_fade (server, TRUE);

  return FALSE;
}

static void
add_signal (gchar *name)
{
  GSList *tmp = NULL;

  if ((strcmp (name, ".") == 0) || strcmp (name, "..") == 0)
    return;

  g_message ("adding signal `%s'", name);

  for (tmp = signal_list; tmp != NULL; tmp = g_slist_next (tmp))
    {
      if (strcmp (tmp->data, name) == 0)
        {
          g_message ("   ** that signal is already being listened for; skipping");
          return;
        }
    }

  signal_list = g_slist_prepend (signal_list, g_strdup (name));
}

static gboolean
xsplash_add_signal (const gchar  *option_name,
                    const gchar  *value,
                    gpointer      data,
                    GError      **error)
{
  add_signal (g_strdup (value));

  return TRUE;
}

void
sig_handler (int signum)
{
  g_message ("**  interrupted **");
  gtk_main_quit ();
}

static void
start_daemon_mode (void)
{
  pid_t pid, sid;

  if (daemon_mode)
    {
      /* fork from our parent process */
      pid = fork ();

      if (pid < 0)
        exit (EXIT_FAILURE);

      if (pid > 0)
        exit (EXIT_SUCCESS);

      sid = setsid ();
      if (sid < 0)
        exit (EXIT_FAILURE);
    }
}

int
main (int argc, char *argv[])
{
  XsplashServer   *server;
  GError          *error = NULL;
  GOptionContext  *context;
  DBusGConnection *system_bus;
  DBusGProxy      *bus_proxy;
  guint            nameret = 0;
  gchar           *log_filename = NULL;
  struct passwd   *pwd;

  pwd = getpwnam (XSPLASH_USER);
  if (pwd == NULL)
    {
      g_error ("Unknown user: %s", XSPLASH_USER);
      return 1;
    }

  if (setresgid (pwd->pw_gid, pwd->pw_gid, pwd->pw_gid))
    {
      g_error ("Failed to setresgid to `%s'", XSPLASH_USER);
      return 1;
    }

  if (setresuid (pwd->pw_uid, pwd->pw_uid, pwd->pw_uid))
    {
      g_error ("Failed to setresuid to `%s'", XSPLASH_USER);
      return 1;
    }

  context = g_option_context_new ("xsplash");
  g_option_context_add_main_entries (context,
                                     entries,
                                     "xsplash");
  g_option_context_add_group (context, gtk_get_option_group (TRUE));
  if (!g_option_context_parse (context, &argc, &argv, &error))
    {
      exit (1);
    }

  log_filename = g_strdup_printf ("/var/log/gdm/xsplash-%s.log", gdm_session ? "gdm" : "user");
  log_init (log_filename); g_message ("xsplash started");
  g_free (log_filename);

  gtk_init (&argc, &argv);

  signal (SIGHUP, sig_handler);
  signal (SIGINT, sig_handler);
  signal (SIGTERM, sig_handler);

  if (background_image == NULL)
    background_image = get_background_filename ();

  if (logo_image == NULL)
    logo_image = get_logo_filename ();

  if (throbber_image == NULL)
    throbber_image = get_throbber_filename ();

  g_message ("background_image = %s", background_image);
  g_message ("logo_image = %s", logo_image);
  g_message ("throbber_image = %s", throbber_image);

  system_bus = dbus_g_bus_get (DBUS_BUS_SYSTEM, NULL);

  bus_proxy = dbus_g_proxy_new_for_name (system_bus,
                                         DBUS_SERVICE_DBUS,
                                         DBUS_PATH_DBUS,
                                         DBUS_INTERFACE_DBUS);

  if (!org_freedesktop_DBus_request_name (bus_proxy,
                                          XSPLASH_DBUS_NAME,
                                          0, &nameret, &error))
    {
      g_warning ("Unable to call to request name.  You probably don't have sufficient privileges.");
      return 1;
    }

  if (nameret != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER)
    {
      g_warning ("Unable to get name");
      return 1;
    }

  server = g_object_new (xsplash_server_get_type (),
                         "dbus-object", XSPLASH_DBUS_OBJECT,
                         "system-bus", system_bus,
                         "dbus-proxy", bus_proxy,
                         NULL);

  g_timeout_add_seconds (timeout,
			 temporary_hack_for_initial_fade,
			 server);

  gtk_main ();

  g_message ("exiting...");

  if (background_image != NULL)
    g_free (background_image);

  if (logo_image != NULL)
    g_free (logo_image);

  if (throbber_image != NULL)
    g_free (throbber_image);

  if (signal_list != NULL)
    {
      g_slist_foreach (signal_list, (GFunc)g_free, NULL);
      g_slist_free (signal_list);
      signal_list = NULL;
    }

  return 0;
}

gboolean
xsplash_server_add_wait_signal (XsplashServer  *server,
                                gchar          *waitfor,
                                GError        **error)
{
  XsplashServerPrivate *priv = XSPLASH_SERVER_GET_PRIVATE (server);

  g_message ("received a new signal to wait for: %s", waitfor);

  if (waitfor)
    {
      add_signal (waitfor);
    }

  gdk_window_raise (priv->window->window);

  return TRUE;
}

gboolean
xsplash_server_signal_loaded (XsplashServer  *server,
                              gchar          *app,
                              GError        **error)
{
  g_message ("received signal: %s", app);

  if (gdm_session)
    {
      g_message ("gdm mode: fading out");
      begin_fade (server, TRUE);
    }
  else
    {
      GSList *l = NULL;

      /* Remove this signal if it's in our list; if the list is empty, begin fading */
      for (l = signal_list; l != NULL; l = l->next)
        {
          if (strcmp (app, (gchar *)l->data) == 0)
            {
              g_message ("received signal %s", (gchar*)l->data);

              g_free (l->data);
              signal_list = g_slist_delete_link (signal_list, l);

              /* There are no more signals we are listening for */
              if (signal_list == NULL)
                {
                  begin_fade (server, TRUE);
                }
            }
        }
    }

  return TRUE;
}
