/*****************************************************************************
 *
 * grail - Gesture Recognition And Instantiation Library
 *
 * Copyright (C) 2010 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors:
 *      Henrik Rydberg <rydberg@bitmath.org>
 *
 ****************************************************************************/

#include <grail.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>

static grail_mask_t flag_mask[DIM_GRAIL_TYPE_BYTES];

static int tp_get_clients(struct grail *ge,
			  struct grail_client_info *clients, int max_clients,
			  const struct grail_coord *coords, int num_coords,
			  const grail_mask_t *types, int type_bytes)
{
	memset(&clients[0], 0, sizeof(clients[0]));
	clients[0].id.client = 345;
	clients[0].id.root = 1;
	clients[0].id.event = 2;
	clients[0].id.child = 3;
	memcpy(clients[0].mask, flag_mask, sizeof(flag_mask));
	return 1;
}

static void tp_event(struct grail *ge, const struct input_event *ev)
{
	fprintf(stderr, "%lu.%06u %04x %04x %d\n",
		ev->time.tv_sec, (unsigned)ev->time.tv_usec,
		ev->type, ev->code, ev->value);
}

static void tp_gesture(struct grail *ge, const struct grail_event *ev)
{
	struct grail_contact touch[32];
	int i, ntouch;
	fprintf(stderr, "gesture %d %d %d %d - %f %f %d - %d\n",
		ev->type, ev->id, ev->client_id.client, ev->client_id.event,
		ev->pos.x, ev->pos.y, ev->ntouch,
		ev->status);
	ntouch = grail_get_contacts(ge, touch, 32);
	for (i = 0; i < ntouch; i++) {
		const struct grail_contact *t = &touch[i];
		fprintf(stderr, "\t%d: %d %d\n", i, t->id, t->tool_type);
		fprintf(stderr, "\t   %f %f\n", t->pos.x, t->pos.y);
		fprintf(stderr, "\t   %f %f %f %f\n", t->touch_major,
			t->touch_minor, t->width_major, t->width_minor);
		fprintf(stderr, "\t   %f %f\n", t->angle, t->pressure);
	}
	for (i = 0; i < ev->nprop; i++)
		fprintf(stderr, "\t%d: %f\n", i, ev->prop[i]);
}

static void loop_device(struct grail *ge, int fd)
{
	while (!grail_idle(ge, fd, 5000))
		grail_pull(ge, fd);
}

int main(int argc, char *argv[])
{
	struct grail ge;
	int i, fd;

	if (argc < 3) {
		fprintf(stderr, "Usage: %s <mask> <device>\n", argv[0]);
		return -1;
	}

	memset(&ge, 0, sizeof(ge));
	ge.get_clients = tp_get_clients;
	ge.event = tp_event;
	ge.gesture = tp_gesture;

	unsigned long mask = strtoull(argv[1], 0, 0);
	for (i = 0; i < DIM_GRAIL_TYPE; i++)
		if ((mask >> i) & 1)
			grail_mask_set(flag_mask, i);

	fd = open(argv[2], O_RDONLY | O_NONBLOCK);
	if (fd < 0) {
		fprintf(stderr, "error: could not open device\n");
		return -1;
	}
	if (ioctl(fd, EVIOCGRAB, 1)) {
		fprintf(stderr, "error: could not grab the device\n");
		return -1;
	}

	if (grail_open(&ge, fd)) {
		fprintf(stderr, "error: could not open touch device\n");
		return -1;
	}
	//grail_filter_abs_events(&ge, 1);

	struct grail_coord min = { -2, -1 }, max = { 2, 1 };
	grail_set_bbox(&ge, &min, &max);

	loop_device(&ge, fd);
	grail_close(&ge, fd);

	ioctl(fd, EVIOCGRAB, 0);
	close(fd);
	return 0;
}
