/**
 * @file geis_internal.h
 * @brief internal interface of the GEIS API
 *
 * This file is the internal interface to the GEIS API object.  It provides the
 * implementation hooks for plugins (back ends, servers, and extensions) to pass
 * information through the API.
 */

/*
 * Copyright 2010, 2011 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef GEIS_PRIVATE_H_
#define GEIS_PRIVATE_H_

#include <geis/geis.h>

#include "geis_error.h"
#include "geis_backend.h"
#include "geis_backend_token.h"
#include "geis_backend_multiplexor.h"
#include "geis_class.h"
#include "geis_device.h"
#include "geis_region.h"
#include "geis_subscription.h"


/**
 * Increases the reference count of an API instance object.
 *
 * @param[in] geis  The API instance.
 *
 * @returns the same API instance.
 */
Geis geis_ref(Geis geis);

/**
 * Decremenets the reference count of an API instance object.
 *
 * @param[in] geis  The API instance.
 *
 * If the reference count on the API instance object drops to zero, the object
 * is destroyed.
 */
void geis_unref(Geis geis);

/**
 * Gets the error stack from the geis object.
 *
 * @param[in] geis  The API instance.
 */
GeisErrorStack *geis_error_stack(Geis geis);

/**
 * Gets the subscription container from the geis object.
 *
 * @param[in] geis  The API instance.
 */
GeisSubBag geis_subscription_bag(Geis geis);

/**
 * Adds a back end file descriptor to multiplex.
 *
 * @param[in] geis     The API instance.
 * @param[in] fd       The file descriptor to add.
 * @param[in] callback The file descriptor event callback.
 * @param[in] context  A contextual datum.
 */
void geis_multiplex_fd(Geis                        geis,
                       int                         fd,
                       GeisBackendFdEventCallback  callback,
                       void                       *context);

/**
 * Removes a back end file descriptor from the multiplex.
 *
 * @param[in] geis     The API instance.
 * @param[in] fd       The file descriptor to remove.
 */
void geis_demultiplex_fd(Geis geis, int fd);

/**
 * Posts a new event through the API.
 *
 * @param[in] geis  The API instance.
 * @param[in] event The GEIS event.
 */
void geis_post_event(Geis geis, GeisEvent event);

typedef enum GeisProcessingResult {
  GEIS_PROCESSING_IGNORED       =  0,
  GEIS_PROCESSING_DISPOSE_EVENT = 10,
  GEIS_PROCESSING_COMPLETE      = 20,
  GEIS_PROCESSING_FAIL          = 99
} GeisProcessingResult;

typedef GeisProcessingResult (*GeisProcessingCallback)(GeisEvent  event,
                                                       void      *context);
/**
 * Registers an event-processing callback.
 *
 * @param[in] geis            The API instance.
 * @param[in] priority        The event-handling priority.
 * @param[in] event_callback  The event-handling callback.
 * @param[in] context         The context to supply to the callback.
 */
void geis_register_processing_callback(Geis                    geis,
                                       int                     priority,
                                       GeisProcessingCallback  event_callback,
                                       void                   *context);

/**
 * @defgroup geis_v2_plugin Plugin Interfaces
 * @ingroup geis_v2
 * Interfaces for plugin implementors.
 *
 * The Advanced API provides an extension capability through a defined plugin
 * interface.
 *
 * @{
 */

/**
 * A callback to add a term to a back end filter token.
 *
 * The callback receives the back end filter token, a back end callback context,
 * the name of the attribute, a filter operation, and the filter value.
 *
 * @param[in] token   The back-end filter token.
 * @param[in] context The back-end callback context.
 * @param[in] name    The filterable attribute name.
 * @param[in] op      The filter operation.
 * @param[in] value   The filter term value.
 */
typedef GeisStatus (*AddTermCallback)(GeisBackendToken     token,
                                      void                *context,
                                      GeisString           name,
                                      GeisFilterOperation  op,
                                      void                *value);

/**
 * The description of an attribute that can be used in subscirption filter
 * terms.
 *
 * When a facility (class, device, or region) registers the attributes that can
 * be used to build subscription filter terms, it passes a list of these structs
 * and the ABI instance maintains a collection of them.  The facility just needs
 * to indicate the name and type of the attribute and provide a callback and
 * context for creating the filter term.
 */
typedef struct GeisFilterableAttribute
{
  GeisString       name;
  GeisAttrType     type;
  AddTermCallback  add_term_callback;
  void            *add_term_context;
} *GeisFilterableAttribute;


/**
 * Handy sugar.
 */
static inline void geis_filterable_attribute_init(GeisFilterableAttribute fa,
                                                  GeisString              n,
                                                  GeisAttrType            t,
                                                  AddTermCallback         ac,
                                                  void                   *acc)
{
  fa->name = n;
  fa->type = t;
  fa->add_term_callback = ac;
  fa->add_term_context = acc;
}

/**
 * Resgisters a new gesture class with the API.
 *
 * @param[in] geis                  The API instance to which the new gesture
 *                                  class will be added.
 * @param[in] gesture_class         The new gesture class.
 * @param[in] filterable_attr_count The number of filterable attributes in the
 *                                  following structure.
 * @param[in] filterable_attrs      The set of attributes that may be used if
 *                                  filter terms for this class.
 *
 * An extension must register a gesture class if it will be providing new
 * gesture class events.  Any filterable attributes (other than gesture class
 * name, which is mandatory) must be provided.  A filterable attribute for the
 * class name will be automatically added.
 */
void geis_register_gesture_class(Geis                    geis,
                                 GeisGestureClass        gesture_class,
                                 GeisSize                filterable_attr_count,
                                 GeisFilterableAttribute filterable_attrs);

GeisGestureClassBag geis_gesture_classes(Geis geis);

/**
 * Resgisters a new input device with the API.
 *
 * @param[in] geis                  The API instance to which the new input
 *                                  device will be added.
 * @param[in] device                The new input device.
 * @param[in] filterable_attr_count The number of filterable attributes in the
 *                                  following structure.
 * @param[in] filterable_attrs      The set of attributes that may be used if
 *                                  filter terms for this device.
 */
void geis_register_device(Geis                    geis,
                          GeisDevice              device,
                          GeisSize                filterable_attr_count,
                          GeisFilterableAttribute filterable_attrs);

GeisDeviceBag geis_devices(Geis geis);

/**
 * Resgisters a new region (type) with the API.
 *
 * @param[in] geis                  The API instance to which the new region
 *                                  will be added.
 * @param[in] region                The new region.
 * @param[in] filterable_attr_count The number of filterable attributes in the
 *                                  following structure.
 * @param[in] filterable_attrs      The set of attributes that may be used if
 *                                  filter terms for this class.
 */
void geis_register_region(Geis                    geis,
                          GeisRegion              region,
                          GeisSize                filterable_attr_count,
                          GeisFilterableAttribute filterable_attrs);

/**
 * Resgisters a new special filters with the API.
 *
 * @param[in] geis                  The API instance to which the new region
 *                                  will be added.
 * @param[in] filterable_attr_count The number of filterable attributes in the
 *                                  following structure.
 * @param[in] filterable_attrs      The set of attributes that may be used if
 *                                  filter terms for this class.
 */
void geis_register_special(Geis                    geis,
                           GeisSize                filterable_attr_count,
                           GeisFilterableAttribute filterable_attrs);

/**
 * Invokes the filterable attribute's add_term callback for a named attribute.
 *
 * @param[in] geis     The API instance.
 * @param[in] facility The facility.
 * @param[in] token    The back end filter token.
 * @param[in] name     The filter attribute name.
 * @param[in] op       The filter operation.
 * @param[in] value    The filter value.
 */
GeisStatus geis_filterable_attribute_foreach(Geis                geis,
                                             GeisFilterFacility  facility,
                                             GeisBackendToken    token,
                                             GeisString          name,
                                             GeisFilterOperation op,
                                             void               *value);

/**
 * Gets the type of a named device attr.
 *
 * @param[in]  geis      The API instance.
 * @param[in]  attr_name The name of the device attr.
 *
 * Gets the type of a device attr by name, assuming the attr is known.
 *
 * There is a basic assumption that all device attrs of the same name are of the
 * same type.
 *
 * @returns the type of the attr, GEIS_ATTR_TYPE_UNKNOWN if the attr is unknown.
 */
GeisAttrType geis_get_device_attr_type(Geis geis, GeisString attr_name);

/**
 * Gets the type of a named class attr.
 *
 * @param[in]  geis      The API instance.
 * @param[in]  attr_name The name of the class attr.
 *
 * Gets the type of a class attr by name, assuming the attr is known.
 *
 * There is a basic assumption that all class attrs of the same name are of the
 * same type.
 *
 * @returns the type of the attr, GEIS_ATTR_TYPE_UNKNOWN if the attr is unknown.
 */
GeisAttrType geis_get_class_attr_type(Geis geis, GeisString attr_name);

/**
 * Gets the type of a named region attr.
 *
 * @param[in]  geis      The API instance.
 * @param[in]  attr_name The name of the region attr.
 *
 * Gets the type of a region attr by name, assuming the attr is known.
 *
 * There is a basic assumption that all region attrs of the same name are of the
 * same type.
 *
 * @returns the type of the attr, GEIS_ATTR_TYPE_UNKNOWN if the attr is unknown.
 */
GeisAttrType geis_get_region_attr_type(Geis geis, GeisString attr_name);

/**
 * Gets the type of a named special attr.
 *
 * @param[in]  geis      The API instance.
 * @param[in]  attr_name The name of the special attr.
 *
 * Gets the type of a special attr by name, assuming the attr is known.
 *
 * There is a basic assumption that all special attrs of the same name are of the
 * same type.
 *
 * @returns the type of the attr, GEIS_ATTR_TYPE_UNKNOWN if the attr is unknown.
 */
GeisAttrType geis_get_special_attr_type(Geis geis, GeisString attr_name);

/* @} */

#endif /* GEIS_PRIVATE_H_ */
