/**
 * @file libutouch-geis/geis_filter_term.c
 * @brief implementation of the uTouch GEIS v2.0 API filter term module
 *
 * Copyright 2011 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
#include "geis_config.h"
#include "geis_filter_term.h"

#include "geis_atomic.h"
#include "geis_attr.h"
#include "geis_error.h"
#include "geis_logging.h"
#include <stdlib.h>


/*
 * One of the terms of a filter. 
 */
struct _GeisFilterTerm
{
  GeisRefCount        refcount;
  GeisFilterFacility  facility;
  GeisFilterOperation op;
  GeisAttr            attr;
};


/*
 * All of the terms of a filter.
 */
struct _GeisFilterTermBag
{
  GeisFilterTerm *store;
  GeisSize        store_size;
  GeisSize        count;
};

static const GeisSize term_bag_growth_constant = 2;


GeisFilterTermBag
geis_filter_term_bag_new(GeisSize store_size)
{
  GeisFilterTermBag bag = calloc(1, sizeof(struct _GeisFilterTermBag));
  if (!bag)
  {
    geis_error("failed to allocate filter termbag");
    goto final_exit;
  }

  bag->store_size = store_size ? store_size : 3;
  bag->count = 0;
  bag->store = calloc(bag->store_size, sizeof(GeisFilterTerm));
  if (!bag->store)
  {
    geis_error("failed to allocate filter bag store");
    goto unwind_bag;
  }
  goto final_exit;

unwind_bag:
  free(bag);
  bag = NULL;
final_exit:
  return bag;
}


/*
 * Creates a new filter term bag by deep-copying an existing filter term bag.
 */
GeisFilterTermBag
geis_filter_term_bag_clone(GeisFilterTermBag original)
{
  GeisSize i;
  GeisFilterTermBag bag = geis_filter_term_bag_new(original->store_size);
  if (!bag)
  {
    goto final_exit;
  }

  bag->count = original->count;
  for (i = 0; i < bag->count; ++i)
  {
    bag->store[i] = geis_filter_term_ref(original->store[i]);
  }

final_exit:
  return bag;
}


void
geis_filter_term_bag_delete(GeisFilterTermBag bag)
{
  GeisSize i;
  for (i = 0; i < bag->count; ++i)
  {
    geis_filter_term_unref(bag->store[i]);
  }
  free(bag->store);
  free(bag);
}


GeisSize
geis_filter_term_bag_count(GeisFilterTermBag bag)
{
  return bag->count;
}


GeisFilterTerm
geis_filter_term_bag_term(GeisFilterTermBag bag, GeisSize index)
{
  GeisFilterTerm term = NULL;
  if (index < bag->count)
  {
    term = bag->store[index];
  }
  return term;
}


GeisStatus
geis_filter_term_bag_insert(GeisFilterTermBag bag,
                            GeisFilterTerm    term)
{
  GeisStatus status = GEIS_STATUS_UNKNOWN_ERROR;
  if (bag->count >= bag->store_size)
  {
    GeisSize new_store_size = bag->store_size * term_bag_growth_constant;
    GeisFilterTerm *new_store = realloc(bag->store,
             new_store_size * sizeof(struct _GeisFilterTerm));
    if (!new_store)
    {
      geis_error("failed to reallocate filter term bag");
      goto error_exit;
    }
    bag->store = new_store;
    bag->store_size = new_store_size;
  }
  bag->store[bag->count++] = term;
  status = GEIS_STATUS_SUCCESS;
  goto final_exit;

error_exit:
final_exit:
  return status;
}


GeisFilterTerm
geis_filter_term_new(GeisFilterFacility  facility,
                     GeisFilterOperation operation,
                     GeisAttr            attr)
{
  GeisFilterTerm term = calloc(1, sizeof(struct _GeisFilterTerm));
  if (!term)
  {
    geis_error("failed to allocate filter termbag");
    goto final_exit;
  }
  term->facility = facility;
  term->op       = operation;
  term->attr     = attr;
  geis_filter_term_ref(term);

final_exit:
  return term;
}


static void
_filter_term_destroy(GeisFilterTerm term)
{
  geis_attr_delete(term->attr);
  free(term);
}


/*
 * Increments the filter term's reference count.
 */
GeisFilterTerm
geis_filter_term_ref(GeisFilterTerm term)
{
  geis_atomic_ref(&term->refcount);
  return term;
}


/*
 * Decrements the filter term's reference count and maybe destroys the term.
 */
void
geis_filter_term_unref(GeisFilterTerm term)
{
  if (0 == geis_atomic_unref(&term->refcount))
  {
    _filter_term_destroy(term);
  }
}


GeisFilterFacility
geis_filter_term_facility(GeisFilterTerm term)
{
  return term->facility;
}


GeisFilterOperation
geis_filter_term_operation(GeisFilterTerm term)
{
  return term->op;
}


GeisAttr
geis_filter_term_attr(GeisFilterTerm term)
{
  return term->attr;
}



