/**
 * @file geis_backend.h
 * @brief internal GEIS back end base class public interface
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
#ifndef GEIS_BACKEND_H_
#define GEIS_BACKEND_H_

#include "geis/geis.h"

/**
 * Provides a virtual "base class" for various GEIS back ends.
 *
 * The GEIS is an API that implements the facade pattern over some number of
 * actual "back end" implementations.  Most internal operations of the API are
 * performed through this "virtual base class" interface rather than through the
 * concrete back ends (isn't object-oriented programming great?).
 */
typedef struct _GeisBackend *GeisBackend;


/**
 * Destroys the back end.
 *
 * @parameter[in] be  The back end.
 *
 * This function behaves like a virtual destructor and chains through to the
 * concrete destructor call.
 */
void geis_backend_delete(GeisBackend be);

/**
 * Gets the name of the back end.
 *
 * @parameter[in] be  The back end.
 *
 * This accessor is useful for diagnostics.
 */
GeisString geis_backend_name(GeisBackend be);

/**
 * Asks the back end to create a described subscription.
 */
GeisStatus geis_backend_subscribe(GeisBackend be, GeisSubscription sub);

/**
 * Asks a back end to remove a subscription.
 */
GeisStatus geis_backend_unsubscribe(GeisBackend be, GeisSubscription sub);

#endif /* GEIS_BACKEND_H_ */
