# -*- coding: utf-8 -*-
#
# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Platform - independent support code for utils."""

import os
import sys

try:
    from configparser import (SafeConfigParser, NoSectionError, NoOptionError)
except ImportError:
    from ConfigParser import (SafeConfigParser, NoSectionError, NoOptionError)

from twisted.internet import defer
from twisted.internet.utils import getProcessValue
from twisted.python import procutils

from dirspec.basedir import load_config_paths
from dirspec.utils import get_program_path

from ubuntuone.controlpanel.utils import logger

UPDATE_BIN_NAME = 'ubuntuone-updater'
UPDATE_CONFIG_NAME = 'update.conf'
DARWIN_APP_NAMES = {UPDATE_BIN_NAME: 'UbuntuOne Updater.app'}


def get_bin_cmd(exe_name):
    """Return list of cmds for an exe from the control panel project."""
    fallback_dir = os.path.join(os.path.dirname(__file__),
                                 os.pardir, os.pardir, os.pardir,
                                 "bin")

    cmd_args = [get_program_path(exe_name,
                                 fallback_dirs=[fallback_dir],
                                 app_names=DARWIN_APP_NAMES)]

    # procutils.which('python') returns python.exe on windows:
    if getattr(sys, 'frozen', None) is None:
        cmd_args.insert(0, procutils.which('python')[0])

    return cmd_args


def get_update_config():
    """Read the update.conf file and return a triple of
    the URL, version, and whether updates should be checked."""
    no_config = (None, None, None, False)
    url, version, channel, check = no_config

    files = [os.path.join(dir, UPDATE_CONFIG_NAME)
             for dir in load_config_paths("ubuntuone")]

    conf = SafeConfigParser()
    read_files = conf.read(files)

    if not read_files:
        logger.debug(
            "Unable to find an update.conf file - not checking for updates")
        return no_config

    try:
        url = conf.get("Update", "url")
        version = conf.get("Update", "version_id")
        channel = conf.get("Update", "channel")
        check = conf.getint("Update", "check_for_updates")
    except (NoSectionError, NoOptionError) as e:
        logger.debug("could not read update.conf - not checking for updates")
        logger.debug("   (exception was %r)" % e)
        return no_config

    return url, version, channel, check


@defer.inlineCallbacks
def call_updater(custom_call=None, install=False):
    """Call the Ubuntu One updater."""
    result = False
    retcode = None
    try:
        updater_cmd = get_bin_cmd(UPDATE_BIN_NAME)
    except OSError, e:
        logger.debug("Could not find updater command: %r" % e)
        defer.returnValue(result)

    url, local_version, channel, should_check = get_update_config()

    if not should_check:
        logger.debug("ignoring update check")
        defer.returnValue(result)

    args = ["--url", url,
            "--current", str(local_version),
            "--release", channel]
    if install:
        args.append("--install")

    exe = updater_cmd[0]
    args = updater_cmd[1:] + args

    logger.debug("calling %s update process: %r args %r",
                 "custom" if custom_call else "regular",
                 exe, args)

    if custom_call:
        custom_call(exe, args)
    else:
        # updater returns 1 if available, zero if not
        retcode = yield getProcessValue(exe, args)
        result = retcode == 1

    logger.debug('call_updater: %s, updater cmd: %r, return code: %r,'
                 ' result: %r', "installing" if install else "checking",
                 updater_cmd, retcode, result)
    defer.returnValue(result)
