# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the UniqueApplication class."""

from PyQt4 import QtCore
from twisted.internet.defer import inlineCallbacks

from ubuntuone.controlpanel.gui.tests import FakeSignal
from ubuntuone.controlpanel.gui.qt import uniqueapp
from ubuntuone.controlpanel.tests import TestCase


#pylint: disable=C0103
class FakeLocalSocket(object):
    """A fake QLocalSocket."""

    def __init__(self):
        self.connect_calls = []
        self.connect_timeouts = []
        self.connect_succeeds = True
        self.message = None

    def connectToServer(self, *args, **kwargs):
        """Fake connectToServer."""
        self.connect_calls.append((args, kwargs))

    def waitForConnected(self, timeout):
        """Fake waitForConnected."""
        self.connect_timeouts.append(timeout)
        return self.connect_succeeds

    def write(self, message):
        """Fake write."""
        self.message = message

    def flush(self):
        """Fake flush."""

    def close(self):
        """Fake close."""

    def waitForReadyRead(self):
        """Fake waitForReadyRead."""

    def readAll(self):
        """Fake readAll: return the message."""
        return self.message


class FakeLocalServer(object):

    """A fake QLocalServer."""

    def __init__(self, connected=True):
        self.newConnection = FakeSignal()
        self.listen_args = []
        self.socket = None
        self._removed_key = None
        self._is_connected = connected

    def listen(self, *args, **kwargs):
        """Fake listen."""
        self.listen_args.append((args, kwargs))
        return self._is_connected

    def nextPendingConnection(self):
        """Fake nextPendingConnection."""
        return self.socket

    def removeServer(self, key):
        """Fake removeServer."""
        self._removed_key = key

    def errorString(self):
        """Fake errorString."""
        return 'error'


class FakeApplication(object):
    """A fake QApplication."""


class UniqueAppTestCase(TestCase):
    """Test the UniqueAppplication class."""

    @inlineCallbacks
    def setUp(self):
        yield super(UniqueAppTestCase, self).setUp()
        self.local_socket = FakeLocalSocket()
        self.patch(uniqueapp.QtNetwork, "QLocalSocket",
            lambda: self.local_socket)
        self.local_server = FakeLocalServer()
        self.patch(uniqueapp.QtNetwork, "QLocalServer",
            lambda parent: self.local_server)
        self.patch(uniqueapp.sys, "exit", self._set_called)
        self.fake_quit = FakeSignal()
        self.patch(uniqueapp.UniqueApplication, "aboutToQuit", self.fake_quit)
        self.patch(uniqueapp.QtGui, "QApplication", FakeApplication)

    def test_cleanup_called_on_init(self):
        """Check that cleanup is called on initialization."""
        uniapp = uniqueapp.UniqueApplication([], "key")
        self.assertEqual("key", uniapp.server._removed_key)

    def test_on_failed_connection(self):
        """Check the flow of the program on connection fail."""
        data = []
        local_server = FakeLocalServer(False)
        self.patch(uniqueapp.QtNetwork, "QLocalServer",
            lambda parent: local_server)
        self.patch(uniqueapp.logger, "debug", data.append)
        uniqueapp.UniqueApplication([], "key")
        self.assertEqual(data, ['error'])

    def test_client_socket(self):
        """Check that the client socket is used correctly."""
        self.local_socket.connect_succeeds = True
        uniqueapp.UniqueApplication([], "key")
        self.assertEqual(self.local_socket.connect_calls,
        [(("key", QtCore.QIODevice.WriteOnly), {})])
        self.assertEqual(self.local_socket.connect_timeouts,
        [500])
        # The connection succeeds, so it should stop
        self.assertEqual(self._called, ((), {}))

    def test_server_socket(self):
        """Check that the server socket is used correctly."""
        self.local_socket.connect_succeeds = False
        uniqueapp.UniqueApplication([], "key")
        self.assertEqual(self.local_socket.connect_calls,
        [(("key", QtCore.QIODevice.WriteOnly), {})])
        self.assertEqual(self.local_socket.connect_timeouts,
        [500])

        # Should not exit
        self.assertEqual(self._called, False)

    def test_signal_connection(self):
        """Check that new_instance is correctly connected."""
        app = uniqueapp.UniqueApplication([], "key")
        # Yes, this is ugly. I can't find any other meaningful
        # way to compare them though.
        self.assertEqual(str(app.server.newConnection.target[0].__self__),
            str(app.new_instance))
        self.assertEqual(app.server.newConnection.target[1],
            app._process_messages)

    def test_cleanup(self):
        """Check that cleanup is called with the right key."""
        app = uniqueapp.UniqueApplication([], "key")
        self.assertEqual(self.fake_quit.target, [app.cleanup])

    def test_send_messages_valid(self):
        """Check the message is created correctly."""
        self.local_socket.connect_succeeds = True
        argv = ['python', 'ubuntuone-control-panel-qt',
            '--switch-to', 'share_links']
        uniqueapp.UniqueApplication(argv, "key")
        expected = "--switch-to=share_links"
        self.assertEqual(self.local_socket.message, expected)

    def test_send_messages_invalid(self):
        """Check the message is created correctly."""
        self.local_socket.connect_succeeds = True
        argv = ['python', 'ubuntuone-control-panel-qt']
        uniqueapp.UniqueApplication(argv, "key")
        expected = ""
        self.assertEqual(self.local_socket.message, expected)

    def test_process_message_with_message(self):
        """Check that we are able to parse the message received."""
        data = []
        self.local_socket.connect_succeeds = True
        argv = ['python', 'ubuntuone-control-panel-qt',
            '--switch-to', 'share_links']
        app = uniqueapp.UniqueApplication(argv, "key")
        self.local_server.socket = self.local_socket
        app.switch_to.connect(data.append)
        app.activate_window.connect(lambda: data.append(True))

        app.server.newConnection.emit()
        self.assertEqual(data, ["share_links", True])

    def test_process_message_no_message(self):
        """Check that we are able to parse the message received."""
        data = []
        self.local_socket.connect_succeeds = True
        argv = ['python', 'ubuntuone-control-panel-qt']
        app = uniqueapp.UniqueApplication(argv, "key")
        self.local_server.socket = self.local_socket
        app.switch_to.connect(data.append)
        app.activate_window.connect(lambda: data.append(True))

        app.server.newConnection.emit()
        self.assertEqual(data, [True])
