# -*- coding: utf-8 -*-
#
# Copyright 2012-2013 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Darwin-specific GUI event handling code."""

import sys
import subprocess

from PyQt4 import QtGui
from twisted.internet.defer import inlineCallbacks
from twisted.internet.error import ReactorNotRunning


from dirspec.utils import get_program_path

from ubuntuone.platform.tools import SyncDaemonTool

from ubuntuone.controlpanel import cache
from ubuntuone.controlpanel.backend import (STATUS_KEY, FILE_SYNC_ERROR)
from ubuntuone.controlpanel.gui.qt.main import twisted_main
from ubuntuone.controlpanel.logger import setup_logging


logger = setup_logging('qt.main.darwin')


class DarwinFileDialog(object):
    FOLDER_CHOOSER_SCRIPT = """
    tell application "System Events"
    activate
    set home to POSIX file "%s" as alias
    POSIX path of( choose folder default location home )
    end tell
    """

    @classmethod
    def getExistingDirectory(cls, directory=None, **kwargs):
        """Use command-line applescript to avoid QTBUG-28146."""
        try:
            s = cls.FOLDER_CHOOSER_SCRIPT % directory
            folder = subprocess.check_output(['osascript', '-e', s]).strip()
        except subprocess.CalledProcessError as e:
            logger.error("Could not call script to show open panel: %r" % e)
            folder = ''
        return folder


@inlineCallbacks
def handle_cmd_q(app, event, should_quit_sd=False):
    """Handle the quit event on darwin."""
    from twisted.internet import reactor

    if should_quit_sd:
        try:
            st = SyncDaemonTool()
            yield st.quit()
        except:
            pass
    try:
        twisted_main.main_quit(app)
    except ReactorNotRunning as e:
        logger.warning("Ignoring %r in main_quit." % e)

    app.aboutToQuit.emit()

    # iterate() is on a timer that is sometimes delayed. Calling it
    # here ensures that we exit immediately.
    reactor.iterate()


def install_platform_event_handlers(app, quit_kills_sd=False):
    """Add code to catch cmd-Q."""
    # on darwin, a menu item with 'quit' in its title is moved to
    # the application menu and given the cmd-Q shortcut by Qt. If
    # we don't provide one, Qt will give us one that just calls
    # QApplication::quit(), which doesn't quit syncdaemon and
    # causes problems with the qtreactor.

    # QMenuBar with no parent gives us an app-wide menu bar.
    # Note that Qt does not make this a singleton, though, so we have
    # to keep it around.
    menubar = QtGui.QMenuBar(None)

    quit_action = QtGui.QAction("quit", menubar,
                                triggered=lambda x:
                                handle_cmd_q(app,
                                             x,
                                             quit_kills_sd))
    quit_action.setObjectName("darwin-cmd-q")
    menu = menubar.addMenu("This string is not used.")
    menu.addAction(quit_action)

    return menubar


class MenubarIconLauncher(cache.Cache):
    """Listens to status, launches the menubar app once it is safe to do so."""

    def __init__(self):
        """Register as listener."""
        super(MenubarIconLauncher, self).__init__()
        self.backend.add_status_changed_handler(self.handle_status_update)

    def handle_status_update(self, result):
        """Process updates, launch menu on non-error"""

        if result[STATUS_KEY] == FILE_SYNC_ERROR:
            logger.warning("not starting menu, file sync in error state")
            return

        self.start_menu_process()
        self.backend.remove_status_changed_handler(self.handle_status_update)

    def start_menu_process(self):
        """Find and launch separate menu process."""
        if getattr(sys, 'frozen', None) is None:
            logger.warning("Can not launch pyobjc menu from source, ignoring.")
            return
        menu_app_name = 'Ubuntu One Menu'
        path = get_program_path(menu_app_name,
                                app_names={menu_app_name:
                                           'Ubuntu One Menu.app'})
        subprocess.Popen(path)
