# -*- coding: utf-8 -*-
#
# Copyright 2011-2013 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Provide the correct ui main module."""

import argparse
import os
import sys

from PyQt4 import QtCore
from ubuntu_sso.utils.locale import fix_turkish_locale

from ubuntuone.controlpanel.gui.qt.gui import start
# Module used to include the resources into this file
from ubuntuone.controlpanel.gui.qt.ui import images_rc
from ubuntuone.controlpanel.gui.qt.uniqueapp import UniqueApplication

# Avoid unused import warnings
assert(images_rc)

# Invalid name "source", pylint: disable=C0103
if sys.platform in ('win32', 'darwin'):
    from ubuntuone.controlpanel.gui.qt.main import twisted_main
    source = twisted_main
else:
    from ubuntuone.controlpanel.gui.qt.main import dbus_main
    source = dbus_main

if sys.platform == 'darwin':
    from ubuntuone.controlpanel.gui.qt.main.darwin import (
        install_platform_event_handlers,
        MenubarIconLauncher)
    assert(install_platform_event_handlers)
    assert(MenubarIconLauncher)
else:
    class MenubarIconLauncher:
        """Dummy. Separate menu launcher not used on win/linux."""
    install_platform_event_handlers = lambda app, **kwargs: None

# pylint: enable=C0103

from ubuntuone.controlpanel.utils import (install_config_and_daemons,
                                          add_u1_folder_to_favorites)


def parser_options():
    """Parse command line parameters."""
    result = argparse.ArgumentParser()
    result.add_argument("--switch-to", dest="switch_to",
                        metavar="PANEL_NAME", default="",
                        help="Start Ubuntu One in the "
                            "PANEL_NAME tab. Possible values are: "
                            "folders, share_links, devices, settings, account")
    result.add_argument("--minimized", dest="minimized", action="store_true",
                        default=False,
                        help="Start Ubuntu One "
                             "only in the notification area, with no "
                             "visible window. Implies --with-icon")
    result.add_argument("--with-icon", dest="with_icon", action="store_true",
                        default=False,
                        help="Start Ubuntu One "
                             "with an icon in the notification area.")
    result.add_argument("--installer", dest="installer", action="store_true",
                        default=False,
                        help="Show the license agreement page first.")
    return result


def install_qt4reactor():
    """Import and install the qt4reactor."""
    import qt4reactor
    qt4reactor.install()


def main(args, install_reactor_darwin=False):
    """Start the Qt mainloop and open the main window."""
    fix_turkish_locale()

    # Disable the overlay-scrollbar GTK module that was
    # added in Ubuntu 12.10 because it breaks Qt (LP:1007421)
    gtk_mod = os.getenv('GTK_MODULES')
    if gtk_mod is not None:
        gtk_mod = ':'.join([mod for mod in
            gtk_mod.split(':') if mod != 'overlay-scrollbar'])
        os.environ['GTK_MODULES'] = gtk_mod

    # The following cannot be imported outside this function
    # because u1trial already provides a reactor.

    args = ['ubuntuone-installer'] + args
    if sys.platform == 'darwin':
        args += ['-graphicssystem', 'raster']
    app = UniqueApplication(args, "ubuntuone-control-panel")

    # on darwin, must install qt4reactor after UniqueApplication init.
    # otherwise qt4reactor will create a QCoreApplication for us,
    # instead of the QApplication we need
    if install_reactor_darwin and sys.platform == 'darwin':
        install_qt4reactor()

    # Install translator for standard dialogs.
    locale = unicode(QtCore.QLocale.system().name())
    translator = QtCore.QTranslator()
    translator.load("qt_" + locale,
        QtCore.QLibraryInfo.location(QtCore.QLibraryInfo.TranslationsPath))
    app.installTranslator(translator)

    parser = parser_options()
    # Use only the arguments that are not recognized by Qt
    # and after the name of the binary (bug #956143)
    arg_list = [unicode(arg) for arg in app.arguments()]
    bin_position = arg_list.index(sys.argv[0]) + 1
    args = parser.parse_args(args=arg_list[bin_position:])
    switch_to = args.switch_to
    minimized = args.minimized
    with_icon = args.with_icon
    installer = args.installer
    source.main(app)

    data = []
    qss_files = [":/ubuntuone.qss", source.PLATFORM_QSS]
    if app.isRightToLeft():
        qss_files.append(":/ubuntuone-rtl.qss")
    for qss_name in qss_files:
        qss = QtCore.QResource(qss_name)
        data.append(unicode(qss.data()))
    app.setStyleSheet('\n'.join(data))

    install_config_and_daemons()
    add_u1_folder_to_favorites()

    if sys.platform == 'darwin':
        with_icon = False
        # cmd-q in CP shouldn't kill SD or the menu:
        quit_kills_sd = False
    else:
        quit_kills_sd = not with_icon

    # need to keep a reference to the menu or our handler will be
    # removed
    menubar = install_platform_event_handlers(app,
                                              quit_kills_sd=quit_kills_sd)
    menubar

    # Unused variable 'window', 'icon', pylint: disable=W0612
    icon, window = start(lambda: source.main_quit(app),
                         minimized=minimized, with_icon=with_icon,
                         installer=installer)
    if window:
        window.switch_to(switch_to)
        app.new_instance.connect(window.raise_)
    # pylint: enable=W0612
    if icon:
        app.new_instance.connect(icon.restore_window)

    if sys.platform == 'darwin':
        # start separate menu app instead of Qt systray menu, waiting
        # until it is safe to do so without IPC races:
        menu_launcher = MenubarIconLauncher()
        assert(menu_launcher)

    source.main_start(app)
