/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*-
 *
 * Authors: Alejandro J. Cura <alecu@canonical.com>
 *
 * Copyright (C) 2010 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "config.h"

#include <glib/gi18n-lib.h>
#include <gmodule.h>
#include <gconf/gconf-client.h>
#include <gtk/gtk.h>

#include "gnome-settings-daemon/gnome-settings-plugin.h"
#include "gsd-ubuntuone.h"

#ifdef REGISTER_PLUGIN
GNOME_SETTINGS_PLUGIN_REGISTER (GsdUbuntuOne, gsd_ubuntuone)
#else
G_DEFINE_TYPE (GsdUbuntuOne, gsd_ubuntuone, G_TYPE_OBJECT);
#endif

#define CHECKED_BOOKMARK_FILE_KEY "/apps/gnome_settings_daemon/plugins/ubuntuone/checked_bookmark_file"

static void
bookmark_file_loaded (GObject *source, GAsyncResult *res, gpointer user_data)
{
        gchar *contents = NULL;
        GError *error = NULL;

        g_file_load_contents_finish (G_FILE (source), res, &contents, NULL, NULL, &error);
        if (error == NULL) {
                gchar **lines, *u1_location;
                gint i;
                gboolean add_it = TRUE;

                u1_location = g_build_filename ("file://", g_get_home_dir (), "Ubuntu%20One", NULL);

                lines = g_strsplit (contents, "\n", -1);
                for (i = 0; lines[i] != NULL; i++) {
                        /* Ignore empty lines */
                        if (lines[i][0] != '\0' && lines[i][0] != ' ') {
                                if (g_str_has_prefix (lines[i], u1_location)) {
                                        add_it = FALSE;
                                        break;
                                }
                        }
                }

                if (add_it) {
                        gchar *new_contents = g_strdup_printf ("%s\n%s Ubuntu One\n", contents, u1_location);

                        if (g_file_replace_contents (G_FILE (source),
                                                     (const gchar *) new_contents,
                                                     strlen (new_contents),
                                                     NULL,
                                                     FALSE,
                                                     0, NULL, NULL, &error)) {
                                GConfClient *conf_client;

                                conf_client = gconf_client_get_default ();
                                gconf_client_set_bool (conf_client, CHECKED_BOOKMARK_FILE_KEY, TRUE, NULL);
                        } else {
                                g_warning ("Could not save bookmarks file: %s\n", error->message);
                                g_error_free (error);
                        }
                }

                g_free (contents);
                g_free (u1_location);
                g_strfreev (lines);
        } else {
                g_warning ("Could not load bookmark file: %s\n", error->message);
                g_error_free (error);
        }
}

static void
check_bookmark_file (void)
{
        gchar *filename;
        GConfClient *conf_client;
        GError *error = NULL;

        /* We only check the bookmark file if we haven't already done so */
        conf_client = gconf_client_get_default ();
        if (!gconf_client_get_bool (conf_client, CHECKED_BOOKMARK_FILE_KEY, &error)) {
                if (error == NULL) {
                        gchar *u1_folder;

                        u1_folder = g_build_filename (g_get_home_dir (), "Ubuntu One", NULL);
                        if (g_file_test ((const gchar *) u1_folder, G_FILE_TEST_IS_DIR)) {
                                /* Load the bookmark file */
                                filename = g_build_filename (g_get_home_dir (), ".gtk-bookmarks", NULL);
                                if (filename != NULL) {
                                        GFile *file;

                                        file = g_file_new_for_path (filename);
                                        g_file_load_contents_async (file, NULL, bookmark_file_loaded, NULL);

                                        g_object_unref (G_OBJECT (file));
                                        g_free (filename);
                                }
                        }

                        g_free (u1_folder);
                } else {
                        g_warning ("Error getting settings: %s\n", error->message);
                        g_error_free (error);
                }
        }

        g_object_unref (conf_client);
}

static void
credentials_found_cb (SyncdaemonAuthentication *auth, SyncdaemonCredentials *credentials, gpointer user_data)
{
        check_bookmark_file ();
}

static gboolean
delayed_syncdaemon_start (gpointer data)
{
	GsdUbuntuOne *plugin;
        SyncdaemonAuthentication *auth;

	plugin = GSD_UBUNTUONE (data);

	plugin->syncdaemon = syncdaemon_daemon_new ();

        /* Check for authentication */
        auth = syncdaemon_daemon_get_authentication (plugin->syncdaemon);
        g_signal_connect (auth, "credentials_found",
                          G_CALLBACK (credentials_found_cb), NULL);

	return FALSE;
}

static void
gsd_ubuntuone_init (GsdUbuntuOne *plugin)
{
}

void
gsd_ubuntuone_activate (GnomeSettingsPlugin *gsp_object)
{
	GsdUbuntuOne *plugin = GSD_UBUNTUONE (gsp_object);

	g_timeout_add (DELAYED_START_TIMEOUT, delayed_syncdaemon_start, plugin);
}

static void
gsd_ubuntuone_dispose (GObject *object)
{
	GsdUbuntuOne *plugin = GSD_UBUNTUONE (object);

	if (plugin->syncdaemon != NULL)
                g_object_unref (plugin->syncdaemon);

	G_OBJECT_CLASS (gsd_ubuntuone_parent_class)->dispose (object);
}

static void
gsd_ubuntuone_class_init (GsdUbuntuOneClass *klass)
{
	GObjectClass *g_class = G_OBJECT_CLASS (klass);
	GnomeSettingsPluginClass *gsp_class = (GnomeSettingsPluginClass *) klass;

	g_class->dispose = gsd_ubuntuone_dispose;
	gsp_class->activate = gsd_ubuntuone_activate;
}
