/*
 * Syncdaemon API
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "config.h"
#ifdef HAVE_GDBUS
#include <gio/gio.h>
#else
#include <dbus/dbus-glib.h>
#endif
#include "syncdaemon-daemon.h"
#include "syncdaemon-interface.h"
#include "syncdaemon-marshal.h"

G_DEFINE_TYPE(SyncdaemonInterface, syncdaemon_interface, G_TYPE_OBJECT)

struct _SyncdaemonInterfacePrivate {
	SyncdaemonDaemon *daemon;
#ifdef HAVE_GDBUS
	GDBusConnection *bus;
	GDBusProxy *proxy;
#else
	DBusGConnection *bus;
	DBusGProxy *proxy;
#endif
};

enum {
	PROP_0,
	PROP_DAEMON
};

static void
syncdaemon_interface_finalize (GObject *object)
{
	SyncdaemonInterface *interface = SYNCDAEMON_INTERFACE (object);

	if (interface->priv != NULL) {
		if (interface->priv->proxy != NULL)
			g_object_unref (G_OBJECT (interface->priv->proxy));

		if (interface->priv->bus != NULL) {
#ifdef HAVE_GDBUS
			g_object_unref (G_OBJECT (interface->priv->bus));
#else
			dbus_g_connection_unref (interface->priv->bus);
#endif
		}

		if (interface->priv->daemon != NULL)
			g_object_unref (G_OBJECT (interface->priv->daemon));

		g_free (interface->priv);
		interface->priv = NULL;
	}

	G_OBJECT_CLASS (syncdaemon_interface_parent_class)->finalize (object);
}

static void
syncdaemon_interface_set_property (GObject *object,
				   guint prop_id,
				   const GValue *value,
				   GParamSpec *pspec)
{
        SyncdaemonInterface *interface = SYNCDAEMON_INTERFACE (object);

        switch (prop_id) {
        case PROP_DAEMON:
                if (interface->priv->daemon != NULL)
                        g_object_unref (G_OBJECT (interface->priv->daemon));
                interface->priv->daemon = g_object_ref (G_OBJECT (g_value_get_object (value)));
                break;
        default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
                break;
        }
}

static void
syncdaemon_interface_get_property (GObject *object,
				   guint prop_id,
				   GValue *value,
				   GParamSpec *pspec)
{
        SyncdaemonInterface *interface = SYNCDAEMON_INTERFACE (object);

        switch (prop_id) {
        case PROP_DAEMON:
                g_value_set_object (value, interface->priv->daemon);
                break;
        default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
                break;
        }
}

static void
syncdaemon_interface_class_init (SyncdaemonInterfaceClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = syncdaemon_interface_finalize;
	object_class->set_property = syncdaemon_interface_set_property;
	object_class->get_property = syncdaemon_interface_get_property;

	/* Install properties */
	g_object_class_install_property (object_class,
					 PROP_DAEMON,
					 g_param_spec_object ("daemon",
							      "SyncdaemonDaemon object",
							      "Daemon object from which this interface was created",
							      SYNCDAEMON_TYPE_DAEMON,
							      G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

	/* Register DBus marshallers */
	dbus_g_object_register_marshaller (_syncdaemon_marshal_VOID__STRING_POINTER,
					   G_TYPE_NONE,
					   G_TYPE_STRING,
					   dbus_g_type_get_map ("GHashTable",
								G_TYPE_STRING,
								G_TYPE_STRING),
					   G_TYPE_INVALID);
	dbus_g_object_register_marshaller (_syncdaemon_marshal_VOID__POINTER_STRING,
					   G_TYPE_NONE,
					   dbus_g_type_get_map ("GHashTable",
								G_TYPE_STRING,
								G_TYPE_STRING),
					   G_TYPE_STRING,
					   G_TYPE_INVALID);
}

static void
syncdaemon_interface_init (SyncdaemonInterface *interface)
{
	GError *error = NULL;

	interface->priv = g_new0 (SyncdaemonInterfacePrivate, 1);

	/* Initialize DBus */
#ifdef HAVE_GDBUS
	interface->priv->bus = g_bus_get_sync (G_BUS_TYPE_SESSION, NULL, &error);
#else
	interface->priv->bus = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
#endif
	if (error != NULL) {
		g_warning ("Couldn't get session bus: %s", error->message);
		g_error_free (error);
	}
}

static void
#ifdef HAVE_GDBUS
#else
signal_error_cb (DBusGProxy *proxy, const gchar *signal, GHashTable *extra_args, gpointer user_data)
#endif
{
	SyncdaemonInterface *interface = SYNCDAEMON_INTERFACE (user_data);

	if (interface->priv->daemon != NULL)
		g_signal_emit_by_name (interface->priv->daemon, "error", signal, extra_args);
}

/**
 * syncdaemon_interface_setup_proxy:
 * @interface: A #SyncdaemonInterface object
 * @unique_bus_name: Unique DBus interface name
 * @object_path: Object path for this interface
 * @interface_name: Interface name for this interface
 *
 * This function should only be used by SyncdaemonInterface-based classes, to
 * setup the DBus proxy to be used for communication with the underlying DBus
 * interface.
 *
 * Return value: The DBus proxy associated with this interface.
 */
GObject *
syncdaemon_interface_setup_proxy (SyncdaemonInterface *interface,
				  const gchar *unique_bus_name,
				  const gchar *object_path,
				  const gchar *interface_name)
{
	GError *error = NULL;

	g_return_val_if_fail (SYNCDAEMON_IS_INTERFACE (interface), NULL);

	if (interface->priv->proxy != NULL)
		g_object_unref (G_OBJECT (interface->priv->proxy));

#ifdef HAVE_GDBUS
        interface->priv->proxy = g_dbus_proxy_new_sync (interface->priv->bus, , , ,
							unique_bus_name, object_path, interface_name,
							NULL, &error);
#else
	interface->priv->proxy = dbus_g_proxy_new_for_name (interface->priv->bus, unique_bus_name,
							    object_path,
							    interface_name);
#endif

	if (interface->priv->proxy != NULL) {
#ifdef HAVE_GDBUS
#else
		dbus_g_proxy_add_signal (interface->priv->proxy, "SignalError",
					 G_TYPE_STRING,
					 dbus_g_type_get_map ("GHashTable",
							      G_TYPE_STRING,
							      G_TYPE_STRING),
					 G_TYPE_INVALID);
		dbus_g_proxy_connect_signal (interface->priv->proxy, "SignalError",
					     G_CALLBACK (signal_error_cb), interface, NULL);
#endif
	} else {
#ifdef HAVE_GDBUS
		g_warning ("Could not get proxy for %s: %s", interface_name, error->message);
		g_error_free (error);
#else
		g_warning ("Could not get proxy for %s", interface_name);
#endif
	}

	return interface->priv->proxy;
}

/**
 * syncdaemon_interface_get_proxy_object:
 * @interface: A #SyncdaemonInterface object
 *
 * Return the DBus proxy object associated with the given Syncdaemon interface.
 *
 * This function should not be used by applications, as this is just a convenience
 * function for internal implementation of the classes in the library.
 *
 * Return value: The DBus proxy associated with this interface, or NULL if it hasn't
 * been setup yet.
 */
GObject *
syncdaemon_interface_get_proxy_object (SyncdaemonInterface *interface)
{
	g_return_val_if_fail (SYNCDAEMON_IS_INTERFACE (interface), NULL);

	return interface->priv->proxy;
}
