# -*- coding: utf-8 -*-
#
# Author: Rodney Dawes <rodney.dawes@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
""" Tests for the ubuntuone-preferences control panel app."""

import new
import os
import gnomekeyring

from contrib.mocker import MockerTestCase
from contrib.testing.testcase import DBusTwistedTestCase, FakeLogin
from twisted.internet import defer
from twisted.python.failure import Failure
from ubuntuone.syncdaemon import dbus_interface, tools

class SyncDaemonTool(tools.SyncDaemonTool):
    """A subclass of tools.SyncDaemonTool that keep track of the deferreds."""

    def __init__(self, *args, **kwargs):
        super(SyncDaemonTool, self).__init__(*args, **kwargs)
        self._deferreds = set()

    def __getattribute__(self, attr):
        attribute = super(SyncDaemonTool, self).__getattribute__(attr)
        import inspect
        if inspect.ismethod(attribute) or inspect.isfunction(attribute):
            def wrapper(*args, **kwargs):
                value = attribute(*args, **kwargs)
                if isinstance(value, defer.Deferred):
                    self._deferreds.add(value)
                return value
            return wrapper
        else:
            return attribute


class InvalidSignalError(Exception):
    """Exception for when we get the wrong signal called."""
    pass


class PreferencesTests(MockerTestCase, DBusTwistedTestCase):
    """Basic tests for the ubuntuone-preferences app."""

    _path = os.path.join(os.getcwd(), "bin", "ubuntuone-preferences")
    u1prefs = new.module('u1prefs')
    execfile(_path, u1prefs.__dict__)
    u1prefs.DBUS_IFACE_AUTH_PATH = '/oauthdesktop'

    def setUp(self):
        MockerTestCase.setUp(self)
        DBusTwistedTestCase.setUp(self)
        self.oauth = FakeLogin(self.bus)
        self._old_path = dbus_interface.DBUS_PATH_AUTH
        dbus_interface.DBUS_PATH_AUTH = '/oauthdesktop'

        # For testing keyring queries
        self.keyring = self.mocker.mock()
        self.item = self.mocker.mock(gnomekeyring.Found)

        self.item_id = 999

        self.item.item_id
        self.mocker.result(self.item_id)
        self.mocker.count(0, None)

        self.item.secret
        self.mocker.result('oauth_token=access_key'
                           '&oauth_token_secret=access_secret')
        self.mocker.count(0, None)

        self.keyring.find_items_sync(
            None,
            {'ubuntuone-realm': 'https://ubuntuone.com',
             'oauth-consumer-key': 'ubuntuone'})
        self.mocker.count(0, None)
        self.mocker.result([self.item])
        self.keyring.ITEM_GENERIC_SECRET
        self.mocker.count(0, None)
        self.mocker.result(None)

        self.u1prefs.make_rest_request = self.make_rest_request
        self.u1prefs.SyncDaemonTool = SyncDaemonTool

    @defer.inlineCallbacks
    def tearDown(self):
        # collect all signal receivers registered during the test
        signal_receivers = set()
        with self.bus._signals_lock:
            for group in self.bus._signal_recipients_by_object_path.values():
                for matches in group.values():
                    for match in matches.values():
                        signal_receivers.update(match)
        yield self.cleanup_signal_receivers(signal_receivers)
        self.oauth.shutdown()
        dbus_interface.DBUS_PATH_AUTH = self._old_path
        yield DBusTwistedTestCase.tearDown(self)

    def make_rest_request(self, url=None, method='GET',
                          callback=None, keyring=None):
        """Override the real request call to mock some stuff."""
        if callback:
            callback(self.content)
        else:
            raise Exception('No callback provided.')

    def test_bw_throttling(self):
        """Test that toggling bw throttling works correctly."""
        self.mocker.replay()
        widget = self.u1prefs.DevicesWidget(None, keyring=self.keyring)
        widget.update_bw_settings = self.mocker.mock()
        try:
            widget.devices = []
            widget.list_devices()
            self.assertFalse(widget.bw_limited,
                             "the bandwidth should start out not limited")
            self.assertTrue(widget.bw_chk,
                            "the checkbox should be present")
            self.assertFalse(widget.bw_chk.get_active(),
                             "the checkbox should start out unchecked")
            self.assertFalse(widget.up_spin.get_property('sensitive') or
                             widget.dn_spin.get_property('sensitive'),
                             "the spinbuttons should start out unsensitive")
            widget.bw_chk.set_active(True)
            self.assertTrue(widget.bw_chk.get_active(),
                            "the checkbox should now be checked")
            self.assertTrue(widget.up_spin.get_property('sensitive') and
                            widget.dn_spin.get_property('sensitive'),
                            "the spinbuttons should now be sensitive")
        finally:
            widget.destroy()

    def test_list_devices_fills_devices_list_with_fake_result_when_empty(self):
        self.mocker.replay()
        widget = self.u1prefs.DevicesWidget(None, keyring=self.keyring)
        widget.update_bw_settings = self.mocker.mock()
        try:
            widget.devices = []
            widget.list_devices()
            # the devices list is no longer empty
            self.assertTrue(widget.devices)
            # it has 'fake' data (referring to the local machine)
            self.assertTrue('FAKE' in widget.devices[0])
        finally:
            widget.destroy()

    def test_list_devices_shows_devices_list(self):
        self.mocker.replay()
        widget = self.u1prefs.DevicesWidget(None, keyring=self.keyring)
        widget.update_bw_settings = self.mocker.mock()
        try:
            widget.devices = []
            widget.list_devices()
            # fake data now in devices
            interesting = []
            for i in widget.get_children():
                clsname = i.__class__.__name__
                if clsname == 'Image':
                    interesting.append((clsname, i.get_icon_name()[0]))
                if clsname in ('Label', 'Button', 'CheckButton'):
                    interesting.append((clsname, i.get_label()))
            # check there is an image of a computer in there
            self.assertTrue(('Image', 'computer') in interesting)
            # check a placeholder for the local machine description is there
            self.assertTrue(('Label', '<LOCAL MACHINE>') in interesting)
            # check the bw limitation stuff is there
            self.assertTrue(('CheckButton', '_Limit Bandwidth Usage')
                            in interesting)
            self.assertTrue(('Label', 'Maximum _download speed (KB/s):')
                            in interesting)
            self.assertTrue(('Label', 'Maximum _upload speed (KB/s):')
                            in interesting)
            # check the 'Remove' button is *not* there
            self.assertTrue(('Button', 'Remove') not in interesting)
        finally:
            widget.destroy()

    def test_list_devices_shows_real_devices_list(self):
        self.mocker.replay()
        widget = self.u1prefs.DevicesWidget(None, keyring=self.keyring)
        widget.update_bw_settings = self.mocker.mock()
        try:
            widget.devices = [{'kind': 'Computer',
                               'description': 'xyzzy',
                               'token': 'blah'},
                              {'kind': 'Phone',
                               'description': 'quux',
                               'token': '1234'}]
            widget.list_devices()
            # fake data now in devices
            interesting = []
            for i in widget.get_children():
                clsname = i.__class__.__name__
                if clsname == 'Image':
                    interesting.append((clsname, i.get_icon_name()[0]))
                if clsname in ('Label', 'Button', 'CheckButton'):
                    interesting.append((clsname, i.get_label()))
            # check there is an image of a computer in there
            self.assertTrue(('Image', 'computer') in interesting)
            # and of a phone
            self.assertTrue(('Image', 'phone') in interesting)
            # check a label of the local machine description is there
            self.assertTrue(('Label', 'xyzzy') in interesting)
            # check the bw limitation stuff is not there (no local machine)
            self.assertTrue(('CheckButton', '_Limit Bandwidth Usage')
                            not in interesting)
            self.assertTrue(('Label', 'Download (kB/s):') not in interesting)
            self.assertTrue(('Label', 'Upload (kB/s):') not in interesting)
            # check the 'Remove' button is there
            self.assertTrue(('Button', 'Remove') in interesting)
        finally:
            widget.destroy()

    def test_quota_display(self):
        """Test that quota display works correctly."""
        self.mocker.replay()
        dialog = self.u1prefs.UbuntuOneDialog(keyring=self.keyring)
        self.assertTrue(dialog is not None)
        self.assertEqual(dialog.usage_graph.get_fraction(), 0.0)
        dialog.update_quota_display(1024, 2048)
        self.assertEqual(dialog.usage_graph.get_fraction(), 0.5)
        dialog.destroy()
        return defer.DeferredList(dialog.sdtool._deferreds)

    def test_request_quota_info(self):
        """Test that we can request the quota info properly."""
        self.content = {"total":2048, "used":1024}
        self.mocker.replay()
        dialog = self.u1prefs.UbuntuOneDialog(keyring=self.keyring)
        self.assertTrue(dialog is not None)
        self.assertEqual(dialog.usage_graph.get_fraction(), 0.0)
        dialog.request_quota_info()
        self.assertEqual(dialog.usage_graph.get_fraction(), 0.5)
        dialog.destroy()
        return defer.DeferredList(dialog.sdtool._deferreds)

    def test_no_overquota_notice_on_low_usage(self):
        """Test that the quota notice is not visible if usage is low."""
        self.content = {"total":2048, "used":1024}
        self.mocker.replay()
        dialog = self.u1prefs.UbuntuOneDialog(keyring=self.keyring)
        self.assertTrue(dialog is not None)
        dialog.request_quota_info()
        # the label should just be the blank '\n'
        self.assertEqual(dialog.overquota_label.get_text(), '\n')
        # and the icon should be blank, too
        self.assertEqual(dialog.overquota_img.get_icon_name()[0], None)
        dialog.destroy()
        return defer.DeferredList(dialog.sdtool._deferreds)

    def test_overquota_notice_and_upgrade_offer_on_free_high_usage(self):
        """Test that the quota notice is visible if usage is 95%.

        Also make sure the label mentions upgrading.
        """
        self.content = {"total": 100, "used": 95}
        self.mocker.replay()
        dialog = self.u1prefs.UbuntuOneDialog(keyring=self.keyring)
        self.assertTrue(dialog is not None)
        dialog.request_quota_info()
        # don't check the exact text, as it will probably
        # change. Check we're setting some amount of text,
        self.assertTrue(len(dialog.overquota_label.get_text()) > 100)
        # and check that we mention upgrading
        self.assertTrue('upgrad' in dialog.overquota_label.get_text().lower())
        # and check that the icon used is just informational
        self.assertEqual(dialog.overquota_img.get_icon_name()[0],
                         'dialog-information')
        dialog.destroy()
        return defer.DeferredList(dialog.sdtool._deferreds)

    def test_overquota_notice_and_not_upgrade_offer_on_paid_high_usage(self):
        """Test that the quota notice is visible if usage is 95%.

        Also make sure the label does not mention upgrading.
        """
        self.content = {"total": 50<<30, "used": 49<<30}
        self.mocker.replay()
        dialog = self.u1prefs.UbuntuOneDialog(keyring=self.keyring)
        self.assertTrue(dialog is not None)
        dialog.request_quota_info()
        # don't check the exact text, as it will probably
        # change. Check we're setting some amount of text,
        self.assertTrue(len(dialog.overquota_label.get_text()) > 100)
        # and check that we mention upgrading
        self.assertTrue('upgrad' not in
                        dialog.overquota_label.get_text().lower())
        # and check that the icon used is just informational
        self.assertEqual(dialog.overquota_img.get_icon_name()[0],
                         'dialog-information')
        dialog.destroy()
        return defer.DeferredList(dialog.sdtool._deferreds)

    def test_overquota_warning_and_upgrade_offer_there_on_free_over_quota(self):
        """Test that the quota notice is visible if usage is 100%.

        Also make sure the label mentions upgrading.
        """
        self.content = {"total": 100, "used": 100}
        self.mocker.replay()
        dialog = self.u1prefs.UbuntuOneDialog(keyring=self.keyring)
        self.assertTrue(dialog is not None)
        dialog.request_quota_info()
        # don't check the exact text, as it will probably
        # change. Check we're setting some amount of text,
        self.assertTrue(len(dialog.overquota_label.get_text()) > 100)
        # and check that we mention upgrading
        self.assertTrue('upgrad' in dialog.overquota_label.get_text().lower())
        # and check that the icon used is a warning
        self.assertEqual(dialog.overquota_img.get_icon_name()[0],
                         'dialog-warning')
        dialog.destroy()
        return defer.DeferredList(dialog.sdtool._deferreds)

    def test_overquota_warning_there_on_paid_over_quota(self):
        """Test that the quota notice is visible if usage is 100%."""
        self.content = {"total": 50<<30, "used": 50<<30}
        self.mocker.replay()
        dialog = self.u1prefs.UbuntuOneDialog(keyring=self.keyring)
        self.assertTrue(dialog is not None)
        dialog.request_quota_info()
        # don't check the exact text, as it will probably
        # change. Check we're setting some amount of text,
        self.assertTrue(len(dialog.overquota_label.get_text()) > 100)
        # and check that the icon used is a warning
        self.assertEqual(dialog.overquota_img.get_icon_name()[0],
                         'dialog-warning')
        dialog.destroy()
        return defer.DeferredList(dialog.sdtool._deferreds)

    def test_request_account_info(self):
        """Test that we can request the account info properly."""
        self.content = {"username": "ubuntuone", "nickname": "Ubuntu One",
                        "email": "uone@example.com"}
        self.mocker.replay()
        dialog = self.u1prefs.UbuntuOneDialog(keyring=self.keyring)
        self.assertTrue(dialog is not None)
        dialog.request_account_info()
        self.content = {"total":2048, "used":1024}
        dialog.request_quota_info()
        self.assertEqual(dialog.name_label.get_text(), 'Ubuntu One')
        self.assertEqual(dialog.mail_label.get_text(), 'uone@example.com')
        # ensure the plan label says "Free" and the upgrade button is there
        self.assertEqual(dialog.plan_label.get_text(), 'Free')
        self.assertTrue(dialog.upgrade_link.get_visible())
        # whoops, the user upgraded
        self.content = {"total": 50<<30, "used":1024}
        dialog.request_quota_info()
        # ensure the plan label says "Paid" and the upgrade button is not there
        self.assertEqual(dialog.plan_label.get_text(), 'Paid')
        self.assertFalse(dialog.upgrade_link.get_visible())
        dialog.destroy()
        return defer.DeferredList(dialog.sdtool._deferreds)

    def test_toggle_bookmarks(self):
        """Test toggling the bookmarks service on/off."""
        toggle_db_sync = self.mocker.mock()
        self.expect(toggle_db_sync('bookmarks', False))
        self.expect(toggle_db_sync('bookmarks', True))
        self.expect(toggle_db_sync('bookmarks', False))
        self.mocker.replay()
        dialog = self.u1prefs.UbuntuOneDialog(keyring=self.keyring)
        self.assertTrue(dialog is not None)
        dialog.toggle_db_sync = toggle_db_sync
        dialog.bookmarks_check.set_active(True)
        self.assertTrue(dialog.bookmarks_check.get_active())
        dialog.bookmarks_check.set_active(False)
        self.assertFalse(dialog.bookmarks_check.get_active())
        dialog.bookmarks_check.set_active(True)
        self.assertTrue(dialog.bookmarks_check.get_active())
        dialog.destroy()
        return defer.DeferredList(dialog.sdtool._deferreds)

    def test_toggle_contacts(self):
        """Test toggling the contacts service on/off."""
        toggle_db_sync = self.mocker.mock()
        self.expect(toggle_db_sync('contacts', False))
        self.expect(toggle_db_sync('contacts', True))
        self.expect(toggle_db_sync('contacts', False))
        self.mocker.replay()
        dialog = self.u1prefs.UbuntuOneDialog(keyring=self.keyring)
        self.assertTrue(dialog is not None)
        dialog.toggle_db_sync = toggle_db_sync
        dialog.abook_check.set_active(True)
        self.assertTrue(dialog.abook_check.get_active())
        dialog.abook_check.set_active(False)
        self.assertFalse(dialog.abook_check.get_active())
        dialog.abook_check.set_active(True)
        self.assertTrue(dialog.abook_check.get_active())
        dialog.destroy()
        return defer.DeferredList(dialog.sdtool._deferreds)


    def test_toggle_files(self):
        """Test toggling the files service on/off."""
        self.mocker.replay()
        dialog = self.u1prefs.UbuntuOneDialog(keyring=self.keyring)
        self.assertTrue(dialog is not None)
        dialog.files_check.set_active(True)
        self.assertTrue(dialog.files_check.get_active())
        dialog.files_check.set_active(False)
        self.assertFalse(dialog.files_check.get_active())
        dialog.files_check.set_active(True)
        self.assertTrue(dialog.files_check.get_active())
        dialog.destroy()
        return defer.DeferredList(dialog.sdtool._deferreds)

    def test_toggle_files_and_music(self):
        """Test toggling the files and music services on/off."""
        self.mocker.replay()
        dialog = self.u1prefs.UbuntuOneDialog(keyring=self.keyring)
        self.assertTrue(dialog is not None)
        def files_toggled(checkbutton):
            enabled = checkbutton.get_active()
            if enabled:
                dialog.music_check.set_sensitive(True)
            else:
                dialog.music_check.set_sensitive(False)

        def music_toggled(checkbutton):
            pass

        dialog.files_check_toggled = files_toggled
        dialog.music_check_toggled = music_toggled
        dialog.connect_file_sync_callbacks()

        dialog.files_check.set_active(False)
        self.assertFalse(dialog.files_check.get_active())
        self.assertFalse(dialog.music_check.props.sensitive)
        dialog.files_check.set_active(True)
        self.assertTrue(dialog.files_check.get_active())
        self.assertTrue(dialog.music_check.props.sensitive)
        dialog.music_check.set_active(True)
        self.assertTrue(dialog.music_check.get_active())
        dialog.music_check.set_active(False)
        self.assertFalse(dialog.music_check.get_active())
        dialog.music_check.set_active(True)
        self.assertTrue(dialog.music_check.get_active())
        dialog.files_check.set_active(False)
        self.assertFalse(dialog.files_check.get_active())
        self.assertFalse(dialog.music_check.props.sensitive)
        dialog.destroy()
        return defer.DeferredList(dialog.sdtool._deferreds)

    def test_login_check(self):
        """Test that our login check works correctly."""
        self.mocker.replay()
        def got_new_creds(realm=None, consumer_key=None, sender=None):
            """ Override the callback """
            d.callback(True)

        def got_auth_denied():
            """ Override the callback """
            d.errback(Failure(InvalidSignalError()))

        def got_oauth_error(message=None):
            """ Override the callback """
            d.errback(Failure(InvalidSignalError()))

        def got_dbus_error(error):
            """Override the dbus error handler."""
            d.errback(Failure(error))

        d = defer.Deferred()

        login = self.u1prefs.UbuntuoneLoginHandler(dialog=None)
        login.got_newcredentials = got_new_creds
        login.got_authdenied = got_auth_denied
        login.got_oautherror = got_oauth_error
        login.got_dbus_error = got_dbus_error
        login.register_signal_handlers()
        self.u1prefs.do_login_request(bus=self.bus,
                                      error_handler=got_dbus_error)

        return d
