# ubuntuone.syncdaemon.volume_manager - manages volumes
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
""" The all mighty Volume Manager """
from __future__ import with_statement

import logging
import os
import re
import shutil
import stat
import sys
from contextlib import contextmanager
from itertools import ifilter

from ubuntuone.storageprotocol import request
from ubuntuone.syncdaemon.marker import MDMarker
from ubuntuone.syncdaemon import file_shelf


class Share(object):
    """Representas a share or mount point"""

    def __init__(self, path, share_id=request.ROOT, name=None,
                 access_level='View', accepted=False, other_username=None,
                 other_visible_name=None, subtree=None):
        """ Creates the instance.

        The received path should be 'bytes'
        """
        if path is None:
            self.path = None
        else:
            self.path = os.path.normpath(path)
        self.id = str(share_id)
        self.access_level = access_level
        self.accepted = accepted
        self.name = name
        self.other_username = other_username
        self.other_visible_name = other_visible_name
        self.subtree = subtree
        self.free_bytes = None

    @classmethod
    def from_response(cls, share_response, path):
        """ Creates a Share instance from a ShareResponse.

        The received path should be 'bytes'
        """
        share = cls(path, str(share_response.id), share_response.name,
                    share_response.access_level, share_response.accepted,
                    share_response.other_username,
                    share_response.other_visible_name, share_response.subtree)
        return share

    @classmethod
    def from_notify_holder(cls, share_notify, path):
        """ Creates a Share instance from a NotifyShareHolder.

        The received path should be 'bytes'
        """
        share = cls(path, share_id=str(share_notify.share_id),
                    name=share_notify.share_name,
                    access_level=share_notify.access_level,
                    other_username=share_notify.from_username,
                    other_visible_name=share_notify.from_visible_name,
                    subtree=share_notify.subtree)
        return share

    def can_write(self):
        """ check the access_level of this share,
        returns True if it's 'Modify'.
        """
        return self.access_level == 'Modify'


class VolumeManager(object):
    """Manages shares and mount points."""

    METADATA_VERSION = '5'

    def __init__(self, main):
        """Create the instance and populate the shares/d attributes
        from the metadata (if it exists).
        """
        self.log = logging.getLogger('ubuntuone.SyncDaemon.VM')
        self.m = main
        self._data_dir = os.path.join(self.m.data_dir, 'vm')
        self._shares_dir = os.path.join(self._data_dir, 'shares')
        self._shared_dir = os.path.join(self._data_dir, 'shared')
        self._version_file = os.path.join(self._data_dir, '.version')

        if not os.path.exists(self._data_dir):
            # first run, the data dir don't exist. No metadata to upgrade
            md_version = VolumeManager.METADATA_VERSION
            os.makedirs(self._data_dir)
            self._update_metadata_version()
        elif os.path.exists(self._version_file):
            with open(self._version_file) as fh:
                md_version = fh.read().strip()
            if not md_version:
                # we don't have a version of the metadata but a .version file?
                # assume it's None and do an upgrade from version 0
                md_version = None
        else:
            md_version = None

        # upgrade the metadata
        if md_version != VolumeManager.METADATA_VERSION:
            upgrade_method = getattr(self, "_upgrade_metadata_%s" % md_version)
            upgrade_method(md_version)
        if not os.path.exists(self._shares_dir):
            os.makedirs(self._shares_dir)
        if not os.path.exists(self._shared_dir):
            os.makedirs(self._shared_dir)

        # build the dir layout
        if not os.path.exists(self.m.root_dir):
            self.log.debug('creating root dir: %r', self.m.root_dir)
            os.makedirs(self.m.root_dir)
        if not os.path.exists(self.m.shares_dir):
            self.log.debug('creating shares directory: %r', self.m.shares_dir)
            os.makedirs(self.m.shares_dir)
        # create the shares symlink
        if not os.path.exists(self.m.shares_dir_link):
            self.log.debug('creating Shares symlink: %r -> %r',
                           self.m.shares_dir_link, self.m.shares_dir)
            os.symlink(self.m.shares_dir, self.m.shares_dir_link)
        # make the shares_dir read only
        os.chmod(self.m.shares_dir, 0555)
        # make the root read write
        os.chmod(self.m.root_dir, 0775)

        self.shares = ShareFileShelf(self._shares_dir)
        self.shared = ShareFileShelf(self._shared_dir)
        if self.shares.get(request.ROOT) is None:
            self.root = Share(self.m.root_dir)
        else:
            self.root = self.shares[request.ROOT]
        self.root.access_level = 'Modify'
        self.root.path = self.m.root_dir
        self.shares[request.ROOT] = self.root
        self.marker_share_map = {}
        self.list_shares_retries = 0
        self.retries_limit = 5

    def init_root(self):
        """ Creates the root mdid. """
        self.log.debug('init_root')
        self._create_share_dir(self.root)
        try:
            self.m.fs.get_by_path(self.root.path)
        except KeyError:
            self.m.fs.create(path=self.root.path,
                             share_id=request.ROOT, is_dir=True)

    def on_server_root(self, root):
        """Asociate server root"""
        self.log.debug('on_server_root(%s)', root)
        mdobj = self.m.fs.get_by_path(self.root.path)
        if getattr(mdobj, 'node_id', None) is None:
            self.m.fs.set_node_id(self.root.path, root)
        share = self.shares[request.ROOT]
        self.root.subtree = share.subtree = root
        self.shares[request.ROOT] = share
        self.m.action_q.inquire_account_info()
        self.m.action_q.inquire_free_space(request.ROOT)
        self.refresh_shares()
        return mdobj.mdid

    def refresh_shares(self):
        """ Reuqest the list of shares to the server. """
        # request the list of shares
        self.m.action_q.list_shares()

    def handle_SYS_CONNECTION_MADE(self):
        """
        The system is connected!. If we don't know our root's uuid yet, ask
        for it.
        """
        share = self.shares[request.ROOT]
        if share.subtree is None:
            mdobj = self.m.fs.get_by_path(share.path)
            self.m.get_root(MDMarker(mdobj.mdid))
        else:
            self.on_server_root(share.subtree)

    def handle_AQ_SHARES_LIST(self, shares_list):
        """ handle AQ_SHARES_LIST event """
        self.log.debug('handling shares list: ')
        self.list_shares_retries = 0
        shares = []
        shared = []
        for a_share in shares_list.shares:
            share_id = getattr(a_share, 'id',
                               getattr(a_share, 'share_id', None))
            self.log.debug('share %r: id=%s, name=%r', a_share.direction,
                           share_id, a_share.name)
            if a_share.direction == "to_me":
                dir_name = self._build_dirname(a_share.name,
                                               a_share.other_visible_name)
                path = os.path.join(self.m.shares_dir, dir_name)
                share = Share.from_response(a_share, path)
                shares.append(share.id)
                self.add_share(share)
            elif a_share.direction == "from_me":
                try:
                    mdobj = self.m.fs.get_by_node_id("", a_share.subtree)
                    path = self.m.fs.get_abspath(mdobj.share_id, mdobj.path)
                except KeyError:
                    # we don't have the file/md of this shared subtree yet
                    # for the moment ignore this share
                    self.log.warning("we got a share with 'from_me' direction,"
                            " but don't have the node_id in the metadata yet")
                    path = None
                share = Share.from_response(a_share, path)
                shared.append(share.id)
                self.add_shared(share)

        # housekeeping of the shares and shared shelf's each time we get the
        # list of shares
        self.log.debug('deleting dead shares')
        for share in ifilter(lambda item: item and item not in shares,
                             self.shares):
            self.log.debug('deleting share: id=%s', share)
            self.share_deleted(share)
        for share in ifilter(lambda item: item and item not in shared,
                             self.shared):
            self.log.debug('deleting shared: id=%s', share)
            del self.shared[share]

    def _build_dirname(self, share_name, visible_name):
        '''Builds the root path using the share information.'''
        dir_name = share_name + u' from ' + visible_name

        # Unicode boundary! the name is Unicode in protocol and server,
        # but here we use bytes for paths
        dir_name = dir_name.encode("utf8")
        return dir_name

    def handle_AQ_LIST_SHARES_ERROR(self, error):
        """ handle AQ_LIST_SHARES_ERROR event """
        # just call list_shares again, until we reach the retry limit
        if self.list_shares_retries <= self.retries_limit:
            self.m.action_q.list_shares()
            self.list_shares_retries += 1

    def handle_SV_FREE_SPACE(self, share_id, free_bytes):
        """ handle SV_FREE_SPACE event """
        self.update_free_space(str(share_id), free_bytes)

    def handle_SV_SHARE_CHANGED(self, message, info):
        """ handle SV_SHARE_CHANGED event """
        if message == 'changed':
            if str(info.share_id) not in self.shares:
                self.log.debug("New share notification, share_id: %s",
                         info.share_id)
                dir_name = self._build_dirname(info.share_name,
                                               info.from_visible_name)
                path = os.path.join(self.m.shares_dir, dir_name)
                share = Share.from_notify_holder(info, path)
                self.add_share(share)
            else:
                self.log.debug('share changed! %s', info.share_id)
                self.share_changed(info)
        elif message == 'deleted':
            self.log.debug('share deleted! %s', info.share_id)
            self.share_deleted(str(info.share_id))

    def handle_AQ_CREATE_SHARE_OK(self, share_id, marker):
        """ handle AQ_CREATE_SHARE_OK event. """
        share = self.marker_share_map.get(marker)
        if share is None:
            self.m.action_q.list_shares()
        else:
            share.id = share_id
            self.add_shared(share)
            if marker in self.marker_share_map:
                del self.marker_share_map[marker]

    def handle_AQ_CREATE_SHARE_ERROR(self, marker, error):
        """ handle AQ_CREATE_SHARE_ERROR event. """
        if marker in self.marker_share_map:
            del self.marker_share_map[marker]

    def handle_SV_SHARE_ANSWERED(self, share_id, answer):
        """ handle SV_SHARE_ANSWERED event. """
        share = self.shared.get(share_id, None)
        if share is None:
            # oops, we got an answer for a share we don't have,
            # probably created from the web.
            # refresh the share list
            self.refresh_shares()
        else:
            share.accepted = True if answer == 'Yes' else False
            self.shared[share_id] = share

    def handle_AQ_ANSWER_SHARE_OK(self, share_id, answer):
        """ Handle successfully accepting a share """
        if answer == 'Yes':
            share = self.shares[share_id]
            self._create_fsm_object(share)
            self._create_share_dir(share)
            self.m.action_q.query([(share.id, str(share.subtree), "")])
            self.m.action_q.inquire_free_space(share.id)


    def add_share(self, share):
        """ Add a share to the share list, and creates the fs mdobj. """
        self.log.info('Adding new share with id: %s - path: %r',
                      share.id, share.path)
        if share.id in self.shares:
            del self.shares[share.id]
        self.shares[share.id] = share
        if share.accepted:
            self._create_fsm_object(share)
            self._create_share_dir(share)
            self.m.action_q.query([(share.id, str(share.subtree), "")])
            self.m.action_q.inquire_free_space(share.id)

    def update_free_space(self, share_id, free_bytes):
        """ Update free space for a given share."""
        if share_id in self.shares:
            share = self.shares[share_id]
            share.free_bytes = free_bytes

    def accept_share(self, share_id, answer):
        """ Calls AQ.accept_share with answer ('Yes'/'No')."""
        self.log.debug("Accept share, with id: %s - answer: %s ",
                       share_id, answer)
        share = self.shares[share_id]
        share.accepted = answer
        self.shares[share_id] =  share
        answer_str = "Yes" if answer else "No"
        self.m.action_q.answer_share(share_id, answer_str)

    def share_deleted(self, share_id):
        """ process the share deleted event. """
        self.log.debug("Share (id: %s) deleted. ", share_id)
        share = self.shares.get(share_id, None)
        if share is None:
            # we don't have this share, ignore it
            self.log.warning("Got a share deleted notification (%r), "
                             "but don't have the share", share_id)
        else:
            self._delete_fsm_object(share)
            del self.shares[share_id]

    def share_changed(self, share_holder):
        """ process the share changed event """
        # the holder id is a uuid, use the str
        share = self.shares.get(str(share_holder.share_id), None)
        if share is None:
            # we don't have this share, ignore it
            self.log.warning("Got a share changed notification (%r), "
                             "but don't have the share", share_holder.share_id)
        else:
            share.access_level = share_holder.access_level
            self.shares[share.id] = share

    def _create_share_dir(self, share):
        """ Creates the share root dir, and set the permissions. """
        # XXX: verterok: This is going to be moved into fsm
        # if the share don't exists, create it
        if not os.path.exists(share.path):
            with allow_writes(os.path.dirname(share.path)):
                os.mkdir(share.path)
            # add the watch after the mkdir
            if share.can_write():
                self.log.debug('adding inotify watch to: %s', share.path)
                self.m.event_q.inotify_add_watch(share.path)
        # if it's a ro share, change the perms
        if not share.can_write():
            os.chmod(share.path, 0555)

    def _create_fsm_object(self, share):
        """ Creates the mdobj for this share in fs manager. """
        try:
            self.m.fs.get_by_path(share.path)
        except KeyError:
            self.m.fs.create(path=share.path, share_id=share.id, is_dir=True)
            self.m.fs.set_node_id(share.path, share.subtree)

    def _delete_fsm_object(self, share):
        """ Deletes the share and it files/folders metadata from fsm. """
        #XXX: partially implemented, this should be moved into fsm?.
        # should delete all the files in the share?
        if share.can_write():
            try:
                self.m.event_q.inotify_rm_watch(share.path)
            except (ValueError, RuntimeError, TypeError), e:
                # pyinotify has an ugly error management, if we can call
                # it that, :(. We handle this here because it's possible
                # and correct that the path is not there anymore
                self.log.warning("Error %s when trying to remove the watch"
                                 " on %r", e, share.path)
        # delete all the metadata but dont touch the files/folders
        # pylint: disable-msg=W0612
        for path, is_dir in self.m.fs.get_paths_starting_with(share.path):
            self.m.fs.delete_metadata(path)

    def create_share(self, path, username, name, access_level):
        """ create a share for the specified path, username, name """
        self.log.debug('create share(%r, %s, %s, %s)',
                       path, username, name, access_level)
        mdobj = self.m.fs.get_by_path(path)
        mdid = mdobj.mdid
        marker = MDMarker(mdid)
        share = Share(self.m.fs.get_abspath("", mdobj.path), share_id=marker,
                      name=name, access_level=access_level,
                      other_username=username, other_visible_name=None,
                      subtree=mdobj.node_id)
        self.marker_share_map[marker] = share
        self.m.action_q.create_share(mdobj.node_id, username, name,
                                     access_level, marker)

    def add_shared(self, share):
        """ Add a share with direction == from_me """
        self.log.info('New shared subtree: id: %s - path: %r',
                      share.id, share.path)
        current_share = self.shared.get(share.id)
        if current_share is None:
            self.shared[share.id] = share
        else:
            for k in share.__dict__:
                setattr(current_share, k, getattr(share, k))
            self.shared[share.id] = current_share

    def _upgrade_metadata_None(self, md_version):
        """Upgrade the shelf layout, for *very* old clients."""
        self.log.debug('Upgrading the share shelf layout')
        # the shelf already exists, and don't have a .version file
        # first backup the old data
        backup = os.path.join(self._data_dir, '0.bkp')
        if not os.path.exists(backup):
            os.makedirs(backup)
        # pylint: disable-msg=W0612
        # filter 'shares' and 'shared' dirs, in case we are in the case of
        # missing version but existing .version file
        filter_known_dirs = lambda d: d != os.path.basename(self._shares_dir) \
                and d != os.path.basename(self._shared_dir)
        for dirname, dirs, files in os.walk(self._data_dir):
            if dirname == self._data_dir:
                for dir in filter(filter_known_dirs, dirs):
                    if dir != os.path.basename(backup):
                        shutil.move(os.path.join(dirname, dir),
                                    os.path.join(backup, dir))
        # add the old module FQN to sys.modules in order to load the metadata
        sys.modules['canonical.ubuntuone.storage.syncdaemon.volume_manager'] = \
                sys.modules['ubuntuone.syncdaemon.volume_manager']
        # regenerate the shelf using the new layout using the backup as src
        old_shelf = ShareFileShelf(backup)
        if not os.path.exists(self._shares_dir):
            os.makedirs(self._shares_dir)
        new_shelf = ShareFileShelf(self._shares_dir)
        for key in old_shelf.keys():
            new_shelf[key] = old_shelf[key]
        # undo the change to sys.modules
        del sys.modules['canonical.ubuntuone.storage.syncdaemon.volume_manager']
        # now upgrade to metadata 3
        self._upgrade_metadata_2(md_version)
        self._upgrade_metadata_3(md_version)
        self._update_metadata_version()

    def _upgrade_metadata_1(self, md_version):
        """ Upgrade to version 2. upgrade all pickled Share to the new
        package/module layout
        """
        self.log.debug('upgrading share shelfs from metadata 1')
        # add the old module FQN to sys.modules in order to load the metadata
        sys.modules['canonical.ubuntuone.storage.syncdaemon.volume_manager'] = \
                sys.modules['ubuntuone.syncdaemon.volume_manager']
        shares = ShareFileShelf(self._shares_dir)
        for key in shares.keys():
            shares[key] = shares[key]
        shared = ShareFileShelf(self._shared_dir)
        for key in shared.keys():
            shared[key] = shared[key]
        # undo the change to sys.modules
        del sys.modules['canonical.ubuntuone.storage.syncdaemon.volume_manager']
        # now upgrade to metadata 3
        self._upgrade_metadata_2(md_version)
        self._update_metadata_version()

    def _upgrade_metadata_2(self, md_version):
        """
        Upgrade to version 3

        renames foo.conflict files to foo.u1conflict, foo.conflict.N
        to foo.u1conflict.N, foo.partial to .u1partial.foo, and
        .partial to .u1partial
        """
        self.log.debug('upgrading from metadata 2 (bogus)')
        for top in self.m.root_dir, self.m.shares_dir:
            for dirpath, dirnames, filenames in os.walk(top):
                with allow_writes(dirpath):
                    for names in filenames, dirnames:
                        self._upgrade_names(dirpath, names)
        self._upgrade_metadata_3(md_version)
        self._update_metadata_version()

    def _upgrade_names(self, dirpath, names):
        """
        Do the actual renaming for _upgrade_metadata_2
        """
        for pos, name in enumerate(names):
            new_name = name
            if re.match(r'.*\.partial$|\.u1partial(?:\..+)?', name):
                if name == '.partial':
                    new_name = '.u1partial'
                else:
                    new_name = re.sub(r'^(.+)\.partial$',
                                      r'.u1partial.\1', name)
                if new_name != name:
                    while os.path.lexists(os.path.join(dirpath, new_name)):
                        # very, very strange
                        self.log.warning('Found a .partial and .u1partial'
                                         ' for the same file: %s!' % new_name)
                        new_name += '.1'
            elif re.search(r'\.(?:u1)?conflict(?:\.\d+)?$', name):
                new_name = re.sub(r'^(.+)\.conflict((?:\.\d+)?)$',
                                  r'\1.u1conflict\2', name)
                if new_name != name:
                    while os.path.lexists(os.path.join(dirpath, new_name)):
                        m = re.match(r'(.*\.u1conflict)((?:\.\d+)?)$', new_name)
                        base, num = m.groups()
                        if not num:
                            num = '.1'
                        else:
                            num = '.' + str(int(num[1:])+1)
                        new_name = base + num
            if new_name != name:
                old_path = os.path.join(dirpath, name)
                new_path = os.path.join(dirpath, new_name)
                self.log.debug('renaming %r to %r' % (old_path, new_path))
                os.rename(old_path, new_path)
                names[pos] = new_name


    def _upgrade_metadata_3(self, md_version):
        """
        Upgrade to version 4 (new layout!)
        move "~/Ubuntu One/Shared With" Me to XDG_DATA/ubuntuone/shares
        move "~/Ubuntu One/My Files" contents to "~/Ubuntu One"
        """
        self.log.debug('upgrading from metadata 3 (new layout)')
        old_share_dir = os.path.join(self.m.root_dir, 'Shared With Me')
        old_root_dir = os.path.join(self.m.root_dir, 'My Files')
        # change permissions
        os.chmod(self.m.root_dir, 0775)
        # update the path's in metadata and move the folder
        if os.path.exists(old_share_dir) and not os.path.islink(old_share_dir):
            os.chmod(old_share_dir, 0775)
            if not os.path.exists(os.path.dirname(self.m.shares_dir)):
                os.makedirs(os.path.dirname(self.m.shares_dir))
            self.log.debug('moving shares dir from: %r to %r',
                           old_share_dir, self.m.shares_dir)
            shutil.move(old_share_dir, self.m.shares_dir)
        # update the shares metadata
        shares = ShareFileShelf(self._shares_dir)
        for key in shares.keys():
            share = shares[key]
            if share.path is not None:
                share.path = share.path.replace(old_share_dir,
                                                self.m.shares_dir)
                shares[key] = share

        shared = ShareFileShelf(self._shared_dir)
        for key in shared.keys():
            share = shared[key]
            if share.path is not None:
                share.path = share.path.replace(old_root_dir, self.m.root_dir)
            shared[key] = share
        # move the My Files contents, taking care of dir/files with the same
        # in the new root
        if os.path.exists(old_root_dir):
            self.log.debug('moving My Files contents to the root')
            # make My Files rw
            os.chmod(old_root_dir, 0775)
            path_join = os.path.join
            for relpath in os.listdir(old_root_dir):
                old_path = path_join(old_root_dir, relpath)
                new_path = path_join(self.m.root_dir, relpath)

                if os.path.exists(new_path):
                    shutil.move(new_path, new_path+'.u1conflict')
                if relpath == 'Shared With Me':
                    # remove the Shared with Me symlink inside My Files!
                    self.log.debug('removing shares symlink from old root')
                    os.remove(old_path)
                else:
                    self.log.debug('moving %r to %r', old_path, new_path)
                    shutil.move(old_path, new_path)
            self.log.debug('removing old root: %r', old_root_dir)
            os.rmdir(old_root_dir)

        self._upgrade_metadata_4(md_version)
        # update the .version file
        self._update_metadata_version()

    def _upgrade_metadata_4(self, md_version):
        """
        Upgrade to version 5 (fix the broken symlink!)
        """
        self.log.debug('upgrading from metadata 4 (broken symlink!)')
        if os.path.islink(self.m.shares_dir_link):
            target = os.readlink(self.m.shares_dir_link)
            if os.path.normpath(target) == self.m.shares_dir_link:
                # the symnlink points to itself
                self.log.debug('removing broken shares symlink: %r -> %r',
                               self.m.shares_dir_link, target)
                os.remove(self.m.shares_dir_link)

        self._update_metadata_version()

    def _update_metadata_version(self):
        """write the version of the metadata"""
        if not os.path.exists(os.path.dirname(self._version_file)):
            os.makedirs(os.path.dirname(self._version_file))
        with open(self._version_file, 'w') as fd:
            fd.write(VolumeManager.METADATA_VERSION)
            # make sure the data get to disk
            fd.flush()
            os.fsync(fd.fileno())


@contextmanager
def allow_writes(path):
    """ a very simple context manager to allow writting in RO dirs. """
    prev_mod = stat.S_IMODE(os.stat(path).st_mode)
    os.chmod(path, 0755)
    yield
    os.chmod(path, prev_mod)


class ShareFileShelf(file_shelf.FileShelf):
    """ Custom file shelf that allow request.ROOT as key, it's replaced
    by the string: root_node_id.
    """

    def __init__(self, *args, **kwargs):
        """ Create the instance. """
        super(ShareFileShelf, self).__init__(*args, **kwargs)
        self.key = 'root_node_id'

    def key_file(self, key):
        """ override default key_file, to handle key == request.ROOT"""
        if key == request.ROOT:
            key = self.key
        return super(ShareFileShelf, self).key_file(key)

    def keys(self):
        """ override default keys, to handle key == request.ROOT"""
        for key in super(ShareFileShelf, self).keys():
            if key == self.key:
                yield request.ROOT
            else:
                yield key

