# ubuntuone.syncdaemon.state - manages SyncDaemon state
#
# Author: John Lenton <john.lenton@canonical.com>
#
# Author: Rodney Dawes <rodney.dawes@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
The state of the SyncDaemon is managed here.
"""
import logging

from twisted.internet import reactor

from ubuntuone.syncdaemon import states

class SyncDaemonStateManager(object):
    """
    Keep track of the state of the sync daemon.
    """

    def __init__(self, main, timeout, max_timeouts):
        self.main = main
        self.timeout = timeout
        self.max_timeouts = max_timeouts
        self.num_timeouts = 0
        self.watchdog = None
        self.state = states.INIT
        self.main.event_q.subscribe(self)
        all_states = (o for o in vars(states).itervalues()
                      if isinstance(o, states.SyncDaemonState))
        self.interesting_events = set(sum((i.transitions.keys()
                                           for i in all_states), []))
        self.logger = logging.getLogger('ubuntuone.SyncDaemon.State')

    def handle_default(self, event, *args, **kwargs):
        """
        Forward interesting events on to the current state object.
        """
        if event not in self.interesting_events:
            return

        # this try/except catches everything on purpose
        # pylint: disable-msg=W0702
        try:
            if self.num_timeouts > self.max_timeouts:
                next_state = states.EXCESSIVE_TIMEOUTS
            else:
                next_state = self.state.next(event)
            if self.watchdog is not None:
                self.logger.debug('Event arrived: cancelling timeout watchdog')
                self.watchdog.cancel()
                self.watchdog = None
                self.num_timeouts = 0
            if next_state.is_handshake:
                self.logger.debug('Setting up timeout watchdog')
                self.watchdog = reactor.callLater(
                    self.timeout * (1 + self.num_timeouts / 2.),
                    self.handle_timeout)
            if next_state != self.state:
                if next_state.enter is not None:
                    next_state.enter(self.main)
            if self.state.is_connected and not next_state.is_connected:
                client = self.main.action_q.client
                if client is not None:
                    client.disconnect()
        except:
            next_state = states.UNKNOWN_ERROR
            self.logger.error('%s --[%s]--> ERROR!!!' % (self.state.name,
                                                         event),
                              exc_info=1)
        # pylint: enable-msg=W0702

        self.logger.debug('%s --[%s]--> %s' % (self.state.name,
                                               event,
                                               next_state.name))
        self.state = next_state
        self.main.event_q.push('SYS_STATE_CHANGED', state=next_state)

    def handle_timeout(self):
        """
        React to the timeout of a handshake state.
        """
        self.watchdog = None
        self.num_timeouts += 1
        self.main.event_q.push('SYS_HANDSHAKE_TIMEOUT')

    @property
    def name(self):
        """
        Report our name as the name of the current state.
        """
        return self.state.name

    def __repr__(self):
        """
        Return a textual representation of the object. Used for str()
        and repr() calls (so a print will show this)
        """
        return '<SyncDaemonStateManager [current state: %s]>' % self.state.name
    def __cmp__(self, other):
        """
        Compare according to the current state.
        """
        return cmp(self.state, other)
