# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT AN WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""Specific tests for the qt implementation."""

from PyQt4.QtNetwork import QSslError
from twisted.internet import defer
from ubuntuone.devtools.testcases import TestCase

from ubuntu_sso.utils.webclient import qtnetwork


class FakeQNetworkProxy(object):
    """A fake network proxy class."""

    def __init__(self, called):
        """Create a new instance."""
        self.called = called

    def __call__(self, *args, **kwargs):
        """Fake construnctor."""
        return self

    # pylint: disable=C0103
    def setApplicationProxy(self, proxy):
        """Set the application proxy."""
        self.called.append(('setApplicationProxy', proxy))
    # pylint: enable=C0103


class FakeNetworkProxyFactory(object):
    """A fake network proxy factory."""

    def __init__(self, called, proxy):
        """Create a new instance."""
        self.called = called
        self.proxy = proxy

    def __call__(self, *args, **kwargs):
        """Fake construnctor."""
        return self

    # pylint: disable=C0103
    def systemProxyForQuery(self, query):
        """Get the system proxy."""
        self.called.append(('systemProxyForQuery', query))
        return [self.proxy]
    # pylint: enable=C0103


class SetupProxyTestCase(TestCase):
    """Test the proxy setup."""

    @defer.inlineCallbacks
    def setUp(self):
        """Set the different tests."""
        yield super(SetupProxyTestCase, self).setUp()
        self.called = []
        self.proxy = 'fake_proxy'

        self.network_proxy = FakeQNetworkProxy(self.called)
        self.patch(qtnetwork, 'QNetworkProxy', self.network_proxy)

        self.network_proxy_factory = FakeNetworkProxyFactory(self.called,
                                                              self.proxy)
        self.patch(qtnetwork, 'QNetworkProxyFactory',
                                              self.network_proxy_factory)

        self.settings = dict(https=dict(username='user', password='pasword'))
        self.patch(qtnetwork.gsettings, 'get_proxy_settings',
                lambda: self.settings)

        def fake_build_proxy(settings):
            """Fake build proxy."""
            self.called.append(('build_proxy', settings))
            return self.proxy

        self.patch(qtnetwork, 'build_proxy', fake_build_proxy)
        qtnetwork.WebClient.proxy_instance = None
        self.patch(qtnetwork.QNetworkAccessManager, 'setProxy',
                lambda _, p: self.called.append(('setProxy', p)))
        self.addCleanup(self._clean_webclient_instance)

    def _set_old_platform(self, platform):
        """Set back the platform."""
        qtnetwork.sys.platform = platform

    def _clean_webclient_instance(self):
        """Set the webclient not to have a proxy."""
        qtnetwork.WebClient.proxy_instance = None


class SetupLinuxProxyTestCase(SetupProxyTestCase):
    """Test setting up the proxy."""

    @defer.inlineCallbacks
    def setUp(self):
        """Set the different tests."""
        yield super(SetupLinuxProxyTestCase, self).setUp()
        old_platform = qtnetwork.sys.platform
        qtnetwork.sys.platform = 'linux'
        self.addCleanup(self._set_old_platform, old_platform)

    def test_setup_proxy(self):
        """Test setting the proxy."""
        qtnetwork.WebClient()
        self.assertEqual(self.proxy, qtnetwork.WebClient.proxy_instance)
        self.assertIn(('setApplicationProxy', self.proxy), self.called)
        self.assertIn(('setProxy', self.proxy), self.called)
        self.assertIn(('build_proxy', self.settings), self.called)

    def test_setup_instance_present(self):
        """Test when the instanc is present."""
        # set the instance and assert we did not reset it
        qtnetwork.WebClient.proxy_instance = 'other_proxy'
        qtnetwork.WebClient()
        self.assertNotEqual(self.proxy, qtnetwork.WebClient.proxy_instance)
        self.assertNotIn(('setApplicationProxy', self.proxy), self.called)
        self.assertNotIn(('setProxy', self.proxy), self.called)
        self.assertNotIn(('build_proxy', self.settings), self.called)


class SetupWindowsProxyTestCase(SetupProxyTestCase):
    """Test setting up the proxy."""

    @defer.inlineCallbacks
    def setUp(self):
        """Set the different tests."""
        yield super(SetupWindowsProxyTestCase, self).setUp()
        old_platform = qtnetwork.sys.platform
        qtnetwork.sys.platform = 'win32'
        self.addCleanup(self._set_old_platform, old_platform)
        self.query = 'query'

        self.patch(qtnetwork, 'QNetworkProxyQuery', lambda _: self.query)

    def test_setup_proxy(self):
        """Test setting the proxy."""
        qtnetwork.WebClient()
        self.assertEqual(self.proxy, qtnetwork.WebClient.proxy_instance)
        self.assertIn(('systemProxyForQuery', self.query), self.called)
        self.assertIn(('setApplicationProxy', self.proxy), self.called)

    def test_setup_instance_present(self):
        """Test when the instanc is present."""
        qtnetwork.WebClient.proxy_instance = 'other_proxy'
        qtnetwork.WebClient()
        self.assertNotEqual(self.proxy, qtnetwork.WebClient.proxy_instance)
        self.assertNotIn(('systemProxyForQuery', self.query), self.called)
        self.assertNotIn(('setApplicationProxy', self.proxy), self.called)


class FakeQNetworkReply(object):
    """A fake QNetworkReply."""

    def __init__(self):
        """Initialize this fake."""
        url = qtnetwork.QUrl("http://fake_server")
        self._request = qtnetwork.QNetworkRequest(url)

    def request(self):
        """Return a test network request."""
        return self._request


class SSLErrorsTestCase(TestCase):
    """Tests for when SSL errors happen."""

    def test_handle_ssl_errors(self):
        """Test the _handle_ssl_errors method."""
        result = []
        self.patch(qtnetwork.logger, "error", result.append)
        client = qtnetwork.WebClient()
        reply = FakeQNetworkReply()
        errors = [QSslError()]
        # pylint: disable=W0212
        client._handle_ssl_errors(reply, errors)
        # pylint: enable=W0212
        self.assertTrue(type(result[0]), unicode)
