const { classes: Cc, interfaces: Ci, results: Cr, utils: Cu } = Components;

const NS_PREFBRANCH_PREFCHANGE_TOPIC_ID = "nsPref:changed";

Cu.import("resource://gre/modules/Services.jsm");
Cu.import("resource://gre/modules/AddonManager.jsm");
Cu.import("resource://gre/modules/XPCOMUtils.jsm");

XPCOMUtils.defineLazyGetter(this, "ctypes", function() {
  Cu.import("resource://gre/modules/ctypes.jsm");
  return ctypes;
});

var EXPORTED_SYMBOLS = ["require", "CTypesUtils", "getLogger", "SimpleObjectWrapper"];

var global = this;
var scopes = {};
var base;
var logPrefRoot;

function formatLogMessage(aType, aName, aStr, aException) {
  let message = aType.toUpperCase() + " " + aName + ": " + aStr;
  if (aException)
    return message + ": " + aException;
  return message;
}

function getStackDetails(aException) {
  // Defensively wrap all this to ensure that failing to get the message source
  // doesn't stop the message from being logged
  try {
    if (aException) {
      if (aException instanceof Ci.nsIException) {
        return {
          sourceName: aException.filename,
          lineNumber: aException.lineNumber
        };
      }

      return {
        sourceName: aException.fileName,
        lineNumber: aException.lineNumber
      };
    }

    let stackFrame = Components.stack.caller.caller.caller;
    return {
      sourceName: stackFrame.filename,
      lineNumber: stackFrame.lineNumber
    };
  }
  catch (e) {
    return {
      sourceName: null,
      lineNumber: 0
    };
  }
}

function Logger(aDomain, aTarget) {
  this.domain = aDomain;

  var self = this;
  ["log", "warn", "error"].forEach(function(aName) {
    upper = aName.toUpperCase();
    delete aTarget[upper];
    aTarget[upper] = function() {
      self[aName].apply(self, arguments);
    };
  }, this);

  this.setupPrefObserver();
}

Logger.prototype = {
  branch: null,
  domain: null,

  setupPrefObserver: function Logger_setupPrefObserver() {
    if (!logPrefRoot || this.branch) {
      return;
    }

    this.branch = Services.prefs.getBranch(logPrefRoot);
    this.branch.addObserver("", this, false);

    this.observe(this.branch, NS_PREFBRANCH_PREFCHANGE_TOPIC_ID, null);
  },

  observe: function Logger_observe(aSubject, aTopic, aData) {
    try {
      if (aSubject != this.branch) {
        throw Cr.NS_ERROR_FAILURE;
      }

      if (aTopic == NS_PREFBRANCH_PREFCHANGE_TOPIC_ID) {
        let domains = aSubject.getCharPref("domains").split(/\s*,\s*/);
        this._debugLogEnabled = (aSubject.getBoolPref("enabled") &&
                                 (domains.indexOf(this.domain) != -1 ||
                                  domains.indexOf("all") != -1));
      }
    } catch(e) { this._debugLogEnabled = false; }
  },

  _debugLogEnabled: false,
  get debugLogEnabled() {
    if (!this.branch) {
      this.setupPrefObserver();
      if (!this.branch) {
        return false;
      }
    }

    return this._debugLogEnabled;
  },

  log: function Logger_log(aStr, aException) {
    if (this.debugLogEnabled) {
      let message = formatLogMessage("log", this.domain, aStr, aException);
      Services.console.logStringMessage(message);
      dump("*** " + message + "\n");
    }
  },

  warn: function Logger_warn(aStr, aException) {
    let message = formatLogMessage("warn", this.domain, aStr, aException);

    let stack = getStackDetails(aException);

    let consoleMessage = Cc["@mozilla.org/scripterror;1"].
                         createInstance(Ci.nsIScriptError);
    consoleMessage.init(message, stack.sourceName, null, stack.lineNumber, 0,
                        Ci.nsIScriptError.warningFlag, "component javascript");
    Services.console.logMessage(consoleMessage);

    if (this.debugLogEnabled) {
      dump("*** " + message + "\n");
    }
  },

  error: function Logger_error(aStr, aException) {
    let message = formatLogMessage("error", this.domain, aStr, aException);

    let stack = getStackDetails(aException);

    let consoleMessage = Cc["@mozilla.org/scripterror;1"].
                         createInstance(Ci.nsIScriptError);
    consoleMessage.init(message, stack.sourceName, null, stack.lineNumber, 0,
                        Ci.nsIScriptError.errorFlag, "component javascript");
    Services.console.logMessage(consoleMessage);

    if (this.debugLogEnabled) {
      dump("*** " + message + "\n");
    }
  }
};

function logDomainFromSpec(aUri) {
  let uri = Services.io.newURI(aUri, null, null);
  return uri.host + uri.path.replace(/\..*$/, "").replace(/\//g, ".");
}

function absResourceUriSpec(aPrePath, aPath) {
  return Services.io.newURI(aPath, null,
                            Services.io.newURI(aPrePath, null, null)).spec;
}

function getLogger(aDomain, aTarget) {
  return new Logger(aDomain, aTarget);
}

function init() {
  let uri = Services.io.newURI(__URI__, null, null);
  if (uri.scheme != "resource") {
    throw Error("This only works properly from resource URI's at the moment");
  }

  base = uri.prePath;
  logPrefRoot = "extensions." + uri.host + ".logging.";
}

function _internal_require(aModule, aTarget) {
  if (!base) {
    throw Error("Failed to initialize utils.jsm");
  }

  if (typeof(aModule) != "string") {
    throw Error("Invalid module path");
  }

  if (!(aModule in scopes)) {
    try {
      let spec = absResourceUriSpec(base, aModule);

      // Set up basic global object
      scopes[aModule] = new Object();
      scopes[aModule].global = {
        Cc: Cc,
        Cu: Cu,
        Ci: Ci,
        Cr: Cr,

        EXPORTS: [],
        __URI__: spec
      };

      scopes[aModule].global.require = function(aName, aTarget) {
        let target = aTarget === undefined ? scopes[aModule].global :
                                             !aTarget ? null : aTarget;
        return _internal_require(aName, target);
      };

      // Add logging to the new global
      ["LOG", "WARN", "ERROR"].forEach(function(aName) {
        this.__defineGetter__(aName, function() {
          getLogger(logDomainFromSpec(this.__URI__), this);
          return this[aName];
        });

      }, scopes[aModule].global);

      scopes[aModule].global.__defineGetter__("ctypes", function() {
        return ctypes;
      });

      scopes[aModule].global.CTypesUtils = CTypesUtils;

      Services.scriptloader.loadSubScript(spec, scopes[aModule].global);

      scopes[aModule].exports = new Object();
      scopes[aModule].global.EXPORTS.forEach(function(aExport) {
        scopes[aModule].exports[aExport] = scopes[aModule].global[aExport];
      });
    } catch(e) {
      delete scopes[aModule];
      Cu.reportError(e);
      throw Error("Failed to import " + aModule + ". Check error console");
    }
  }

  if (aTarget) {
    scopes[aModule].global.EXPORTS.forEach(function(aExport) {
      aTarget[aExport] = scopes[aModule].global[aExport];
    });
  }

  return scopes[aModule].exports;
}

function CTypesLibrary(aName, aABIs, aDefines) {
  try {
    if (typeof(aName) != "string") {
      throw Error("Invalid library name");
    }

    if (typeof(aABIs) == "number") {
      aABIs = [ aABIs ];
    }

    if (typeof(aABIs) != "object") {
      throw Error("Invalid range of library ABI's");
    }

    if (typeof(aDefines) != "function") {
      throw Error("Invalid defines function");
    }

    var library;
    for each (let abi in aABIs) {
      let soname = "lib" + aName + ".so." + abi.toString();
      try {
        library = ctypes.open(soname);
        this.ABI = abi;
        break;
      } catch(e) {}
    }

    this.close = function() {
      library.close();
      this.ABI = -1;
    };

    this.available = function() {
      return this.ABI != -1;
    };

    if (!library) {
      this.ABI = -1;
      return;
    }

    var self = this;

    function abiMatch(aABIs) {
      if (typeof(aABIs) == "number") {
        aABIs = [ aABIs ];
      }

      return aABIs.some(function(aABI) {
        return self.ABI == aABI;
      });
    }

    function makeNotImplementedFunction(aSymbol) {
      return function() {
        throw Error("The function " + aSymbol + " was not found in lib" +
                    aName + ".so." + self.ABI.toString());
      }
    }

    let lib = {
      /**
       * Bind to a function from the native library
       *
       * @param  aSymbol
       *         The name of the function
       * @param  aRetType
       *         The native function return type
       * @param  aArgTypes (optional)
       *         An array of native argument types
       * @return A JS function which directly invokes the specified
       *         native function
       */
      declare: function(aSymbol, aRetType, aArgTypes) {
        try {
          let args = [];
          args.push(aSymbol);
          args.push(ctypes.default_abi);
          args.push(aRetType);
          if (aArgTypes) {
            aArgTypes.forEach(function(type) {
              args.push(type);
            });
          }
  
          return library.declare.apply(library, args);
        } catch (ex) {
          return makeNotImplementedFunction(aSymbol);
        }
      },

      /**
       * Bind to a function from the native library. The function is
       * attached to the resultant JS object using the specified symbol name
       *
       * @param  aSymbol
       *         The name of the function
       * @param  aRetType
       *         The native function return type
       * @param  aArgTypes (optional)
       *         An array of native argument types
       */
      bind: function(aSymbol, aRetType, aArgTypes) {
        self[aSymbol] = this.declare(aSymbol, aRetType, aArgTypes);
      },

      /**
       * Lazily bind to a function from the native library. The function
       * is attached to the resultant JS object using the specified symbol name
       *
       * @param  aSymbol
       *         The name of the function
       * @param  aRetType
       *         The native function return type
       * @param  aArgTypes (optional)
       *         An array of native argument types
       */
      lazy_bind: function(aSymbol, aRetType, aArgTypes) {
        XPCOMUtils.defineLazyGetter(self, aSymbol, function() {
          return lib.declare(aSymbol, aRetType, aArgTypes);
        });
      },

      /**
       * Lazily bind to a function from the native library, and only allow it
       * to be called via the specified wrapper function. This results in a
       * function being attached to the resultant JS object using the specified
       * symbol name.
       *
       * @param  aSymbol
       *         The name to use to expose the resulting function to JS
       * @param  aWrapper
       *         A wrapper function, which will invoke the native function.
       *         This is invoked with the native ctypes function as
       *         arguments[0], and the function parameters as arguments[1..n]
       * @param  aRetType
       *         The native function return type
       * @param  aArgTypes (optional)
       *         An array of native argument types
       * @param  aNativeSymbol (optional)
       *         The name of the function exposed by the native library.
       *         If omitted, use aSymbol
       */
      lazy_bind_with_wrapper: function(aSymbol, aWrapper,
                                       aRetType, aArgTypes, aNativeSymbol) {
        XPCOMUtils.defineLazyGetter(self, aSymbol, function() {
          var native = lib.declare(aNativeSymbol ? aNativeSymbol : aSymbol,
                                   aRetType, aArgTypes);
          return function() {
            var args = Array.prototype.slice.call(arguments, 0);
            args.unshift(native);

            return aWrapper.apply(self, args);
          }
        });
      },

      /**
       * Lazily bind to a function from the native library, but only for
       * the specified ABI versions. The function is attached to the
       * resultant JS object using the specified symbol name
       *
       * @param  aSymbol
       *         The name of the function
       * @param  aABIs
       *         The ABI versions for which to bind this function. Can
       *         be a number or list
       * @param  aRetType
       *         The native function return type
       * @param  aArgTypes (optional)
       *         An array of native argument types
       */
      lazy_bind_for_abis: function(aSymbol, aABIs, aRetType, aArgTypes) {
        if (abiMatch(aABIs)) {
          this.lazy_bind(aSymbol, aRetType, aArgTypes);
        } else if (!(aSymbol in self)) {
          self[aSymbol] = makeNotImplementedFunction(aSymbol);
        }
      },

     /**
       * Lazily bind to a function from the native library and only allow it
       * to be called via the specified wrapper function, but only for the
       * specified ABI versions. This results in a function being attached to
       * the resultant JS object using the specified symbol name.
       *
       * @param  aSymbol
       *         The name to use to expose the resulting function to JS
       * @param  aABIs
       *         The ABI versions for which to bind this function. Can
       *         be a number or list
       * @param  aWrapper
       *         A wrapper function, which will invoke the native function.
       *         This is invoked with the native ctypes function as
       *         arguments[0], and the function parameters as arguments[1..n]
       * @param  aRetType
       *         The native function return type
       * @param  aArgTypes (optional)
       *         An array of native argument types
       * @param  aNativeSymbol (optional)
       *         The name of the function exposed by the native library.
       *         If omitted, use aSymbol
       */
      lazy_bind_for_abis_with_wrapper: function(aSymbol, aABIs, aWrapper,
                                                aRetType, aArgTypes,
                                                aNativeSymbol) {
        if (abiMatch(aABIs)) {
          this.lazy_bind_with_wrapper(aSymbol, aWrapper, aRetType, aArgTypes,
                                      aNativeSymbol);
        } else if (!(aSymbol in self)) {
          self[aSymbol] = makeNotImplementedFunction(aSymbol);
        }
      },

      /**
       * Bind to a function from the native library, and only allow it
       * to be called via the specified wrapper function. This results in a
       * function being attached to the resultant JS object using the specified
       * symbol name.
       *
       * @param  aSymbol
       *         The name to use to expose the resulting function to JS
       * @param  aWrapper
       *         A wrapper function, which will invoke the native function.
       *         This is invoked with the native ctypes function as
       *         arguments[0], and the function parameters as arguments[1..n]
       * @param  aRetType
       *         The native function return type
       * @param  aArgTypes (optional)
       *         An array of native argument types
       * @param  aNativeSymbol (optional)
       *         The name of the function exposed by the native library.
       *         If omitted, use aSymbol
       */
      bind_with_wrapper: function(aSymbol, aWrapper, aRetType, aArgTypes,
                                  aNativeSymbol) {
        var native = this.declare(aNativeSymbol ? aNativeSymbol : aSymbol,
                                  aRetType, aArgTypes);
        self[aSymbol] = function() {
          var args = Array.prototype.slice.call(arguments, 0);
          args.unshift(native);

          return aWrapper.apply(self, args);
        }
      },

      /**
       * Bind to a function from the native library, but only for
       * the specified ABI versions. The function is attached to the
       * resultant JS object using the specified symbol name
       *
       * @param  aSymbol
       *         The name of the function
       * @param  aABIs
       *         The ABI versions for which to bind this function. Can
       *         be a number or list
       * @param  aRetType
       *         The native function return type
       * @param  aArgTypes (optional)
       *         An array of native argument types
       */
      bind_for_abis: function(aSymbol, aABIs, aRetType, aArgTypes) {
        if (abiMatch(aABIs)) {
          this.bind(aSymbol, aRetType, aArgTypes)
        } else if (!(aSymbol in self)) {
          self[aSymbol] = makeNotImplementedFunction(aSymbol);
        }
      }
    };

    aDefines.call(this, lib);
  } catch(e) {
    Cu.reportError(e);
    this.ABI = -1;
  }
}

init();

var CTypesUtils = {
  /**
   * Load a native library with ctypes and map native symbols to JS symbols
   *
   * @param  aName
   *         The library basename to load. This is the library SO name
   *         without the "lib" prefix, ".so" suffix or version suffix
   * @param  aABIs
   *         The library ABI's to try and load. This can be specified as
   *         an array, or an integer if you only maintain compatibility with
   *         one version. The first ABI in the list which exists on the
   *         system will be loaded
   * @param  aDefines
   *         A callback function to map JS symbols to native symbols. The
   *         callback will be called with the destination object as the "this"
   *         object, and a single parameter providing various helpers for
   *         binding functions
   * @return A JS object containing the resulting ctypes implementations
   */
  newLibrary: function CTypesUtils_newLibrary(aName, aABIs, aDefines) {
    return new CTypesLibrary(aName, aABIs, aDefines);
  },

  /**
   * Define a set of enums on the specified target object
   *
   * @param  aTarget
   *         The object on which to place the enums
   * @param  aName
   *         The C type name for this enum
   * @param  aStart
   *         The value of the first enum
   * @param  aEnums
   *         An array of strings
   */
  defineEnums: function CTypesUtils_defineEnums(aTarget, aName, aStart,
                                                 aEnums) {
    aTarget[aName] = ctypes.unsigned_int;

    aTarget[aName + "Enums"] = new Object();
    aEnums.forEach(function(aEnum) {
      XPCOMUtils.defineLazyGetter(aTarget[aName + "Enums"], aEnum, function() {
        return aEnums.indexOf(aEnum) + aStart;
      });
    });

    aTarget[aName + "Enums"].toEnum = function(aId) {
      return aEnums[aId - aStart];
    };
  },

  /**
   * Define a set of flags on the specified target object
   *
   * @param  aTarget
   *         The object on which to place the enums
   * @param  aName
   *         The C type name for this enum
   * @param  aStart
   *         The ln2 value of the first flag
   * @param  aFlags
   *         An array of strings
   */
  defineFlags: function CTypesUtils_defineFlags(aTarget, aName, aStart,
                                                aFlags) {
    aTarget[aName] = ctypes.unsigned_int;
    aTarget[aName + "Flags"] = new Object();
    aFlags.forEach(function(aFlag) {
      XPCOMUtils.defineLazyGetter(aTarget[aName + "Flags"], aFlag, function() {
        let i = aFlags.indexOf(aFlag) + aStart;
        return i == 0 ? 0 : 1 << i-1;
      });
    });
  },

  /**
   * Set up a ctype function template and create a helper function for
   * wrapping JS functions as ctype functions, for use as an async callback
   * in asynchronous API's
   *
   * @param  aObj
   *         The object to add the helper and function template to
   * @param  aName
   *         The type name of the callback. Note that the helper function
   *         will be "wrap<type>" (eg, "wrapGAsyncReadyCallback")
   * @param  aRetType
   *         The native return type of the async callback
   * @param  aArgTypes
   *         An array of native argument types to be passed to the callback
   */
  createAsyncCallbackWrapper: function CTypesUtils_createAsyncCallbackWrapper(aObj,
                                                                              aName,
                                                                              aRetType,
                                                                              aArgTypes) {
    var serial = 0;
    var callbacks = {};

    aObj[aName] = ctypes.FunctionType(ctypes.default_abi, aRetType,
                                      aArgTypes).ptr;
    aObj["wrap" + aName] = function(aCallback) {
      let cb = function() {
        try {
          return aCallback.apply(null, arguments);
        } catch(e) { Cu.reportError(e); }
        delete callbacks[id];
      };
      let ccb = aObj[aName](cb);

      while (callbacks[++serial]) {}
      var id = serial;
      callbacks[id] = ccb;

      return ccb;
    }
  }  
};

/**
 * Wrap the specified object, and return an object which only
 * exposes the properties listed in the source object's
 * "__exposedProps__" member
 *
 * @param  aObject
 *         Source object to wrap
 * @return Wrapped object
 */
function SimpleObjectWrapper(aObject) {

  if (!("__exposedProps__" in aObject)) {
    throw Error("__exposedProps missing from object");
  }

  function Wrapper() {
    aObject.__exposedProps__.forEach(function(prop) {

      // The property could be in our prototype chain
      let descriptor;
      let object = aObject;
      while (!descriptor && object) {
        descriptor = Object.getOwnPropertyDescriptor(object, prop);
        object = Object.getPrototypeOf(object);
      }

      if (!descriptor) {
        throw Error("Property " + prop + " was not found on object, or in " +
                    "objects prototype chain");
      }

      let d = {};
      if ("value" in descriptor && typeof(descriptor.value) == "function") {
        // Wrap the source function
        d.value = function() {
          return aObject[prop].apply(aObject, arguments);
        };
      } else {
        // Wrap the source getter/setter
        // For data values, add our own getter/setter on the wrapper
        if ("get" in descriptor || "value" in descriptor) {
          d.get = function() {
            return aObject[prop];
          };
        }

        if ("set" in descriptor || ("value" in descriptor &&
                                    descriptor.writable)) {
          d.set = function(aIn) {
            aObject[prop] = aIn;
          };
        }
      }

      if (!("value" in d) && !("get" in d) && !("set" in d)) {
        throw Error("Cannot wrap property " + prop +
                    " which has no value, getter or setter");
      }

      // "enumerable" is the only other attribute we care about. We
      // don't care about "configurable", as we're going to seal the object
      // anyway
      d.enumerable = descriptor.enumerable;
      Object.defineProperty(this, prop, d);
    }, this);

    Object.seal(this);
  }

  return new Wrapper();
}

/**
 * Load a JS module
 *
 * @param  aModule
 *         The path of the JS module to load, relative
 *         to the prePath component of our own URI
 * @param  aTarget (optional)
 *         Target object on which to add the modules exported
 *         symbols
 * @return An object containing the modules exported symbols
 */
function require(aModule, aTarget) {
  return _internal_require(aModule, aTarget);
}
