# Copyright (C) 2009-2013 Canonical, Ltd.
#
# This library is free software; you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License
# version 3.0 as published by the Free Software Foundation.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License version 3.0 for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library. If not, see
# <http://www.gnu.org/licenses/>.
"""The Ubuntu One Rhythmbox plugin."""

from __future__ import print_function, unicode_literals

import os

from dbus import SessionBus, DBusException
from dbus.mainloop.glib import DBusGMainLoop
from dirspec.utils import user_home
from gi.repository import Gio, GObject, Peas

DBusGMainLoop(set_as_default=True)

U1LIBRARYPATH = os.path.join(user_home, '.ubuntuone',
                             'Purchased from Ubuntu One')


class UbuntuOnePlugin (GObject.GObject, Peas.Activatable):
    """Ubuntu One integration for Rhythmbox."""
    __gtype_name__ = 'UbuntuOnePlugin'
    object = GObject.property(type=GObject.GObject)

    def __init__(self):
        GObject.GObject.__init__(self)
        self.db = None

        # Connect to the session bus
        self._signal = None
        try:
            self._bus = SessionBus()
        except DBusException:
            return

        # RhythmDB settings so we can handle changes
        self.rdbconf = Gio.Settings('org.gnome.rhythmbox.rhythmdb')
        self.rdbconf.connect('changed::locations', self._locations_changed)

    def _locations_changed(self, *args, **kwargs):
        """Handle the locations setting being changed."""
        libraries = self.rdbconf.get_strv('locations')
        library_uri = Gio.File.new_for_path(U1LIBRARYPATH).get_uri()
        if library_uri not in libraries:
            libraries.append(library_uri)
            self.rdbconf.set_strv('locations', libraries)
        # Remove the non-uri path if it exists
        if U1LIBRARYPATH in libraries:
            libraries.remove(U1LIBRARYPATH)
            self.rdbconf.set_strv('locations', libraries)
        # Remove the unescaped uri path if it exists
        unescaped_path = 'file://{0}'.format(U1LIBRARYPATH)
        if unescaped_path in libraries:
            libraries.remove(unescaped_path)
            self.rdbconf.set_strv('locations', libraries)

    def download_finished(self, path, info):
        """A file is finished downloading"""
        if not path.startswith(U1LIBRARYPATH):
            return

        # convert path to URI. Don't use urllib for this; Python and
        # glib escape URLs differently. gio does it the glib way.
        library_uri = Gio.File.new_for_path(path).get_uri()

        # Import the URI
        if not self.db.entry_lookup_by_location(library_uri):
            self.db.add_uri(library_uri)

    def do_activate(self):
        """Plug-in startup."""
        # Add the Ubuntu One purchased music directory if not already added
        self._locations_changed()
        # Get a reference to the db
        self.db = self.object.get_property('db')

        # Connect to the download finished from syncdaemon
        try:
            self._bus.add_signal_receiver(self.download_finished,
                                          signal_name='DownloadFinished')
        except DBusException:
            return

    def do_deactivate(self):
        """Plug-in shutdown."""
        del self.db

        # Disconnect the signal handler for downloads
        try:
            self._bus.remove_signal_receiver(self.download_finished,
                                             signal_name='DownloadFinished')
        except DBusException:
            return
