#!/usr/bin/env python
#
#
# Started 5/8/01 by Damien Elmes <resolve@repose.cx>
# This file is licensed under the GPL.
#

'''
pyosd - a wrapper of libxosd which allows the displaying of "on screen display"
         messages.

         example usage:

         | import pyosd
         | p = pyosd.osd()
         | # try fixed if the default_font gives you an error
         |
         | p.display("eschew obfuscation")
         | p.set_pos(pyosd.POS_BOT)
         | p.display("this will be down the bottom.")


         .. etc.

         Multiple instances can be used to display information in different
         fonts or colours across the screen.
         '''

import _pyosd
import re
import string

POS_TOP=0
POS_BOT=1

TYPE_STRING=0
TYPE_PERCENT=1
TYPE_SLIDER=2

error = _pyosd.error

default_font="-*-helvetica-medium-r-normal-*-*-360-*-*-p-*-*-*"

class osd:
    """ osd is a class used to create an object which can display messages on
    the screen. """

    def __init__(self, font=default_font, colour="#FFFFFF", timeout=3, \
                 pos=POS_TOP, offset=0, shadow=0, lines=2):
        """ Initialise the OSD library.

        This must be done before display() will work. It will automatically
        deinit if necessary.

        font(pyosd.default_font): standard string-style X font description
        colour('#FFFFFF'): standard string-style X colour description
        timeout(3): number of seconds to remain on screen (-1 for infinite)
        pos(POS_TOP): position, one of POS_TOP or POS_BOT
        offset(0): vertical offset from pos
        shadow(0): black shadow size
        lines(2): the max number of lines available to display at once

        Unfortunately due to constraints with the underlying library, lines has
        to be hard coded on init() at the moment. """

        self._max_lines = lines

        self._osd = _pyosd.init(font, colour, timeout, pos, offset, shadow,
        lines)

        # save this as we won't have access to it on del
        self._deinit = _pyosd.deinit


    def __del__(self):
        """ Shut down and clean up.

        Note that init() will automatically do this step if necessary. """

        # prevent python from dying if a user's silly enough to call this
        # directly
        if hasattr(self, '_osd'):
            self._deinit(self._osd)
            del self._osd

    def display(self, arg, type=TYPE_STRING, line=0):
        """ Display a string/bargraph/percentage using information from init()

        arg: a string or integer from 1-100, depending on the type
        -- defaults --
        type(TYPE_STRING): one of TYPE_STRING, TYPE_PERCENT, or TYPE_SLIDER
        line(0): the line to display text on

        The underlying library currently doesn't zero out previous lines that
        aren't being used, so if you wish to display something on, say, line 1,
        make sure you simultaneously display "" on line 0.
        """

        if line >= self._max_lines or line < 0:
            raise ValueError, "specified line is out of range"

        if type==TYPE_STRING:
            _pyosd.display_string(self._osd, line, arg)

        elif type==TYPE_PERCENT:
            _pyosd.display_perc(self._osd, line, arg)

        elif type==TYPE_SLIDER:
            _pyosd.display_slider(self._osd, line, arg)

        else:
            raise ValueError, "type not in list of valid values!"

    def set_font(self, font):
        """ Change the font.

        This takes effect immediately. """

        _pyosd.set_font(self._osd, font)

    def set_colour(self, c):
        """ Change the colour.

        This will update when display() is next called. """

        _pyosd.set_colour(self._osd, c)

    def set_timeout(self, t):
        """ Change the timeout.

        This takes effect immediately; anything that is currently displayed
        will wait the new timeout time before clearing. """

        _pyosd.set_timeout(self._osd, t)

    def set_pos(self, p):
        """ Change the position to the top or bottom.

        This change is immediate. """

        _pyosd.set_pos(self._osd, p)

    def set_offset(self, o):
        """ Change the vertical offset from the top or bottom.

        This change is immediate. """

        _pyosd.set_offset(self._osd, o)

    def set_shadow_offset(self, o):
        """ Set the offset of the shadow.

        This change will take effect on the next display() """

        _pyosd.set_shadow_offset(self._osd, o)
