#include <Python.h>

staticforward PyTypeObject CfuncType;

static void
cfunc_dealloc(PyObject* self)
{
    PyObject_Del(self);
}

static PyObject *
cfunc_repr(PyObject *self)
{
	char buf[256];
	CfuncObject *me = (CfuncObject *) self;
	sprintf(buf, "<cfunc '%s' at %08lx check-self:%d align:%d  io:(%d, %d)>", 
		 me->descr.name, (unsigned long ) me->descr.fptr, 
		 me->descr.chkself, me->descr.align, 
		 me->descr.wantIn, me->descr.wantOut);
	return PyString_FromString(buf);
}

/* Call a standard "stride" function
**
** Stride functions always take one input and one output array.
** They can handle n-dimensional data with arbitrary strides (of
** either sign) for both the input and output numarray. Typically
** these functions are used to copy data, byteswap, or align data.
**
**
** It expects the following arguments:
**
**   Number of iterations for each dimension as a tuple
**   Input Buffer Object
**   Offset in bytes for input buffer
**   Input strides (in bytes) for each dimension as a tuple
**   Output Buffer Object
**   Offset in bytes for output buffer
**   Output strides (in bytes) for each dimension as a tuple
**   An integer (Optional), typically the number of bytes to copy per
*       element.
**
** Returns None
**
** The arguments expected by the standard stride functions that this
** function calls are:
**
**   Number of dimensions to iterate over
**   Long int value (from the optional last argument to
**      callStrideConvCFunc)
**      often unused by the C Function
**   An array of long ints. Each is the number of iterations for each
**      dimension. NOTE: the previous argument as well as the stride
**      arguments are reversed in order with respect to how they are
**      used in Python. Fastest changing dimension is the first element
**      in the numarray!
**   A void pointer to the input data buffer.
**   The starting offset for the input data buffer in bytes (long int).
**   An array of long int input strides (in bytes) [reversed as with
**      the iteration array]
**   A void pointer to the output data buffer.
**   The starting offset for the output data buffer in bytes (long int).
**   An array of long int output strides (in bytes) [also reversed]
*/


static PyObject *
NA_callStrideConvCFuncCore(
	PyObject *self, int nshape, maybelong *shape,
	PyObject *inbuffObj,  long inboffset, 
	int ninbstrides, maybelong *inbstrides,
	PyObject *outbuffObj, long outboffset, 
	int noutbstrides, maybelong *outbstrides,
	long nbytes)
{
	CfuncObject *me = (CfuncObject *) self;
	CFUNC_STRIDE_CONV_FUNC funcptr;
	void *inbuffer, *outbuffer;
	long inbsize, outbsize;
	maybelong i, lshape[MAXDIM], in_strides[MAXDIM], out_strides[MAXDIM];
	maybelong shape_0, inbstr_0, outbstr_0;

	if (nshape == 0) {   /* handle rank-0 numarray. */
		nshape = 1;
		shape = &shape_0; 
		inbstrides = &inbstr_0;
		outbstrides = &outbstr_0;
		shape[0] = 1;
		inbstrides[0] = outbstrides[0] = 0;
	}

	for(i=0; i<nshape; i++) 
		lshape[i] = shape[nshape-1-i];
	for(i=0; i<nshape; i++) 
		in_strides[i] = inbstrides[nshape-1-i];
	for(i=0; i<nshape; i++) 
		out_strides[i] = outbstrides[nshape-1-i];
	
	if (!PyObject_IsInstance(self , (PyObject *) &CfuncType)
	    || me->descr.type != CFUNC_STRIDING)
		return PyErr_Format(PyExc_TypeError,
		      "NA_callStrideConvCFuncCore: problem with cfunc");
	
	if ((inbsize = NA_getBufferPtrAndSize(inbuffObj, 1, &inbuffer)) < 0)
		return PyErr_Format(_Error,
		      "%s: Problem with input buffer", me->descr.name);
	
	if ((outbsize = NA_getBufferPtrAndSize(outbuffObj, 0, &outbuffer)) < 0)
		return PyErr_Format(_Error,
		      "%s: Problem with output buffer (read only?)", 
				    me->descr.name);
	
	/* Check buffer alignment and bounds */
	if (NA_checkOneStriding(me->descr.name, nshape, lshape,
				inboffset, in_strides, inbsize,
				(me->descr.sizes[0] == -1) ? 
				nbytes : me->descr.sizes[0],
				me->descr.align) ||
	    NA_checkOneStriding(me->descr.name, nshape, lshape,
				outboffset, out_strides, outbsize,
				(me->descr.sizes[1] == -1) ? 
				nbytes : me->descr.sizes[1],
				me->descr.align))
		return NULL;
	
	/* Cast function pointer and perform stride operation */
	funcptr = (CFUNC_STRIDE_CONV_FUNC) me->descr.fptr;
	if ((*funcptr)(nshape-1, nbytes, lshape, 
		       inbuffer,  inboffset, in_strides,
		       outbuffer, outboffset, out_strides) == 0) {
		Py_INCREF(Py_None);
		return Py_None;
	} else {
		return NULL;
	}
}

static PyObject *
callStrideConvCFunc(PyObject *self, PyObject *args) {
    PyObject *inbuffObj, *outbuffObj, *shapeObj;
    PyObject *inbstridesObj, *outbstridesObj, *otemp;
    CfuncObject *me = (CfuncObject *) self;
    int  nshape, ninbstrides, noutbstrides, nargs, i;
    maybelong shape[MAXDIM], inbstrides[MAXDIM], outbstrides[MAXDIM];
    long inboffset, outboffset, nbytes=0;

    nargs = PyObject_Length(args);
    if (!PyArg_ParseTuple(args, "OOlOOlO|l",
            &shapeObj, &inbuffObj,  &inboffset, &inbstridesObj,
            &outbuffObj, &outboffset, &outbstridesObj,
            &nbytes)) {
        return PyErr_Format(_Error,
                 "%s: Problem with argument list",
		  me->descr.name);
    }

    if (!PySequence_Check(inbstridesObj))
	    return PyErr_Format(_Error, "%s: bad instrides.", me->descr.name);
    if (!PySequence_Check(outbstridesObj))
	    return PyErr_Format(_Error, "%s: bad outstrides.", me->descr.name);
    /* Check that iteration and stride tuples have the same # elements */
    nshape = PyObject_Length(shapeObj);
    ninbstrides = PyObject_Length(inbstridesObj);
    noutbstrides = PyObject_Length(outbstridesObj);
    if (nshape && (nshape != ninbstrides)) {
        return PyErr_Format(_Error,
            "%s: Missmatch between input iteration and strides tuples",
	    me->descr.name);
    }
    if (nshape && (nshape != noutbstrides)) {
	    PyObject *otemp = PySequence_GetItem(outbstridesObj, -1);
	    if (!otemp) return NULL;
	    if (!PyInt_Check(otemp))
		    return PyErr_Format(
			    PyExc_TypeError, "%s: non-integer stride.",
			    me->descr.name);
	    if (PyInt_AsLong(otemp)) /* allow 0 for reductions. */
		    return PyErr_Format(_Error,
					"%s: Missmatch between output "
					"iteration and strides tuples",
					me->descr.name);
    }
    for (i=0; i<nshape; i++) {
	    otemp = PySequence_GetItem(shapeObj, i);
	    if (!PyInt_Check(otemp) && !PyLong_Check(otemp))
		    return PyErr_Format(_Error, "%s: non-integer shape element.",
					me->descr.name);
	    shape[i] = PyInt_AsLong(otemp);
	    Py_DECREF(otemp);
	    otemp = PySequence_GetItem(inbstridesObj, i);
	    inbstrides[i] = PyInt_AsLong(otemp);
	    Py_DECREF(otemp);
	    otemp = PySequence_GetItem(outbstridesObj, i);
	    if (!PyInt_Check(otemp) && !PyLong_Check(otemp))
		    return PyErr_Format(_Error, "%s: non-integer stride element.",
					me->descr.name);
	    outbstrides[i] = PyInt_AsLong(otemp);
	    Py_DECREF(otemp);
    }
    return NA_callStrideConvCFuncCore(
	    self, nshape, shape,
	    inbuffObj,  inboffset,  ninbstrides, inbstrides,
	    outbuffObj, outboffset, noutbstrides, outbstrides, nbytes);
}

static int 
callStridingHelper(PyObject *aux, long dim, 
		   long nnumarray, PyArrayObject *numarray[], 
		   CFUNC_STRIDED_FUNC f)
{
	int i, j, status=0;
	dim -= 1;
	for(i=0; i<numarray[0]->dimensions[dim]; i++) {
		for (j=0; j<nnumarray; j++)
			numarray[j]->byteoffset += 
				numarray[j]->strides[dim]*i;
		if (dim == 0)
			status |= f(aux, nnumarray, numarray);
		else
			status |= callStridingHelper(aux, 
						     dim, nnumarray, numarray, f);
		for (j=0; j<nnumarray; j++)
			numarray[j]->byteoffset -= 
				numarray[j]->strides[dim]*i;
	}
	return status;
}


static PyObject *
callStridingCFunc(PyObject *self, PyObject *args) {
    CfuncObject *me = (CfuncObject *) self;
    PyObject *aux;
    PyArrayObject *numarray[MAXARRAYS];
    CFUNC_STRIDED_FUNC f;
    int i;

    int nnumarray = PySequence_Length(args)-1;
    if ((nnumarray < 1) || (nnumarray > MAXARRAYS))
	    return PyErr_Format(_Error, "%s, too many or too few numarray.",
				me->descr.name);

    aux = PySequence_GetItem(args, 0);
    if (!aux)
	    return NULL;

    for(i=0; i<nnumarray; i++) {
	    PyObject *otemp = PySequence_GetItem(args, i+1);
	    if (!otemp)
		    return PyErr_Format(_Error, "%s couldn't get array[%d]", 
					me->descr.name, i);
	    if (!NA_NDArrayCheck(otemp))
		    return PyErr_Format(PyExc_TypeError, 
				 "%s arg[%d] is not an array.",
				 me->descr.name, i);
	    numarray[i] = (PyArrayObject *) otemp;
	    Py_DECREF(otemp);
    }
	    
    /* Cast function pointer and perform stride operation */
    f = (CFUNC_STRIDED_FUNC) me->descr.fptr;
    
    if (callStridingHelper(aux, numarray[0]->nd, nnumarray, numarray, f)) {
	    return NULL;
    } else {
	    Py_INCREF(Py_None);
	    return Py_None;
    }
}

/* Convert a standard C numeric value to a Python numeric value.
**
** Handles both nonaligned and/or byteswapped C data.
**
** Input arguments are:
**
**   Buffer object that contains the C numeric value.
**   Offset (in bytes) into the buffer that the data is located at.
**   The size of the C numeric data item in bytes.
**   Flag indicating if the C data is byteswapped from the processor's
**     natural representation.
**
**   Returns a Python numeric value.
*/

static PyObject *
NumTypeAsPyValue(PyObject *self, PyObject *args) {
    PyObject *bufferObj;
    long offset, itemsize, byteswap, i, buffersize;
    Py_complex temp;  /* to hold copies of largest possible type */
    void *buffer;
    char *tempptr;
    CFUNCasPyValue funcptr;
    CfuncObject *me = (CfuncObject *) self;

    if (!PyArg_ParseTuple(args, "Olll", 
			  &bufferObj, &offset, &itemsize, &byteswap))
        return PyErr_Format(_Error,
		"NumTypeAsPyValue: Problem with argument list");

    if ((buffersize = NA_getBufferPtrAndSize(bufferObj, 1, &buffer)) < 0)
        return PyErr_Format(_Error,
                "NumTypeAsPyValue: Problem with array buffer");

    if (offset < 0)
	return PyErr_Format(_Error,
		"NumTypeAsPyValue: invalid negative offset: %d", (int) offset);

    /* Guarantee valid buffer pointer */
    if (offset+itemsize > buffersize)
	    return PyErr_Format(_Error,
		"NumTypeAsPyValue: buffer too small for offset and itemsize.");

    /* Do byteswapping.  Guarantee double alignment by using temp. */
    tempptr = (char *) &temp;
    if (!byteswap) {
        for (i=0; i<itemsize; i++)
            *(tempptr++) = *(((char *) buffer)+offset+i);
    } else {
        tempptr += itemsize-1;
        for (i=0; i<itemsize; i++)
            *(tempptr--) = *(((char *) buffer)+offset+i);
    }

    funcptr = (CFUNCasPyValue) me->descr.fptr;

    /* Call function to build PyObject.  Bad parameters to this function
       may render call meaningless, but "temp" guarantees that its safe.  */
    return (*funcptr)((void *)(&temp));
}

/* Convert a Python numeric value to a standard C numeric value.
**
** Handles both nonaligned and/or byteswapped C data.
**
** Input arguments are:
**
**   The Python numeric value to be converted.
**   Buffer object to contain the C numeric value.
**   Offset (in bytes) into the buffer that the data is to be copied to.
**   The size of the C numeric data item in bytes.
**   Flag indicating if the C data is byteswapped from the processor's
**     natural representation.
**
**   Returns None
*/

static PyObject *
NumTypeFromPyValue(PyObject *self, PyObject *args) {
    PyObject *bufferObj, *valueObj;
    long offset, itemsize, byteswap, i, buffersize;
    Py_complex temp;  /* to hold copies of largest possible type */
    void *buffer;
    char *tempptr;
    CFUNCfromPyValue funcptr;
    CfuncObject *me = (CfuncObject *) self;

    if (!PyArg_ParseTuple(args, "OOlll", 
		  &valueObj, &bufferObj, &offset, &itemsize, &byteswap)) 
        return PyErr_Format(_Error,
                 "%s: Problem with argument list", me->descr.name);

    if ((buffersize = NA_getBufferPtrAndSize(bufferObj, 0, &buffer)) < 0)
	    return PyErr_Format(_Error,
                "%s: Problem with array buffer (read only?)", me->descr.name);

    funcptr = (CFUNCfromPyValue) me->descr.fptr;

    /* Convert python object into "temp". Always safe. */
    if (!((*funcptr)(valueObj, (void *)( &temp))))
        return PyErr_Format(_Error,
		 "%s: Problem converting value", me->descr.name);

    /* Check buffer offset. */
    if (offset < 0)
	return PyErr_Format(_Error,
		"%s: invalid negative offset: %d", me->descr.name, (int) offset);

    if (offset+itemsize > buffersize)
	return PyErr_Format(_Error,
		"%s: buffer too small(%d) for offset(%d) and itemsize(%d)",
			me->descr.name, (int) buffersize, (int) offset, (int) itemsize);

    /* Copy "temp" to array buffer. */
    tempptr = (char *) &temp;
    if (!byteswap) {
        for (i=0; i<itemsize; i++)
            *(((char *) buffer)+offset+i) = *(tempptr++);
    } else {
        tempptr += itemsize-1;
        for (i=0; i<itemsize; i++)
            *(((char *) buffer)+offset+i) = *(tempptr--);
    }
    Py_INCREF(Py_None);
    return Py_None;
}

/* Function to call standard C Ufuncs
**
** The C Ufuncs expect contiguous 1-d data numarray, input and output numarray
** iterate with standard increments of one data element over all numarray.
** (There are some exceptions like arrayrangexxx which use one or more of
** the data numarray as parameter or other sources of information and do not
** iterate over every buffer).
**
** Arguments:
**
**   Number of iterations (simple integer value).
**   Number of input numarray.
**   Number of output numarray.
**   Tuple of tuples, one tuple per input/output array. Each of these
**     tuples consists of a buffer object and a byte offset to start.
**
** Returns None
*/

static PyObject *
NA_callCUFuncCore(PyObject *self, 
		  long niter, long ninargs, long noutargs,
		  PyObject **BufferObj, long *offset)
{
	CfuncObject *me = (CfuncObject *) self;
	char *buffers[MAXARGS];
	long bsizes[MAXARGS];
	long i, pnargs = ninargs + noutargs;
	UFUNC ufuncptr;

	if (pnargs > MAXARGS) 
		return PyErr_Format(PyExc_RuntimeError, "NA_callCUFuncCore: too many parameters");

	if (!PyObject_IsInstance(self, (PyObject *) &CfuncType)
	    || me->descr.type != CFUNC_UFUNC)
		return PyErr_Format(PyExc_TypeError,
		       "NA_callCUFuncCore: problem with cfunc.");
	
	for (i=0; i<pnargs; i++) {
		int readonly = (i < ninargs);
		if (offset[i] < 0)
			return PyErr_Format(_Error,
					    "%s: invalid negative offset:%d for buffer[%d]", 
					    me->descr.name, (int) offset[i], (int) i);
		if ((bsizes[i] = NA_getBufferPtrAndSize(BufferObj[i], readonly, 
							(void *) &buffers[i])) < 0)
			return PyErr_Format(_Error,
					    "%s: Problem with %s buffer[%d].", 
					    me->descr.name, 
					    readonly ? "read" : "write", (int) i);
		buffers[i] += offset[i];
		bsizes[i]  -= offset[i]; /* "shorten" buffer size by offset. */
	}
	
	ufuncptr = (UFUNC) me->descr.fptr;
	
	/* If it's not a self-checking ufunc, check arg count match,
	   buffer size, and alignment for all buffers */
	if (!me->descr.chkself && 
	    (NA_checkIo(me->descr.name, 
			me->descr.wantIn, me->descr.wantOut, ninargs, noutargs) ||
	     NA_checkNCBuffers(me->descr.name, pnargs, 
			       niter, (void **) buffers, bsizes, 
			       me->descr.sizes, me->descr.iters)))
		return NULL;
	
	/* Since the parameters are valid, call the C Ufunc */
	if (!(*ufuncptr)(niter, ninargs, noutargs, (void **)buffers, bsizes)) {
		Py_INCREF(Py_None);
		return Py_None;
	} else {
		return NULL;
	}
}

static PyObject *
callCUFunc(PyObject *self, PyObject *args) {
	PyObject *DataArgs, *ArgTuple;
	long pnargs, ninargs, noutargs, niter, i;
	CfuncObject *me = (CfuncObject *) self;
	PyObject *BufferObj[MAXARGS];
	long     offset[MAXARGS];
	
	if (!PyArg_ParseTuple(args, "lllO",
			      &niter, &ninargs, &noutargs, &DataArgs))
		return PyErr_Format(_Error,
				    "%s: Problem with argument list", me->descr.name);
	
	/* check consistency of stated inputs/outputs and supplied buffers */
	pnargs = PyObject_Length(DataArgs);
	if ((pnargs != (ninargs+noutargs)) || (pnargs > MAXARGS)) 
		return PyErr_Format(_Error,
				    "%s: wrong buffer count for function", me->descr.name);
	
	/* Unpack buffers and offsets, get data pointers */
	for (i=0; i<pnargs; i++) {
		ArgTuple = PySequence_GetItem(DataArgs, i);
		Py_DECREF(ArgTuple);
		if (!PyArg_ParseTuple(ArgTuple, "Ol", &BufferObj[i], &offset[i]))
			return PyErr_Format(_Error,
					    "%s: Problem with buffer/offset tuple", me->descr.name);
	}
	return NA_callCUFuncCore(self, niter, ninargs, noutargs, BufferObj, offset);
}


/* Handle "calling" the cfunc object at the python level. 
   Dispatch the call to the appropriate python-c wrapper based
   on the cfunc type.  Do this dispatch to avoid having to
   check that python code didn't somehow create a mismatch between
   cfunc and wrapper.
*/
static PyObject *
cfunc_call(PyObject *self, PyObject *argsTuple, PyObject *argsDict)
{
	CfuncObject *me = (CfuncObject *) self;
	switch(me->descr.type) {
	case CFUNC_UFUNC:
		return callCUFunc(self, argsTuple);
		break;
	case CFUNC_STRIDING:
		return callStrideConvCFunc(self, argsTuple);
		break;
	case CFUNC_NSTRIDING:
		return callStridingCFunc(self, argsTuple);
	case CFUNC_FROM_PY_VALUE:
		return NumTypeFromPyValue(self, argsTuple);
		break;
	case CFUNC_AS_PY_VALUE:
		return NumTypeAsPyValue(self, argsTuple);
		break;
	default:
		return PyErr_Format( _Error,
		     "cfunc_call: Can't dispatch cfunc '%s' with type: %d.", 
		     me->descr.name, me->descr.type);
	}
}

static PyTypeObject CfuncType = {
    PyObject_HEAD_INIT(NULL)
    0,
    "Cfunc",
    sizeof(CfuncObject),
    0,
    cfunc_dealloc, /*tp_dealloc*/
    0,          /*tp_print*/
    0,          /*tp_getattr*/
    0,          /*tp_setattr*/
    0,          /*tp_compare*/
    cfunc_repr, /*tp_repr*/
    0,          /*tp_as_number*/
    0,          /*tp_as_sequence*/
    0,          /*tp_as_mapping*/
    0,          /*tp_hash */
    cfunc_call, /* tp_call */
};


/* CfuncObjects are created at the c-level only.  They ensure that each
cfunc is called via the correct python-c-wrapper as defined by its 
CfuncDescriptor.  The wrapper, in turn, does conversions and buffer size
and alignment checking.  Allowing these to be created at the python level
would enable them to be created *wrong* at the python level, and thereby
enable python code to *crash* python. 
*/ 
static PyObject*
NA_new_cfunc(CfuncDescriptor *cfd)
{
    CfuncObject* cfunc;
    
    CfuncType.ob_type = &PyType_Type;  /* Should be done once at init.
					  Do now since there is no init. */

    cfunc = PyObject_New(CfuncObject, &CfuncType);
    
    if (!cfunc) {
	    return PyErr_Format(_Error,
			       "NA_new_cfunc: failed creating '%s'",
			       cfd->name);
    }

    cfunc->descr = *cfd;

    return (PyObject*)cfunc;
}

static int NA_add_cfunc(PyObject *dict, char *keystr, CfuncDescriptor *descr) {
    return PyDict_SetItemString(dict, keystr, 
				(PyObject *) NA_new_cfunc(descr));
}



/*
 * Local Variables:
 * mode: C
 * c-file-style: "python"
 * End:
 */
