/* Copyright (C) 2003, 2004 Peter J. Verveer
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met: 
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.      
 */

#define ND_IMPORT_LIBNUMARRAY
#include "nd_image.h"
#undef ND_IMPORT_LIBNUMARRAY
#include "ni_support.h"
#include "ni_filters.h"
#include "ni_fourier.h"
#include "ni_morphology.h"
#include "ni_interpolation.h"
#include "ni_measure.h"

/* return the rank of an array */
int NI_GetArrayRank(PyArrayObject *array)
{
  assert(array != NULL);
  return array->nd;
}

/* return the rank of an array */
NumarrayType NI_GetArrayType(PyArrayObject *array)
{
  assert(array != NULL);
  return array->descr->type_num;
}

/* return the dimensions of an array */
void NI_GetArrayDimensions(PyArrayObject *array, int* dimensions)
{
  int ii;
  assert(array != NULL);
  for(ii = 0; ii < array->nd; ii++)
    dimensions[ii] = array->dimensions[ii];
}

/* return the strides of an array */
void NI_GetArrayStrides(PyArrayObject *array, int* strides)
{
  int ii;
  assert(array != NULL);
  for(ii = 0; ii < array->nd; ii++)
    strides[ii] = array->strides[ii];
}

/* return the data of an array */
char* NI_GetArrayData(PyArrayObject *array)
{
  assert(array != NULL);
  return (char*)NA_OFFSETDATA(array);
}

/* return number of elements */
int NI_Elements(PyArrayObject *array)
{
  return NA_elements(array);
}

/* test for equal shape */
int NI_ShapeEqual(PyArrayObject *array1, PyArrayObject *array2)
{
  return NA_ShapeEqual(array1, array2);
}

/* copy an array */
int NI_CopyArray(PyArrayObject *array1, PyArrayObject *array2)
{
  return NA_copyArray(array1, array2);
}

PyArrayObject* NI_ContiguousArray(PyArrayObject *input) 
{
  return NA_InputArray((PyObject*)input, tAny, NUM_C_ARRAY);
}

PyArrayObject* NI_NewArray(NumarrayType type, int rank, int *dimensions)
{
  int ii;
  maybelong dims[MAXDIM];

  for(ii = 0; ii < rank; ii++)
    dims[ii] = dimensions[ii];
  return NA_vNewArray(NULL, type, rank, dims);
}

/* Get an output array, possibly from an existing object */
int NI_OutputArray(NumarrayType type, int rank, int *dimensions, 
                   PyObject *output_in, PyArrayObject **output)
{
  int ii;
  maybelong oshape[NI_MAXDIM];
  assert (output != NULL && *output == NULL);
  assert(dimensions != NULL);

  for(ii = 0; ii < rank; ii++)
    oshape[ii] = dimensions[ii];

  if (!output_in || (PyObject*)output_in == Py_None) {
    /* No output array is provided, allocate a new array: */
    *(PyArrayObject**)output = NA_vNewArray(NULL, type, rank, oshape);
    if (!*output) {
      PyErr_NoMemory();
      return 0;
    }
  } else {
    /* An object is provided, call NA_OutputArray: */
    *(PyArrayObject**)output = NA_OutputArray((PyObject*)output_in, type, 
                                              NUM_ALIGNED|NUM_NOTSWAPPED);
    if (!*output) {
      PyErr_SetString(PyExc_RuntimeError, "cannot convert output array");
      return 0;
    }
    if ((*output)->nd != rank) {
      PyErr_SetString(PyExc_RuntimeError, "output rank incorrect");
      return 0;
    }
    for(ii = 0; ii < rank; ii++) {
      if ((*output)->dimensions[ii] != dimensions[ii]) {
        PyErr_SetString(PyExc_RuntimeError, "output dimensions incorrect");
        return 0;
      }
    }
  }
  return 1;
}

/* Convert an array of any type, not necessarily contiguous */
static int 
NI_ObjectToArray(PyObject *object, PyArrayObject **array)
{
  *array = NA_InputArray(object, tAny, NUM_ALIGNED|NUM_NOTSWAPPED);
  return *array ? 1 : 0;
}


/* Convert an array of any type, not necessarily contiguous */
static int 
NI_ObjectToIoArray(PyObject *object, PyArrayObject **array)
{
  *array = NA_IoArray(object, tAny, NUM_ALIGNED|NUM_NOTSWAPPED);
  return *array ? 1 : 0;
}


/* Convert a contiguous Bool array */
static int 
NI_ObjectToContiguousBool(PyObject *object, PyArrayObject **array)
{
  *array = NA_InputArray(object, tBool, NUM_C_ARRAY);
  return *array ? 1 : 0;
}


/* Convert a contiguous Float64 array */
static int 
NI_ObjectToContiguousFloat64(PyObject *object, PyArrayObject **array)
{
  *array = NA_InputArray(object, tFloat64, NUM_C_ARRAY);
  return *array ? 1 : 0;
}

/* Convert an array type */
static int 
NI_TypeObjectToTypeNo(PyObject *object, NumarrayType *type)
{
  if (!object || object == Py_None)
    *type = tAny;
  else
    *type = (NumarrayType)NA_typeObjectToTypeNo(object);
  return 1;
}


/* Convert a python object to an array of double values */
static int
NI_ObjectToDoubles(PyObject* obj, double **vector, int *len)
{
  PyArrayObject *array = NULL;
  int result = 0;
  
  assert(vector != NULL && *vector == NULL);

  if ((array = (PyArrayObject*)NA_InputArray(obj, tFloat64, NUM_C_ARRAY))) {
    if (array->nd > 1) {
      PyErr_SetString(PyExc_RuntimeError, "sequences must be 1D");
    } else {
      int ii;
      Float64 *p = (Float64*)NA_OFFSETDATA(array);
      *len = NA_elements(array);
      *vector = (double*)malloc(*len * sizeof(double));
      if (!*vector) {
        PyErr_NoMemory();
        goto exit;
      }
      for(ii = 0; ii < *len; ii++)
        (*vector)[ii] = p[ii];
    }
  } else {
    PyErr_SetString(PyExc_RuntimeError, "cannot convert sequence");
  }
  result = 1;
 exit:
  Py_XDECREF(array);
  return result;
}


/* Convert a python object to an array of integer values */
static int
NI_ObjectToInts(PyObject* obj, int **vector, int *len)
{
  PyArrayObject *array = NULL;
  int result = 0;
  
  assert(vector != NULL && *vector == NULL);

  if ((array = (PyArrayObject*)NA_InputArray(obj, tInt32, NUM_C_ARRAY))) {
    if (array->nd > 1) {
      PyErr_SetString(PyExc_RuntimeError, "sequences must be 1D");
    } else {
      int ii;
      Int32 *p = (Int32*)NA_OFFSETDATA(array);
      *len = NA_elements(array);
      *vector = (int*)malloc(*len * sizeof(int));
      if (!*vector) {
        PyErr_NoMemory();
        goto exit;
      }
      for(ii = 0; ii < *len; ii++)
        (*vector)[ii] = (int)p[ii];
    }
  } else {
    PyErr_SetString(PyExc_RuntimeError, "cannot convert sequence");
  }
  result = 1;
 exit:
  Py_XDECREF(array);
  return result;
}


static PyObject *Py_Correlate1D(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  PyObject *output_object = NULL, *weights_object = NULL;
  int axis, mode, shift, len;
  NumarrayType type;
  double *weights = NULL, cval;
  
  if (PyArg_ParseTuple(args, "O&OiOidiO&", NI_ObjectToArray, &input, 
                       &weights_object, &axis, 
                       &output_object, &mode, &cval, &shift, 
                       NI_TypeObjectToTypeNo, &type)) {
    if (NI_ObjectToDoubles(weights_object, &weights, &len)) {
      NI_Correlate1D(input, weights, len, axis, &output, output_object, 
                     (NI_ExtendMode)mode, cval, shift, type);
    }
  }

  Py_XDECREF(input);
  if (weights)
    free(weights);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}

static PyObject *Py_Correlate(PyObject *obj, PyObject *args)
{
  PyObject *output_object = NULL, *shifts_object = NULL;
  PyArrayObject *input = NULL, *output = NULL, *weights = NULL;
  PyArrayObject *footprint = NULL;
  int mode, *shifts = NULL, nshifts;
  NumarrayType type;
  double cvalue;

  if (PyArg_ParseTuple(args, "O&O&O&OidOO&", NI_ObjectToArray, &input, 
                       NI_ObjectToContiguousFloat64, &weights, 
                       NI_ObjectToContiguousBool, &footprint,
                       &output_object, &mode, &cvalue, &shifts_object, 
                       NI_TypeObjectToTypeNo, &type)) {
    if (NI_ObjectToInts(shifts_object, &shifts, &nshifts)) {
      if (nshifts != input->nd) {
        PyErr_SetString(PyExc_RuntimeError, "not enought shifts provided");
      } else {
        NI_Correlate(input, weights, footprint, &output, output_object, 
                     (NI_ExtendMode)mode, cvalue, shifts, type);
      }
    }     
  }
  
  Py_XDECREF(input);
  Py_XDECREF(weights);
  if (shifts)
    free(shifts);
  Py_XDECREF(footprint);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}

static PyObject *Py_BoxcarFilter1D(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  PyObject *output_object = NULL;
  int axis, filter_size, mode, shift;
  NumarrayType type;
  double cvalue;
  
  if (PyArg_ParseTuple(args, "O&iiOidiO&", NI_ObjectToArray, &input, 
                       &filter_size, &axis, &output_object, &mode, &cvalue, 
                       &shift, NI_TypeObjectToTypeNo, &type)) {
    NI_BoxcarFilter1D(input, filter_size, axis, &output, output_object, 
                      (NI_ExtendMode)mode,  cvalue, shift, type);
  }    

  Py_XDECREF(input);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}


static PyObject *Py_MinimumMaximumFilter1D(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  PyObject *output_object = NULL;
  int axis, filter_size, mode, shift, minimum;
  double cvalue;
  
  if (PyArg_ParseTuple(args, "O&iiOidii", NI_ObjectToArray, &input, 
                       &filter_size, &axis, &output_object, &mode, 
                       &cvalue, &shift, &minimum)) {
    NI_MinimumMaximumFilter1D(input, filter_size, axis, &output,
                              output_object, (NI_ExtendMode)mode, cvalue,
                              shift, minimum);
  }    

  Py_XDECREF(input);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}


static PyObject *Py_MinimumMaximumFilter(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *footprint = NULL;
  PyArrayObject *structure_values = NULL;
  PyObject *output_object = NULL, *shifts_object = NULL;
  PyObject *structure_values_object = NULL;
  int mode, *shifts = NULL, nshifts, minimum;
  double cvalue;

  if (PyArg_ParseTuple(args, "O&O&OOidOi", NI_ObjectToArray, &input,
                       NI_ObjectToContiguousBool, &footprint, 
                       &structure_values_object, &output_object,
                       &mode, &cvalue, &shifts_object, &minimum)) {

    if (!NI_ObjectToInts(shifts_object, &shifts, &nshifts))
      goto exit;

    if (structure_values_object != Py_None) {
      structure_values = NA_InputArray(structure_values_object, tFloat64, 
                                       NUM_C_ARRAY);
      if (!structure_values) {
        PyErr_SetString(PyExc_RuntimeError,
                        "cannot convert structure values");
        goto exit;
      }
    }

    NI_MinimumMaximumFilter(input, footprint, structure_values, &output, 
                            output_object,  (NI_ExtendMode)mode, cvalue, 
                            shifts, minimum);
  }    

 exit:
  Py_XDECREF(input);
  Py_XDECREF(footprint);
  if (shifts)
    free(shifts);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}


static PyObject *Py_RankFilter(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *footprint = NULL;
  PyObject *output_object = NULL, *shifts_object = NULL;
  int mode, rank, percentile_filter, *shifts = NULL, nshifts;
  double percentile, cvalue;

  if (PyArg_ParseTuple(args, "O&idO&OidOi", NI_ObjectToArray, &input,
                       &rank, &percentile, NI_ObjectToContiguousBool,
                       &footprint, &output_object, &mode, &cvalue, 
                       &shifts_object, &percentile_filter)) {
    if (NI_ObjectToInts(shifts_object, &shifts, &nshifts)) {
      NI_RankFilter(input, rank, percentile, footprint, &output, 
                    output_object, (NI_ExtendMode)mode, cvalue, shifts, 
                    percentile_filter);
    }
  }    

  Py_XDECREF(input);
  if (shifts)
    free(shifts);
  Py_XDECREF(footprint);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}

static PyObject *filterFuncCallable = NULL;

static int Py_FilterFunc(double *buffer, int filter_size, double *output, 
                         void *data)
{
  PyArrayObject *py_buffer = NULL;
  PyObject *rv = NULL;

  py_buffer = (PyArrayObject*)PyArray_FromDimsAndData(1, &filter_size,
                                                      PyArray_DOUBLE,
                                                      (char*)buffer);
  if (!py_buffer)
    goto exit;
  rv = PyObject_CallFunctionObjArgs(filterFuncCallable, py_buffer, NULL);
  if (!rv)
    goto exit;
  *output = PyFloat_AsDouble(rv);
 exit:
  Py_XDECREF(py_buffer);
  Py_XDECREF(rv);
  return PyErr_Occurred() ? 0 : 1;
}

static PyObject *Py_GenericFilter(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *footprint = NULL;
  PyObject *output_object = NULL, *shifts_object = NULL, *fnc = NULL;
  int mode, *shifts = NULL, nshifts;
  double cvalue;
  void *func = Py_FilterFunc, *data = NULL;
  NumarrayType type;

  if (PyArg_ParseTuple(args, "O&OO&OidOO&", NI_ObjectToArray, &input, &fnc,
                       NI_ObjectToContiguousBool, &footprint,
                       &output_object, &mode, &cvalue, &shifts_object, 
                       NI_TypeObjectToTypeNo, &type)) {
    if (!NI_ObjectToInts(shifts_object, &shifts, &nshifts)) 
      goto exit;
      
    if (PyCObject_Check(fnc)) {
      func = PyCObject_AsVoidPtr(fnc);
      data = PyCObject_GetDesc(fnc);
    } else if (!PyCallable_Check(fnc)) {
      PyErr_SetString(PyExc_RuntimeError,
                      "function parameter is not callable");
      goto exit;
    }   
    
    Py_XDECREF(filterFuncCallable);
    filterFuncCallable = fnc;
    Py_INCREF(filterFuncCallable);

    NI_GenericFilter(input, func, data, footprint, &output, output_object, 
                     (NI_ExtendMode)mode, cvalue, shifts, type);
  }

exit:
  Py_XDECREF(input);
  Py_XDECREF(filterFuncCallable);
  filterFuncCallable = NULL;
  if (shifts)
    free(shifts);
  Py_XDECREF(footprint);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}


static PyObject *Filter1DCallable = NULL;

static int Py_Filter1DFunc(double *iline, int ilen, double *oline,
                           int olen, void *data)
{
  PyArrayObject *py_ibuffer = NULL, *py_obuffer = NULL; 
  PyObject *rv = NULL;
  int ii;
  double *po = NULL;

  py_ibuffer = (PyArrayObject*)PyArray_FromDimsAndData(1, &ilen,
                                                       PyArray_DOUBLE,
                                                       (char*)iline);
  py_obuffer = (PyArrayObject*)PyArray_FromDimsAndData(1, &olen,
                                                       PyArray_DOUBLE,
                                                       (char*)oline);
  if (!py_ibuffer || !py_obuffer)
    goto exit;
  rv = PyObject_CallFunctionObjArgs(Filter1DCallable, py_ibuffer, 
                                    py_obuffer, NULL);
  if (!rv)
    goto exit;
  po = (double*)NA_OFFSETDATA(py_obuffer);
  for(ii = 0; ii < olen; ii++)
    oline[ii] = po[ii];
 exit:
  Py_XDECREF(py_ibuffer);
  Py_XDECREF(py_obuffer);
  Py_XDECREF(rv);
  return PyErr_Occurred() ? 0 : 1;
}


static PyObject *Py_GenericFilter1D(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  PyObject *output_object = NULL, *fnc = NULL;
  int mode, axis, filter_size, shift;
  double cvalue;
  void *func = Py_Filter1DFunc, *data = NULL;
  NumarrayType type;

  if (PyArg_ParseTuple(args, "O&OiiOidiO&", NI_ObjectToArray, &input, &fnc,
                       &filter_size, &axis, &output_object, &mode,
                       &cvalue, &shift, NI_TypeObjectToTypeNo, &type)) {
      
    if (PyCObject_Check(fnc)) {
      func = PyCObject_AsVoidPtr(fnc);
      data = PyCObject_GetDesc(fnc);
    } else if (!PyCallable_Check(fnc)) {
      PyErr_SetString(PyExc_RuntimeError,
                      "function parameter is not callable");
      goto exit;
    }    
    
    Py_XDECREF(Filter1DCallable);
    Filter1DCallable = fnc;
    Py_INCREF(Filter1DCallable);

    NI_GenericFilter1D(input, func, data, filter_size, axis, &output,
                       output_object, (NI_ExtendMode)mode, cvalue,
                       shift, type);
  }

exit:
  Py_XDECREF(input);
  Py_XDECREF(Filter1DCallable);
  Filter1DCallable = NULL;
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}

static PyObject *Py_BinaryErosion(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *strct = NULL, *mask = NULL;
  PyObject *output_object = NULL, *shifts_object = NULL;
  PyObject *mask_object = NULL;
  int border_value, *shifts = NULL, nshifts, invert_input, invert_output;
  int changed = 0;

  if (PyArg_ParseTuple(args, "O&O&OOiOii", NI_ObjectToArray, &input, 
                       NI_ObjectToContiguousBool, &strct, &mask_object,
                       &output_object, &border_value, &shifts_object, 
                       &invert_input, &invert_output)) {
    
    if (NI_ObjectToInts(shifts_object, &shifts, &nshifts)) {
      if (mask_object != Py_None) {
        mask = NA_InputArray(mask_object, tAny,
                             NUM_ALIGNED|NUM_NOTSWAPPED);
        if (!mask) {
          PyErr_SetString(PyExc_RuntimeError, "cannot convert mask");
        }
      }
      if (!PyErr_Occurred()) {
        NI_BinaryErosion(input, strct, mask, &output, output_object, 
                         border_value, shifts, invert_input,
                         invert_output, &changed);
      }
    }
  }

  Py_XDECREF(input);
  Py_XDECREF(strct);
  Py_XDECREF(mask);
  if (shifts)
    free(shifts);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return Py_BuildValue("Ni", (PyObject*)output, changed);
    } else {
      Py_XDECREF(output);
      return Py_BuildValue("i", changed);
    }
  }
}

static PyObject *Py_DistanceTransformOnePass(PyObject *obj, PyObject *args)
{
  PyArrayObject *strct = NULL, *distances = NULL, *features = NULL;
  PyObject *tmp = NULL;

  if (PyArg_ParseTuple(args, "O&O&O", NI_ObjectToArray, &strct,
                       NI_ObjectToIoArray, &distances, &tmp)) {
    if (tmp != Py_None) {
      features = NA_IoArray(tmp, tAny, NUM_ALIGNED|NUM_NOTSWAPPED);
    }
    if (!PyErr_Occurred()) {
      NI_DistanceTransformOnePass(strct, distances, features);
    }
  }

  Py_XDECREF(strct); 
  Py_XDECREF(distances); 
  Py_XDECREF(features); 
  if (PyErr_Occurred()) {
    return NULL;
  } else {
    Py_INCREF(Py_None);
    return Py_None;
  }
}

static PyObject *Py_DistanceTransformBruteForce(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *features = NULL;
  int metric, return_distances, return_features;
  PyObject *ft_object = NULL, *dt_object = NULL, *sampling_object = NULL;
  int dt_not_inplace, ft_not_inplace, len;
  double *sampling = NULL;

  if (PyArg_ParseTuple(args, "O&iOiiOO", NI_ObjectToArray, &input, &metric,
                       &sampling_object, &return_distances,
                       &return_features, &dt_object, &ft_object)) {
    if (sampling_object && (sampling_object != Py_None)) {
      if (NI_ObjectToDoubles(sampling_object, &sampling, &len)) {
        if (len != input->nd) {
          PyErr_SetString(PyExc_RuntimeError, 
                          "sampling parameter not correct");
        }
      }
    }
  }
  
  if (!PyErr_Occurred()) {
    NI_DistanceTransformBruteForce(input, metric, sampling,
                                   return_distances, return_features,
                                   &output, &features, dt_object,
                                   ft_object);
  }

  Py_XDECREF(input);
  if (sampling)
    free(sampling);
  if (PyErr_Occurred()) {
    if (return_distances) {
      Py_XDECREF(output);
    }
    if (return_features) {
      Py_XDECREF(features);
    }
    return NULL;
  } else {
    dt_not_inplace = !dt_object || dt_object == Py_None;
    ft_not_inplace = !ft_object || ft_object == Py_None;
    if (return_distances && return_features && 
        dt_not_inplace && ft_not_inplace) {
      return Py_BuildValue("NN", output, features);
    } else if (return_distances && dt_not_inplace) {
      return (PyObject*)output;
    } else if (return_features && ft_not_inplace) {
      return (PyObject*)features;
    } else {
      Py_INCREF(Py_None);
      return Py_None;
    }
  }
}

static PyObject *Py_EuclideanFeatureTransform(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *features = NULL;
  PyObject *output_object = NULL, *sampling_object = NULL;
  int len;
  double *sampling = NULL;

  if (PyArg_ParseTuple(args, "O&OO", NI_ObjectToArray, &input, 
                       &sampling_object, &output_object)) {

    if (sampling_object && (sampling_object != Py_None)) {
      if (NI_ObjectToDoubles(sampling_object, &sampling, &len)) {
        if (len != input->nd) {
          PyErr_SetString(PyExc_RuntimeError, 
                          "sampling parameter not correct");
        }
      }
    }
  }

  if (!PyErr_Occurred()) {
    NI_EuclideanFeatureTransform(input, sampling, &features,
                                 output_object);
  }

  Py_XDECREF(input);
  if (sampling)
    free(sampling);
  if (PyErr_Occurred()) {
    Py_XDECREF(features);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)features;
    } else {
       Py_INCREF(Py_None);
       Py_XDECREF(features);
      return Py_None;
    }
  }
}

static PyObject *Py_WatershedIFT(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *markers = NULL, *strct = NULL;
  PyObject *output_object = NULL;

  if (PyArg_ParseTuple(args, "O&O&O&O", NI_ObjectToArray, &input, 
                       NI_ObjectToArray, &markers, 
                       NI_ObjectToContiguousBool, &strct,
                       &output_object)) {
    
    NI_WatershedIFT(input, markers, strct, &output, output_object);
  }

  Py_XDECREF(input);
  Py_XDECREF(markers);
  Py_XDECREF(strct);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}

static PyObject *Py_Label(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *strct = NULL;
  PyObject *output_object = NULL;
  int max_label;

  if (PyArg_ParseTuple(args, "O&O&O", NI_ObjectToArray, &input, 
                       NI_ObjectToContiguousBool, &strct,
                       &output_object)) {
    NI_Label(input, strct, &max_label, &output, output_object);
  }    

  Py_XDECREF(input);
  Py_XDECREF(strct);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return Py_BuildValue("Ni", (PyObject*) output, max_label);
      return (PyObject*)output;
    } else {
      return Py_BuildValue("i", max_label);
      Py_XDECREF(output);
    }
  }
}

static PyObject *Py_FindObjects(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL;
  PyObject *result = NULL, *tuple = NULL, *start = NULL, *end = NULL;
  PyObject *slc = NULL;
  int ii, jj, max_label;
  NI_ObjectRegion **regions = NULL;

  if (!PyArg_ParseTuple(args, "O&i", NI_ObjectToArray, &input,
                        &max_label)) {
    goto exit;
  }
  
  if (max_label < 0)
    max_label = 0;
  if (max_label > 0) {
    regions = (NI_ObjectRegion**)malloc(max_label *
                                        sizeof(NI_ObjectRegion*));
    if (!regions) {
      PyErr_NoMemory();
      goto exit;
    }
  }
   
  for(ii = 0; ii < max_label; ii++)
    regions[ii] = NULL;
  if (!NI_FindObjects(input, max_label, regions))
    goto exit;

  result = PyList_New(max_label);
  if (!result) {
    PyErr_NoMemory();
    goto exit;
  }

  for(ii = 0; ii < max_label; ii++) {
    if (regions[ii]) {
      PyObject *tuple = PyTuple_New(input->nd);
      if (!tuple) {
        PyErr_NoMemory();
        goto exit;
      }
      for(jj = 0; jj < input->nd; jj++) {
        start = PyInt_FromLong(regions[ii]->start[jj]);
        end = PyInt_FromLong(regions[ii]->end[jj]);
        if (!start || !end) {
          PyErr_NoMemory();
          goto exit;
        }
        slc = PySlice_New(start, end, NULL);
        if (!slc) {
          PyErr_NoMemory();
          goto exit;
        }
        Py_XDECREF(start);
        Py_XDECREF(end);
        start = end = NULL;
        PyTuple_SetItem(tuple, jj, slc);
        slc = NULL;
      }
      PyList_SetItem(result, ii, tuple);
      tuple = NULL;
    } else {
      Py_INCREF(Py_None);
      PyList_SetItem(result, ii, Py_None);
    }
  }

  Py_INCREF(result);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(result);
  Py_XDECREF(tuple);
  Py_XDECREF(start);
  Py_XDECREF(end);
  Py_XDECREF(slc);
  if (regions) {
    for(ii = 0; ii < max_label; ii++)
      if (regions[ii])
        free(regions[ii]);
    free(regions);
  }
  if (PyErr_Occurred()) {
    Py_XDECREF(result);
    return NULL;
  } else {
    return result;
  }
}

static PyObject *Py_SplineFilter1D(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  PyObject *output_object = NULL;
  int axis, order;
  NumarrayType type;
  
  if (PyArg_ParseTuple(args, "O&iiOO&", NI_ObjectToArray, &input, &order, 
                       &axis, &output_object, NI_TypeObjectToTypeNo,
                       &type)) {
    NI_SplineFilter1D(input, order, axis, &output, output_object, type);
  }    

  Py_XDECREF(input);
  if (PyErr_Occurred()) { 
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}


static PyObject *Py_AffineTransform(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  PyArrayObject *matrix = NULL;
  PyObject *output_object = NULL,  *odim_object = NULL;
  PyObject *shift_object = NULL;
  int mode, order, *output_dimensions = NULL, orank, slen, ii;
  NumarrayType output_type;
  double cval, *shift = NULL, *matrix_data = NULL;
  Float64 *p;
  
  if (PyArg_ParseTuple(args, "O&O&OOO&Oiid", NI_ObjectToArray, &input, 
                       NI_ObjectToContiguousFloat64, &matrix,
                       &shift_object, &odim_object, NI_TypeObjectToTypeNo,
                       &output_type, &output_object, &order, &mode,
                       &cval)) {

    if (!NI_ObjectToInts(odim_object, &output_dimensions, &orank))
      goto exit;

    if (!NI_ObjectToDoubles(shift_object, &shift, &slen))
      goto exit;
    

    if (slen != input->nd) {
      PyErr_SetString(PyExc_RuntimeError, "shift dimensions not correct");
      goto exit;
    }

    if (matrix->nd == 1) {
      if (matrix->dimensions[0] != input->nd) {
        PyErr_SetString(PyExc_RuntimeError,
                        "matrix dimensions not correct");
        goto exit;
      }
      matrix_data = (double*)malloc(input->nd * sizeof(double));
      if (!matrix_data) {
        PyErr_NoMemory();
        goto exit;
      }
      p = (Float64*)NA_OFFSETDATA(matrix);
      for(ii = 0; ii < input->nd; ii++)
        matrix_data[ii] = (double)p[ii];
      NI_ZoomShift(input, matrix_data, shift, output_dimensions, orank, 
                   output_type, &output, output_object, order, 
                   (NI_ExtendMode)mode, cval);
    } else {
      if (matrix->dimensions[0] != input->nd ||
          matrix->dimensions[1] != orank) {
        PyErr_SetString(PyExc_RuntimeError,
                        "matrix dimensions not correct");
        goto exit;
      }
      matrix_data = (double*)malloc(input->nd * orank * sizeof(double));
      if (!matrix_data) {
        PyErr_NoMemory();
        goto exit;
      }
      p = (Float64*)NA_OFFSETDATA(matrix);
      for(ii = 0; ii < input->nd * orank; ii++)
        matrix_data[ii] = (double)p[ii];
      NI_AffineTransform(input, matrix_data, shift, output_dimensions,
                         orank, output_type, &output, output_object, order,
                         (NI_ExtendMode)mode, cval);
    }
  }

 exit:
  Py_XDECREF(input);
  Py_XDECREF(matrix);
  if (shift)
    free(shift);
  if (matrix_data)
    free(matrix_data);
  if (output_dimensions)
    free(output_dimensions);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}

static PyObject *mapCallable = NULL;

static int Py_Map(int *ocoor, double* icoor, int orank, int irank, 
                  void *data)
{
  PyObject *args = NULL, *rets = NULL;
  int ii;

  args = PyTuple_New(orank);
  if (!args)
    goto exit;
  for(ii = 0; ii < orank; ii++) {
    PyTuple_SetItem(args, ii, PyInt_FromLong(ocoor[ii]));
    if (PyErr_Occurred())
      goto exit;
  }
  rets = PyObject_CallFunctionObjArgs(mapCallable, args, NULL);
  if (!rets)
    goto exit;
  for(ii = 0; ii < irank; ii++) {
    icoor[ii] = PyFloat_AsDouble(PyTuple_GetItem(rets, ii));
    if (PyErr_Occurred())
      goto exit;
  }
 exit:
  Py_XDECREF(args);
  Py_XDECREF(rets);
  return PyErr_Occurred() ? 0 : 1;
}


static PyObject *Py_GeometricTransform(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  PyObject *output_object = NULL, *mapping = NULL, *odim_object = NULL;
  int mode, order, *output_dimensions = NULL, orank;
  NumarrayType output_type;
  double cval;
  void *func = Py_Map, *data = NULL;
  
  if (PyArg_ParseTuple(args, "O&OOO&Oiid", NI_ObjectToArray, &input, 
                       &mapping, &odim_object, 
                       NI_TypeObjectToTypeNo, &output_type, &output_object, 
                       &order, &mode, &cval)) {

    if (!NI_ObjectToInts(odim_object, &output_dimensions, &orank))
      goto exit;

    if (PyCObject_Check(mapping)) {
      func = PyCObject_AsVoidPtr(mapping);
      data = PyCObject_GetDesc(mapping);
    } else if (!PyCallable_Check(mapping)) {
      PyErr_SetString(PyExc_RuntimeError,
                      "mapping parameter is not callable");
      goto exit;
    }
    
    Py_XDECREF(mapCallable);
    mapCallable = mapping;
    Py_INCREF(mapCallable);
    NI_GeometricTransform(input, func, data, output_dimensions, orank, 
                          output_type, &output, output_object, order, 
                          (NI_ExtendMode)mode, cval);
  }    

 exit:
  Py_XDECREF(input);
  Py_XDECREF(mapCallable);
  mapCallable = NULL;
  if (output_dimensions)
    free(output_dimensions);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}


static PyObject *Py_MapCoordinates(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  PyArrayObject *coordinates = NULL;
  PyObject *output_object = NULL, *odim_object = NULL;
  int mode, order, *output_dimensions = NULL, orank;
  NumarrayType output_type;
  double cval;
  
  if (PyArg_ParseTuple(args, "O&O&OO&Oiid", NI_ObjectToArray, &input, 
                       NI_ObjectToArray, &coordinates, &odim_object, 
                       NI_TypeObjectToTypeNo, &output_type,
                       &output_object, &order, &mode, &cval)) {
    if (NI_ObjectToInts(odim_object, &output_dimensions, &orank)) {
      NI_MapCoordinates(input, coordinates, output_dimensions, orank, 
                        output_type, &output, output_object, order,
                        (NI_ExtendMode)mode, cval);
    }
  }    

  Py_XDECREF(input);
  Py_XDECREF(coordinates);
  if (output_dimensions)
    free(output_dimensions);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}


static PyObject *Py_Zoom(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  PyObject *output_object = NULL, *odim_object = NULL, *zoom_object = NULL;
  int mode, order, *output_dimensions = NULL, orank, zlen;
  NumarrayType output_type;
  double cval, *zoom = NULL;
  
  if (PyArg_ParseTuple(args, "O&OOO&Oiid", NI_ObjectToArray, &input, 
                       &zoom_object, &odim_object, NI_TypeObjectToTypeNo, 
                       &output_type, &output_object, &order, &mode, &cval)) {
    if (!NI_ObjectToInts(odim_object, &output_dimensions, &orank))
      goto exit;

    if (!NI_ObjectToDoubles(zoom_object, &zoom, &zlen))
      goto exit;
    
    if (zlen != input->nd) {
      PyErr_SetString(PyExc_RuntimeError, "number of zooms not correct");
      goto exit;
    }
    
    NI_ZoomShift(input, zoom, NULL, output_dimensions, orank, output_type,
                 &output,  output_object, order, (NI_ExtendMode)mode,
                 cval);
  }

 exit:
  Py_XDECREF(input);
  if (zoom)
    free(zoom);
  if (output_dimensions)
    free(output_dimensions);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}


static PyObject *Py_Shift(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  PyObject *output_object = NULL, *odim_object = NULL;
  PyObject *shift_object = NULL;
  int mode, order, *output_dimensions = NULL, orank, slen;
  NumarrayType output_type;
  double cval, *shift = NULL;
  
  if (PyArg_ParseTuple(args, "O&OOO&Oiid", NI_ObjectToArray, &input, 
                       &shift_object, &odim_object, NI_TypeObjectToTypeNo, 
                       &output_type, &output_object, &order, &mode,
                       &cval)) {

    if (!NI_ObjectToInts(odim_object, &output_dimensions, &orank))
      goto exit;

    if (!NI_ObjectToDoubles(shift_object, &shift, &slen))
      goto exit;
    
    if (slen != input->nd) {
      PyErr_SetString(PyExc_RuntimeError, "number of shifts not correct");
      goto exit;
    }

    NI_ZoomShift(input, NULL, shift, output_dimensions, orank, output_type, 
                 &output, output_object, order, (NI_ExtendMode)mode, cval);
  }    

 exit:
  Py_XDECREF(input);
  if (shift)
    free(shift);
  if (output_dimensions)
    free(output_dimensions);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}


static PyObject *Py_FourierGaussian(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  PyObject *output_object = NULL, *sigmas_object = NULL;
  double *sigmas = NULL;
  int len, n, axis;
  
  if (PyArg_ParseTuple(args, "O&OiiO", NI_ObjectToArray, &input, 
                       &sigmas_object, &n, &axis, &output_object)) {
    if (NI_ObjectToDoubles(sigmas_object, &sigmas, &len)) {
      if (len != input->nd) {
        PyErr_SetString(PyExc_RuntimeError, "sigma parameter not correct");
      } else {
        NI_FourierGaussian(input, sigmas, n, axis, &output, output_object);
      }
    }
  }

  Py_XDECREF(input);
  if (sigmas)
    free(sigmas);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}


static PyObject *Py_FourierBoxcar(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  PyObject *output_object = NULL, *sizes_object = NULL;
  double *sizes = NULL;
  int len, n, axis;
  
  if (PyArg_ParseTuple(args, "O&OiiO", NI_ObjectToArray, &input, 
                       &sizes_object, &n, &axis, &output_object)) {
    if (NI_ObjectToDoubles(sizes_object, &sizes, &len)) {
      if (len != input->nd) {
        PyErr_SetString(PyExc_RuntimeError, "size parameter not correct");
      } else {
        NI_FourierBoxcar(input, sizes, n, axis, &output, output_object);
      }
    }
  }

  Py_XDECREF(input);
  if (sizes)
    free(sizes);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}


static PyObject *Py_FourierEllipsoid(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  PyObject *output_object = NULL, *sizes_object = NULL;
  double *sizes = NULL;
  int len, n, axis;
  
  if (PyArg_ParseTuple(args, "O&OiiO", NI_ObjectToArray, &input, 
                       &sizes_object, &n, &axis, &output_object)) {
    if (NI_ObjectToDoubles(sizes_object, &sizes, &len)) {
      if (len != input->nd) {
        PyErr_SetString(PyExc_RuntimeError, "size parameter not correct");
      } else {
        NI_FourierEllipsoid(input, sizes, n, axis, &output, output_object);
      }
    }
  }

  Py_XDECREF(input);
  if (sizes)
    free(sizes);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}


static PyObject *Py_FourierShift(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  PyObject *output_object = NULL, *shifts_object = NULL;
  double *shifts = NULL;
  int len, n, axis;
  
  if (PyArg_ParseTuple(args, "O&OiiO", NI_ObjectToArray, &input, 
                       &shifts_object, &n, &axis, &output_object)) {
    if (NI_ObjectToDoubles(shifts_object, &shifts, &len)) {
      if (len != input->nd) {
        PyErr_SetString(PyExc_RuntimeError, "shift parameter not correct");
      } else {
        NI_FourierShift(input, shifts, n, axis, &output, output_object);
      }
    }
  }

  Py_XDECREF(input);
  if (shifts)
    free(shifts);
  if (PyErr_Occurred()) {
    Py_XDECREF(output);
    return NULL;
  } else {
    if (!output_object || output_object == Py_None) {
      return (PyObject*)output;
    } else {
      Py_INCREF(Py_None);
      Py_XDECREF(output);
      return Py_None;
    }
  }
}

static int _NI_GetIndices(PyObject* indices_object, int** result_indices,
                          int* min_label, int* max_label, int* n_results)
{
  int *indices = NULL, n_indices, ii;

  if (indices_object == Py_None) {
    *min_label = -1;
    *n_results = 1;
  } else {
    if (!NI_ObjectToInts(indices_object, &indices, &n_indices))
      goto exit;
    if (n_indices < 1) {
      PyErr_SetString(PyExc_RuntimeError, "no correct indices provided");
      goto exit;
    } else {
      *min_label = *max_label = indices[0];
      if (*min_label < 0) {
        PyErr_SetString(PyExc_RuntimeError,
                        "negative indices not allowed");
        goto exit;
      }
      for(ii = 1; ii < n_indices; ii++) {
        if (indices[ii] < 0) {
          PyErr_SetString(PyExc_RuntimeError,
                          "negative indices not allowed");
          goto exit;
        }
        if (indices[ii] < *min_label)
          *min_label = indices[ii];
        if (indices[ii] > *max_label)
          *max_label = indices[ii];
      }
      *result_indices = (int*)malloc((*max_label - *min_label + 1) * 
                                     sizeof(int));
      if (!*result_indices) {
        PyErr_NoMemory();
        goto exit;
      }
      for(ii = 0; ii < *max_label - *min_label + 1; ii++)
        (*result_indices)[ii] = -1;
      *n_results = 0;
      for(ii = 0; ii < n_indices; ii++) {
        if ((*result_indices)[indices[ii] - *min_label] >= 0) {
          PyErr_SetString(PyExc_RuntimeError, "duplicate index");
          goto exit;
        }
        (*result_indices)[indices[ii] - *min_label] = ii;
        ++(*n_results);
      }
    }
  }
  
 exit:
  if (indices) 
    free(indices);
  return PyErr_Occurred() == NULL;
}


int _NI_GetLabels(PyObject* labels_object, PyArrayObject** labels)
{
  if (labels_object != Py_None) {
    *labels = NA_InputArray(labels_object, tAny,
                            NUM_ALIGNED|NUM_NOTSWAPPED);
    if (!*labels) {
      PyErr_SetString(PyExc_RuntimeError, "cannot convert labels");
      return 0;
    }
  } else {
    *labels = NULL;
  }
  return 1;
}

PyObject* _NI_BuildMeasurementResultDouble(int n_results, double* values)
{
  PyObject *result = NULL;
  if (n_results > 1) {
    result = PyList_New(n_results);
    if (result) {
      int ii;
      for(ii = 0; ii < n_results; ii++) {
        PyObject* val = PyFloat_FromDouble(values[ii]);
        if (!val) {
          Py_XDECREF(result);
          return NULL;
        }
        PyList_SET_ITEM(result, ii, val);
      }
    }
  } else {
    result = Py_BuildValue("d", values[0]);
  }
  return result;
}


PyObject* _NI_BuildMeasurementResultDoubleTuple(int n_results,
                                                int tuple_size,
                                                double* values)
{
  PyObject *result = NULL;
  int ii, jj;

  if (n_results > 1) {
    result = PyList_New(n_results);
    if (result) {
      for(ii = 0; ii < n_results; ii++) {
        PyObject* val = PyTuple_New(tuple_size);
        if (!val) {
          Py_XDECREF(result);
          return NULL;
        }
        for(jj = 0; jj < tuple_size; jj++) {
          int idx = jj + ii * tuple_size;
          PyTuple_SetItem(val, jj, PyFloat_FromDouble(values[idx]));
          if (PyErr_Occurred()) {
            Py_XDECREF(result);
            return NULL;
          }
        }
        PyList_SET_ITEM(result, ii, val);
      }
    }
  } else {
    result = PyTuple_New(tuple_size);
    if (result) {
      for(ii = 0; ii < tuple_size; ii++) {
        PyTuple_SetItem(result, ii, PyFloat_FromDouble(values[ii]));
        if (PyErr_Occurred()) {
          Py_XDECREF(result);
          return NULL;
        }
      }
    }
  }
  return result;
}


PyObject* _NI_BuildMeasurementResultInt(int n_results, int* values)
{
  PyObject *result = NULL;
  if (n_results > 1) {
    result = PyList_New(n_results);
    if (result) {
      int ii;
      for(ii = 0; ii < n_results; ii++) {
        PyObject* val = PyInt_FromLong(values[ii]);
        if (!val) {
          Py_XDECREF(result);
          return NULL;
        }
        PyList_SET_ITEM(result, ii, val);
      }
    }
  } else {
    result = Py_BuildValue("i", values[0]);
  }
  return result;
}


static PyObject *Py_Sum(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  double *sum = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  sum = (double*)malloc(n_results * sizeof(double));
  if (!sum) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_Statistics(input, labels, min_label, max_label, result_indices, 
                     n_results, sum, NULL, NULL, NULL, NULL, NULL, NULL))
    goto exit;

  result = _NI_BuildMeasurementResultDouble(n_results, sum);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (sum)
    free(sum);
  return result;
}


static PyObject *Py_Mean(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  double *sum = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  int ii, *total = NULL;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  sum = (double*)malloc(n_results * sizeof(double));
  total = (int*)malloc(n_results * sizeof(int));
  if (!sum || !total) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_Statistics(input, labels, min_label, max_label, result_indices, 
                     n_results, sum, total, NULL, NULL, NULL, NULL, NULL))
    goto exit;
  for(ii = 0; ii < n_results; ii++)
    sum[ii] = total[ii] > 0 ? sum[ii] / total[ii] : 0.0;

  result = _NI_BuildMeasurementResultDouble(n_results, sum);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (sum)
    free(sum);
  if (total)
    free(total);
  return result;
}


static PyObject *Py_Variance(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  double *sum = NULL, *variance = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  int *total = NULL;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  sum = (double*)malloc(n_results * sizeof(double));
  total = (int*)malloc(n_results * sizeof(int));
  variance = (double*)malloc(n_results * sizeof(double));
  if (!sum || !total || !variance) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_Statistics(input, labels, min_label, max_label, result_indices, 
                     n_results, sum, total, variance, NULL, NULL, NULL,
                    NULL))
    goto exit;

  result = _NI_BuildMeasurementResultDouble(n_results, variance);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (sum)
    free(sum);
  if (total)
    free(total);
  if (variance)
    free(variance);
  return result;
}

static PyObject *Py_Minimum(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  double *minimum = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  minimum = (double*)malloc(n_results * sizeof(double));
  if (!minimum) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_Statistics(input, labels, min_label, max_label, result_indices, 
                     n_results, NULL, NULL, NULL, minimum, NULL, NULL,
                     NULL))
    goto exit;

  result = _NI_BuildMeasurementResultDouble(n_results, minimum);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (minimum)
    free(minimum);
  return result;
}


static PyObject *Py_Maximum(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  double *maximum = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  maximum = (double*)malloc(n_results * sizeof(double));
  if (!maximum) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_Statistics(input, labels, min_label, max_label, result_indices, 
                     n_results, NULL, NULL, NULL, NULL, maximum, NULL,
                     NULL))
    goto exit;

  result = _NI_BuildMeasurementResultDouble(n_results, maximum);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (maximum)
    free(maximum);
  return result;
}


static PyObject *Py_MinimumPosition(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  double *minimum = NULL;
  int *minimum_position = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  minimum = (double*)malloc(n_results * sizeof(double));
  minimum_position = (int*)malloc(n_results * sizeof(int));
  if (!minimum || !minimum_position) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_Statistics(input, labels, min_label, max_label, result_indices, 
                     n_results, NULL, NULL, NULL, minimum, NULL, 
                     minimum_position, NULL))
    goto exit;

  result = _NI_BuildMeasurementResultInt(n_results, minimum_position);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (minimum_position)
    free(minimum_position);
  if (minimum)
    free(minimum);
  return result;
}


static PyObject *Py_MaximumPosition(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  double *maximum = NULL;
  int *maximum_position = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  maximum = (double*)malloc(n_results * sizeof(double));
  maximum_position = (int*)malloc(n_results * sizeof(int));
  if (!maximum || !maximum_position) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_Statistics(input, labels, min_label, max_label, result_indices, 
                     n_results, NULL, NULL, NULL, NULL, maximum, NULL,
                     maximum_position))
    goto exit;

  result = _NI_BuildMeasurementResultInt(n_results, maximum_position);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (maximum_position)
    free(maximum_position);
  if (maximum)
    free(maximum);
  return result;
}


static PyObject *Py_CenterOfMass(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  double *center_of_mass = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  center_of_mass = (double*)malloc(input->nd * n_results * sizeof(double));
  if (!center_of_mass) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_CenterOfMass(input, labels, min_label, max_label, result_indices, 
                       n_results, center_of_mass))
    goto exit;

  result = _NI_BuildMeasurementResultDoubleTuple(n_results, input->nd,
                                                 center_of_mass);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (center_of_mass)
    free(center_of_mass);
  return result;
}


static PyMethodDef methods[] = {
  {"correlate1d", (PyCFunction)Py_Correlate1D, METH_VARARGS, ""},
  {"correlate", (PyCFunction)Py_Correlate, METH_VARARGS, ""},
  {"boxcar_filter1d", (PyCFunction)Py_BoxcarFilter1D, METH_VARARGS, ""},
  {"minimum_maximum_filter1d", (PyCFunction)Py_MinimumMaximumFilter1D, 
   METH_VARARGS, ""},
  {"minimum_maximum_filter", (PyCFunction)Py_MinimumMaximumFilter, 
   METH_VARARGS, ""},
  {"rank_filter", (PyCFunction)Py_RankFilter, METH_VARARGS, ""},
  {"generic_filter", (PyCFunction)Py_GenericFilter, METH_VARARGS, ""},
  {"generic_filter1d", (PyCFunction)Py_GenericFilter1D, METH_VARARGS, ""},
  {"binary_erosion", (PyCFunction)Py_BinaryErosion, METH_VARARGS, ""},
  {"distance_transform_bf", (PyCFunction)Py_DistanceTransformBruteForce, 
   METH_VARARGS, ""},
  {"distance_transform_op", (PyCFunction)Py_DistanceTransformOnePass, 
   METH_VARARGS, ""},
  {"euclidean_feature_transform",
   (PyCFunction)Py_EuclideanFeatureTransform, METH_VARARGS, ""},
  {"watershed_ift", (PyCFunction)Py_WatershedIFT, METH_VARARGS, ""},
  {"label", (PyCFunction)Py_Label, METH_VARARGS, ""},
  {"find_objects", (PyCFunction)Py_FindObjects, METH_VARARGS, ""},
  {"spline_filter1d", (PyCFunction)Py_SplineFilter1D, METH_VARARGS, ""},
  {"affine_transform", (PyCFunction)Py_AffineTransform, METH_VARARGS, ""},
  {"geometric_transform", (PyCFunction)Py_GeometricTransform, METH_VARARGS, 
   ""},
  {"map_coordinates", (PyCFunction)Py_MapCoordinates, METH_VARARGS, ""},
  {"zoom", (PyCFunction)Py_Zoom, METH_VARARGS, ""},
  {"shift", (PyCFunction)Py_Shift, METH_VARARGS, ""},
  {"sum", (PyCFunction)Py_Sum, METH_VARARGS, ""},
  {"mean", (PyCFunction)Py_Mean, METH_VARARGS, ""},
  {"variance", (PyCFunction)Py_Variance, METH_VARARGS, ""},
  {"minimum", (PyCFunction)Py_Minimum, METH_VARARGS, ""},
  {"maximum", (PyCFunction)Py_Maximum, METH_VARARGS, ""},
  {"minimum_position", (PyCFunction)Py_MinimumPosition, METH_VARARGS, ""},
  {"maximum_position", (PyCFunction)Py_MaximumPosition, METH_VARARGS, ""},
  {"center_of_mass", (PyCFunction)Py_CenterOfMass, METH_VARARGS, ""},
  {"fourier_boxcar", (PyCFunction)Py_FourierBoxcar, METH_VARARGS, ""},
  {"fourier_ellipsoid", (PyCFunction)Py_FourierEllipsoid, METH_VARARGS,
   ""},
  {"fourier_gaussian", (PyCFunction)Py_FourierGaussian, METH_VARARGS, ""},
  {"fourier_shift", (PyCFunction)Py_FourierShift, METH_VARARGS, ""},
  {NULL, NULL, 0, NULL}
};


/* PyMODINIT_FUNC */
void
init_nd_image(void)
{
  Py_InitModule("_nd_image", methods);
  import_libnumarray();
  import_array();
}
