
#include <Python.h>
#include <stdio.h>
#include <math.h>
#include "libnumarray.h"

#ifdef MS_WIN32
#pragma warning(once : 4244)
#endif

#define logical_and(arg1, arg2) (arg1 != 0) & (arg2 != 0)
#define logical_or(arg1, arg2)  (arg1 != 0) | (arg2 != 0)
#define logical_xor(arg1, arg2) ((arg1 != 0) ^ (arg2 != 0)) & 1
#define ufmaximum(arg1, arg2) (((temp1=arg1) > (temp2=arg2)) ? temp1 : temp2)
#define ufminimum(arg1, arg2) (((temp1=arg1) < (temp2=arg2)) ? temp1 : temp2)

static PyObject *_Error;



static double c_bessel(double x, double y)
{
      double x2=x*x, y2=y*y, diff=x2-y2;
      return diff*diff/(x2+y2);
}

static UInt8  c_bar(UInt32 x )
{
      return (x >> 24) & 0xFF;
}


static int bar_UInt32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    UInt32 *tin = (UInt32 *) buffers[0];
    UInt8 *tout = (UInt8 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = c_bar(*tin);
    }
    return 0;
}

UFUNC_DESCR2(bar_UInt32_vector, sizeof(UInt32), sizeof(UInt8));

static int Cos_Float32_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin = (Float32 *) buffers[0];
    Float32 *tout = (Float32 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = cos(*tin);
    }
    return 0;
}

UFUNC_DESCR2(Cos_Float32_vector, sizeof(Float32), sizeof(Float32));

static int bessel_Float32_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32  *tin1  = (Float32 *) buffers[0];
    Float32  *tin2  = (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = c_bessel(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(bessel_Float32_vector_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int bessel_Float32_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 *tin1     = (Float32 *) buffers[0];
    Float32 tscalar   = *(Float32 *) buffers[1];
    Float32 *tout    = (Float32 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = c_bessel(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(bessel_Float32_vector_scalar, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int bessel_Float32_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float32 tscalar = *(Float32 *) buffers[0];
    Float32 *tin2   =  (Float32 *) buffers[1];
    Float32 *tout  = (Float32 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = c_bessel(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(bessel_Float32_scalar_vector, sizeof(Float32), sizeof(Float32), sizeof(Float32));


static int bessel_Float32_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32 *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout = (Float32 *) ((char *) output + outboffset);
    Float32 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            tout = (Float32 *) ((char *) tout + outbstrides[dim]);
            *tout = c_bessel(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            bessel_Float32_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(bessel_Float32_accumulate, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));

static int bessel_Float32_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float32  *tin   = (Float32 *) ((char *) input  + inboffset);
    Float32 *tout  = (Float32 *) ((char *) output + outboffset);
    Float32 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float32 *) ((char *) tin + inbstrides[dim]);
            net = c_bessel(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            bessel_Float32_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(bessel_Float32_reduce, CHECK_ALIGN, sizeof(Float32), sizeof(Float32));

static int Cos_Float64_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin = (Float64 *) buffers[0];
    Float64 *tout = (Float64 *) buffers[1];
    
    for (i=0; i<niter; i++, tin++, tout++) {
        *tout = cos(*tin);
    }
    return 0;
}

UFUNC_DESCR2(Cos_Float64_vector, sizeof(Float64), sizeof(Float64));

static int bessel_Float64_vector_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64  *tin1  = (Float64 *) buffers[0];
    Float64  *tin2  = (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tin2++, tout++) {
        *tout = c_bessel(*tin1, *tin2);
    }
    return 0;
}

UFUNC_DESCR3(bessel_Float64_vector_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int bessel_Float64_vector_scalar(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 *tin1     = (Float64 *) buffers[0];
    Float64 tscalar   = *(Float64 *) buffers[1];
    Float64 *tout    = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin1++, tout++) {
        *tout = c_bessel(*tin1, tscalar);
    }
    return 0;
}

UFUNC_DESCR3vs(bessel_Float64_vector_scalar, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int bessel_Float64_scalar_vector(long niter, long ninargs, long noutargs, void **buffers, long *bsizes) {
    long i;
    Float64 tscalar = *(Float64 *) buffers[0];
    Float64 *tin2   =  (Float64 *) buffers[1];
    Float64 *tout  = (Float64 *) buffers[2];
    
    for (i=0; i<niter; i++, tin2++, tout++) {
        *tout = c_bessel(tscalar, *tin2);
    }
    return 0;
}

UFUNC_DESCR3sv(bessel_Float64_scalar_vector, sizeof(Float64), sizeof(Float64), sizeof(Float64));


static int bessel_Float64_accumulate(
            long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64 *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout = (Float64 *) ((char *) output + outboffset);
    Float64 lastval;
    
    if (dim == 0) {
        for (i=1; i<niters[dim]; i++) {
            lastval = *tout;
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            tout = (Float64 *) ((char *) tout + outbstrides[dim]);
            *tout = c_bessel(lastval ,*tin);
        }
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            bessel_Float64_accumulate(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(bessel_Float64_accumulate, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));

static int bessel_Float64_reduce(long dim, long dummy, maybelong *niters,
            void *input,  long inboffset,  maybelong *inbstrides,
            void *output, long outboffset, maybelong *outbstrides) {
    long i;
    Float64  *tin   = (Float64 *) ((char *) input  + inboffset);
    Float64 *tout  = (Float64 *) ((char *) output + outboffset);
    Float64 net;
    
    if (dim == 0) {
        net = *tout;
        for (i=1; i<niters[dim]; i++) {
            tin = (Float64 *) ((char *) tin + inbstrides[dim]);
            net = c_bessel(net, *tin);
        }
        *tout = net;
    }
    else {
        for (i=0; i<niters[dim]; i++) {
            bessel_Float64_reduce(dim-1, dummy, niters,
               input,  inboffset  + i*inbstrides[dim],  inbstrides,
               output, outboffset + i*outbstrides[dim], outbstrides);
        }
    }
    return 0;
}

STRIDING_DESCR2(bessel_Float64_reduce, CHECK_ALIGN, sizeof(Float64), sizeof(Float64));

static PyMethodDef _fooMethods[] = {

	{NULL,      NULL}        /* Sentinel */
};

static PyObject *init_funcDict(void) {
    PyObject *dict;
    dict = PyDict_New();
    NA_add_cfunc(dict, "('bar', 'vector', (('UInt32',), ('UInt8',)))", (void *) &bar_UInt32_vector_descr);
    NA_add_cfunc(dict, "('Cos', 'vector', (('Float32',), ('Float32',)))", (void *) &Cos_Float32_vector_descr);
    NA_add_cfunc(dict, "('bessel', 'vector_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &bessel_Float32_vector_vector_descr);
    NA_add_cfunc(dict, "('bessel', 'vector_scalar', (('Float32', 'Float32'), ('Float32',)))", (void *) &bessel_Float32_vector_scalar_descr);
    NA_add_cfunc(dict, "('bessel', 'scalar_vector', (('Float32', 'Float32'), ('Float32',)))", (void *) &bessel_Float32_scalar_vector_descr);
    NA_add_cfunc(dict, "('bessel', 'reduce', (('Float32',), ('Float32',)))", (void *) &bessel_Float32_reduce_descr);
    NA_add_cfunc(dict, "('bessel', 'accumulate', (('Float32',), ('Float32',)))", (void *) &bessel_Float32_accumulate_descr);
    NA_add_cfunc(dict, "('Cos', 'vector', (('Float64',), ('Float64',)))", (void *) &Cos_Float64_vector_descr);
    NA_add_cfunc(dict, "('bessel', 'vector_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &bessel_Float64_vector_vector_descr);
    NA_add_cfunc(dict, "('bessel', 'vector_scalar', (('Float64', 'Float64'), ('Float64',)))", (void *) &bessel_Float64_vector_scalar_descr);
    NA_add_cfunc(dict, "('bessel', 'scalar_vector', (('Float64', 'Float64'), ('Float64',)))", (void *) &bessel_Float64_scalar_vector_descr);
    NA_add_cfunc(dict, "('bessel', 'reduce', (('Float64',), ('Float64',)))", (void *) &bessel_Float64_reduce_descr);
    NA_add_cfunc(dict, "('bessel', 'accumulate', (('Float64',), ('Float64',)))", (void *) &bessel_Float64_accumulate_descr);
    return dict;
}

/* platform independent*/
#ifdef MS_WIN32
__declspec(dllexport)
#endif
void init_foo(void) {
    PyObject *m, *d;
    m = Py_InitModule("_foo", _fooMethods);
    d = PyModule_GetDict(m);
    _Error = PyErr_NewException("_foo.error", NULL, NULL);
    PyDict_SetItemString(d, "error", _Error);
    import_libnumarray();
    PyDict_SetItemString(d, "functionDict", init_funcDict());
}
