/* --------------------------------------------------------------------------
 * module_system.c
 * code for general system functions
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * -------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <pbb.h>

#include "gettext_macros.h"
#include "input_manager.h"
#include "config_manager.h"
#include "module_system.h"

struct moddata_system {
	char *userallowed;  /* remember system tag for config file saving */
	char *configwritefile; /* configfile to write: eg. /etc/pbbuttonsd.conf */
	int  autorescan;
	int  rescantimer;
} modbase_system;

int
system_init ()
{
	struct moddata_system *base = &modbase_system;
	static char buf_userallowed[STDBUFFERLEN];
	static char buf_configwritefile[STDBUFFERLEN];
	int sid;

	bzero (buf_userallowed, STDBUFFERLEN);
	bzero (buf_configwritefile, STDBUFFERLEN);
	base->autorescan      = 0;   /* disable by default */
	base->rescantimer     = 0;
	base->userallowed     = buf_userallowed;
	base->configwritefile = buf_configwritefile;

	if ((sid = registerCfgSection ("SYSTEM")) == 0) {
		print_msg (PBB_ERR, _("Can't register configuration section %s, out of memory."), "SYSTEM");
		return E_NOMEM;
	} else {
		registerCfgOptionString (sid, "userallowed", TAG_USERALLOWED, 0,
				N_("user who is allowed to use IPC"));
		registerCfgOptionBool   (sid, "autorescan",  TAG_AUTORESCAN,  0,
				N_("automatic rescan of event devices"));
		registerCfgOptionInt    (sid, "CmdTimeout",  TAG_TIMEFORCMD,  0,
				N_("timeout in seconds for launched scripts, etc."));
	}
	
	register_function (QUERYQUEUE, system_query);
	register_function (CONFIGQUEUE, system_configure);
	return 0;
}

int
system_open (struct tagitem *taglist)
{
	ipc_protect_tag (TAG_SAVECONFIG);  /* tags only for privileged use */

	register_function (T1000QUEUE, system_timer1000);
	return 0;
}

int
system_close ()
{
	return 0;
}

int
system_exit ()
{
	return 0;
}

void
system_query (struct tagitem *taglist)
{
	system_handle_tags (MODE_QUERY, taglist);
}

void
system_configure (struct tagitem *taglist)
{
	system_handle_tags (MODE_CONFIG, taglist);
}

void
system_handle_tags (int cfgure, struct tagitem *taglist)
{
	struct moddata_system *base = &modbase_system;
	struct tagitem args[] = {{ TAG_AMBIENTLIGHT, 0 },
	                         { TAG_SLEEPSUPPORTED, 0 },
	                         { TAG_END, 0 }};
	static char *version = VERSION;
	int val;

	while (taglist->tag != TAG_END) {
		switch (taglist->tag) {
		case TAG_SAVECONFIG:
			if (cfgure) {
				if (strlen(base->configwritefile) > 0)
					writeConfigFile (base->configwritefile);
				else
					tagerror (taglist, E_NOSUPPORT);
			} else
				tagerror (taglist, E_NOREAD);
			break;
		case TAG_USERALLOWED:  /* private tag */
			if (cfgure)	strncpy(base->userallowed, (char*) taglist->data, STDBUFFERLEN);
			else if (strlen (base->userallowed) > 0)
				taglist->data = (long) base->userallowed;
			break;
		case TAG_VERSION:
			if (cfgure)	tagerror (taglist, E_NOWRITE);
			else		taglist->data = (long) version;
			break;
		case TAG_AUTORESCAN:
			if (cfgure)	base->autorescan = taglist->data;
			else		taglist->data = (long) base->autorescan;
			break;
		case TAG_TIMEFORCMD:
			if (cfgure)	set_timeforcmd(taglist->data);
			else		taglist->data = (long) get_timeforcmd();
			break;
		case TAG_SYSINFO:
			if (cfgure)	tagerror (taglist, E_NOWRITE);
			else {
				val = 0;
				if (strlen(base->configwritefile) > 0)
					val |= SYSINFO_CONFIGWRITABLE;
				process_queue (QUERYQUEUE, args);
				if (tagfind (args, TAG_AMBIENTLIGHT, -1) != -1)
					val |= SYSINFO_HAVELMU;
				if (tagfind (args, TAG_SLEEPSUPPORTED, -1) != -1)
					val |= SYSINFO_SLEEPSUPPORTED;
#ifdef WITH_PMUD
				val |= SYSINFO_PMUDSUPPORT;
#endif
#ifdef WITH_OSS
				val |= SYSINFO_HAVEOSS;
#endif
#ifdef WITH_ALSA
				val |= SYSINFO_HAVEALSA;
#endif
				taglist->data = val;
			}
			break;
		case TAG_CFGNAME:    /* private tag */
			if (cfgure)
				system_config_writable ((char*) taglist->data, base->configwritefile);
			break;
		}
		taglist++;
	}
}

void
system_timer1000 (struct tagitem *taglist)
{
	struct moddata_system *base = &modbase_system;

	if (base->autorescan) {
		base->rescantimer++;
		if (base->rescantimer > RESCANTIME) {
			base->rescantimer = 0;
			scanEventDevices ();
		}
	}
}

void
system_config_writable (char *configreadfile, char *configwritefile)
{
	int perms, owner;

	configwritefile[0] = '\0';
	owner = get_owner(configreadfile);
	perms = get_permissions(configreadfile);
	if (owner != getuid() || (perms & 022))
		print_msg (PBB_WARN, _("config file [%s] is insecure, saving of config disabled.\n"), configreadfile);
	else if (owner == getuid() && !(perms & 0200))
		print_msg (PBB_WARN, _("config file [%s] is not writable, saving of config disabled.\n"), configreadfile);
	else {
		strcpy(configwritefile, configreadfile);
		print_msg (PBB_INFO, _("saving of config enabled to %s.\n"), configwritefile);
		return; /* config is writable, name in configwritefile */
	}
}

