/* ----------------------------------------------------------------------------
 * config_manager.c
 *
 * Copyright 2002-2005 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdarg.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <malloc.h>
#include <sys/stat.h>

#include <pbb.h>

#include "gettext_macros.h"
#include "input_manager.h"
#include "config_manager.h"


/* --- private module data structure of the config manager --- */

struct moddata_configmanager {
	struct listhead sectionlist;    /* list of config sections */
	struct listhead optionlist;     /* list of config options */
} modbase_configmanager;


/* Init function called by the program init procedure. Function is known by
   the prginitab. It does module data initialisation. */

int
configmanager_init ()
{
	struct moddata_configmanager *base = &modbase_configmanager;

	initListHeader (&base->sectionlist);
	initListHeader (&base->optionlist);
	return 0;
}

int
configmanager_exit ()
{
	struct moddata_configmanager *base = &modbase_configmanager;

	freeList (&base->sectionlist);
	freeList (&base->optionlist);
	return 0;
}

int
registerCfgSection (char *name)
{
	struct moddata_configmanager *base = &modbase_configmanager;
	struct listnode *it = base->sectionlist.first, *newnode;	
	struct cfgsection *section;
	int newsecid = 0;

	while ((it = it->next)) {
		section = (struct cfgsection *) getNodeData (it->pred);
		if (strcasecmp (name, section->name) == 0)
			return section->sectionid;
		
		if (section->sectionid > newsecid)
			newsecid = section->sectionid;
	}
	
	if ((newnode = newListNode (sizeof (struct cfgsection) + strlen (name) + 1))) {
		section = (struct cfgsection *) getNodeData (newnode);
		section->sectionid = newsecid = newsecid + 1;
		section->name = (char*) section + sizeof (struct cfgsection);
		strcpy (section->name, name);
		appendListNode (&base->sectionlist, newnode);
	}
	
	return newsecid;
}

struct cfgoption *
findCfgOption (char *name)
{
	struct moddata_configmanager *base = &modbase_configmanager;
	struct listnode *it = base->optionlist.first;	
	struct cfgoption *option;

	while ((it = it->next)) {
		option = (struct cfgoption *) getNodeData (it->pred);
		if (strcasecmp (name, option->name) == 0)
			return option;
	}
	return NULL;
}

char *
prependString (char **address, char *text)
{
	if (text) {  /* text might be NULL */
		*address -= strlen(text) + 1;
		strcpy (*address, text);
		return *address;
	} else
		return NULL;
}

int
registerCfgOptionInt (int sid, char *name, long tag, int flags, char *comment)
{
	return registerCfgOption (sid, name, DT_INT, tag, 0, flags, comment);
}

int
registerCfgOptionString (int sid, char *name, long tag, int flags, char *comment)
{
	return registerCfgOption (sid, name, DT_STRING, tag, 0, flags, comment);
}

int
registerCfgOptionKey (int sid, char *name, long tag1, long tag2, int flags, char *comment)
{
	return registerCfgOption (sid, name, DT_KEY, tag1, tag2, flags, comment);
}
	
int
registerCfgOptionBool (int sid, char *name, long tag, int flags, char *comment)
{
	if (flags & FLG_INVBOOL)
		return registerCfgOption (sid, name, DT_INVBOOL, tag, 0, flags, comment);
	else
		return registerCfgOption (sid, name, DT_BOOL, tag, 0, flags, comment);
}


int
registerCfgOption (int sid, char *name, int datatype, long tag1, long tag2, int flags, char *comment)
{
	struct moddata_configmanager *base = &modbase_configmanager;
	struct listnode *newnode;
	struct cfgoption* option;
	int datalen = 0;
	char *buffer;

	if ((option = findCfgOption (name)))
		return E_OPTEXISTS;  /* option is already registered, not really an error */
	
	if (name)    datalen += strlen (name) + 1;
	if (comment) datalen += strlen(comment) + 1;
	if ((newnode = newListNode (sizeof (struct cfgoption) + datalen))) {
		option = (struct cfgoption *) getNodeData (newnode); 
		option->type = datatype;
		option->sectionid = sid;
		option->tag1 = tag1;
		option->tag2 = tag2;
		option->readonly = (flags & FLG_RONLY) ? 1 : 0;
		buffer = getNodeDataEnd (newnode);
		option->name = prependString (&buffer, name);
		option->comment = prependString (&buffer, comment);
		appendListNode (&base->optionlist, newnode);
		return E_OK;
	}
	return E_NOMEM;
}

int
registerCfgOptionList (int sid, char *name, long tag, int flags, char *comment, ...)
{
	struct moddata_configmanager *base = &modbase_configmanager;
	struct listnode *newnode;
	struct cfgoption* option;
	int datalen = 0, n;
	va_list vargs;
	char *tmpstring, *buffer;

	if ((option = findCfgOption (name)))
		return E_OPTEXISTS;  /* option is already registered, not really an error */
	
	if (name)    datalen += strlen (name) + 1;
	if (comment) datalen += strlen(comment) + 1;
	va_start (vargs, comment);
	while ((tmpstring = va_arg (vargs, char*)))
		datalen += strlen(tmpstring) + 1 + sizeof (char*);
	va_end (vargs);

	if ((newnode = newListNode (sizeof (struct cfgoption) + datalen))) {
		option = (struct cfgoption *) getNodeData (newnode); 
		option->type = DT_LIST;
		option->sectionid = sid;
		option->tag1 = tag;
		option->readonly = (flags & FLG_RONLY) ? 1 : 0;
		buffer = getNodeDataEnd (newnode);
		option->name = prependString (&buffer, name);
		option->comment = prependString (&buffer, comment);
		option->items = (char **) ((char *)option + sizeof (struct cfgoption));

		n = 0;
		va_start (vargs, comment);
		while ((tmpstring = va_arg (vargs, char*)))
			option->items[n++] = prependString (&buffer, tmpstring);
		va_end (vargs);
		option->itemscnt = n;
		
		appendListNode (&base->optionlist, newnode);
		return E_OK;
	}
	return E_NOMEM;
}

/* ------ */

struct tagitem *
readConfigFile (char *filename)
{
	struct moddata_configmanager *base = &modbase_configmanager;
	struct stat stat_buf;
	struct tagitem *taglist;
	struct listnode *it;	
	struct cfgoption *option;
	char *buffer, *strptr, linebuffer[STDBUFFERLEN], *token, *arg;
	int bufferlen, n;
	FILE *fd;

	if ((stat(filename, &stat_buf)) == -1) {
		print_msg (PBB_WARN, _("Couldn't read configfile [%s, %s], using defaults.\n"), filename, strerror(errno));
		return NULL;
	}
	bufferlen = stat_buf.st_size;

	if ((buffer = (char *) malloc (bufferlen)) == 0) {
		print_msg (PBB_WARN, _("Couldn't read configfile [%s, out of memory], using defaults.\n"), filename);
		return NULL;
	}

	taglist = (struct tagitem *) buffer;    /* moving pointer */
	strptr = buffer + bufferlen;         /* moving pointer */

	if ((fd = fopen(filename, "r"))) {
		while (fgets(linebuffer, sizeof(linebuffer), fd)) {
			if (linebuffer[0] != '#') {
				cleanup_buffer(linebuffer);

				for (n=0; n < strlen (linebuffer); n++)
					if (linebuffer[n] == ';' || linebuffer[n] == '#') {
						linebuffer[n] = 0;
						break;
					}
				
				if ((token = strtok(linebuffer, "=\n"))) {
					arg = strtok(0, "#;=\n");
					
					it = base->optionlist.first;	
					while ((it = it->next)) {
						option = (struct cfgoption *) getNodeData (it->pred);
						if (strcasecmp (token, option->name) == 0) {
							switch (option->type) {
								case DT_INT:
									readCfgInt (option, &taglist, &strptr, arg);
									break;
								case DT_STRING:
									readCfgString (option, &taglist, &strptr, arg);
									break;
								case DT_KEY:
									readCfgKey (option, &taglist, &strptr, arg);
									break;
								case DT_BOOL:
								case DT_INVBOOL:
									readCfgBool (option, &taglist, &strptr, arg);
									break;
								case DT_LIST:
									readCfgList (option, &taglist, &strptr, arg);
									break;
							}
							break;
						}
					}
					if (it == NULL)
						strtok(0,"=\n");
				}
			}
		}
		taglist->tag = TAG_END;
		fclose(fd);
	}
	return (struct tagitem *) buffer;
}

void
readCfgInt (struct cfgoption *option, struct tagitem **taglist, char **strptr, char *arg)
{
	(*taglist)->tag = option->tag1;
	(*taglist)->data = atoi (arg);
	(*taglist)++;
}


void
readCfgString (struct cfgoption *option, struct tagitem **taglist, char **strptr, char *arg)
{
	char *empty = "";

	if (!arg) arg = empty;
		
	*strptr = *strptr - strlen (arg) - 1;
	strcpy (*strptr, arg);    /* copy argument before end of current stringbuffer */
	(*taglist)->tag = option->tag1;
	(*taglist)->data = (long) *strptr;
	(*taglist)++;
}

void
readCfgKey (struct cfgoption *option, struct tagitem **taglist, char **strptr, char *arg)
{
	int key, mod;

	decode_key_and_modifier(arg, &key, &mod);
	(*taglist)->tag = option->tag1;
	(*taglist)->data = (long) key;
	(*taglist)++;
	(*taglist)->tag = option->tag2;
	(*taglist)->data = (long) mod;
	(*taglist)++;
}

void
decode_key_and_modifier(char *arg, int *key, int *mod)
{
	char *tag;
	int k;

	*mod = 0; *key = 0;
	tag = strtok(arg, "+\n");
	while (tag) {
		k = atoi (tag);
		if (k == 0) {
			/* while (*tag == ' ') tag++;    ignore leading spaces */
			if (!strncmp("shift", tag, 5))
				*mod |= MOD_SHIFT;
			else if (!strncmp("alt", tag, 3))
				*mod |= MOD_ALT;
			else if (!strncmp("ctrl", tag, 4))
				*mod |= MOD_CTRL;
		} else
			*key = k;
		tag = strtok(0, "+\n");
	}
}

void
readCfgBool (struct cfgoption *option, struct tagitem **taglist, char **strptr, char *arg)
{
	(*taglist)->tag = option->tag1;
	if (!strncmp("yes", arg, 3))
		(*taglist)->data = 1;
	else
		(*taglist)->data = 0;

	if (option->type == DT_INVBOOL)
		(*taglist)->data ^= 1;
	
	(*taglist)++;
}

void
readCfgList (struct cfgoption *option, struct tagitem **taglist, char **strptr, char *arg)
{
	int cnt;

	(*taglist)->tag = option->tag1;
	for (cnt = 0; cnt < option->itemscnt; cnt++)
		if (!strcasecmp (option->items[cnt], arg)) {
			(*taglist)->data = cnt;
			(*taglist)++;
			break;
		}
}

/* ---- */

void
writeConfigFile (char *filename)
{
	struct moddata_configmanager *base = &modbase_configmanager;
	struct tagitem *taglist;
	struct listnode *secit, *optit;
	struct cfgsection *section;
	struct cfgoption *option;
	long fpos1, fpos2;
	FILE *fd;

	if ((taglist = (struct tagitem*) malloc (2 * sizeof(struct tagitem) * getNodeCount (&base->optionlist))) == 0) {
	    print_msg (PBB_WARN, _("Couldn't write configfile [%s, out of memory].\n"), filename);
		return;
	}

	taglist_init (taglist);  /* create taglist containing all possible config settings */
	optit = base->optionlist.first;	
	while ((optit = optit->next)) {
		option = (struct cfgoption *) getNodeData (optit->pred);
		taglist_add (taglist, option->tag1, -1);
		if (option->tag2 != 0)
			taglist_add (taglist, option->tag2, -1);
	}
	process_queue (QUERYQUEUE, taglist);   /* ask modules for config settings */

	if ((fd = fopen(filename, "w"))) {
		fputs (_("# Configuration file for pbbuttonsd >= version 0.5\n# for options see man pbbuttonsd.conf\n"), fd);

		secit = base->sectionlist.first;	
		while ((secit = secit->next)) {
			section = (struct cfgsection *) getNodeData (secit->pred);
			fprintf (fd, "\n[%s]\n", section->name);

			optit = base->optionlist.first;	
			while ((optit = optit->next)) {
				option = (struct cfgoption *) getNodeData (optit->pred);

				if (option->sectionid != section->sectionid)
					continue;

				if (option->readonly)
					continue;
			
				/* if the tag can't be found or the query functions sets it to -1,
				 * the option will be commented out. With next program start this
				 * option will be set to default.
				 */
				if (tagfind (taglist, option->tag1, -1) == -1)
					fputs ("#", fd);
				fprintf (fd, "%-22s= ", option->name);
				fpos1 = ftell(fd);
	
				switch (option->type) {
				case DT_INT:
					writeCfgInt (fd, option, taglist);
					break;
				case DT_STRING:
					writeCfgString (fd, option, taglist);
					break;
				case DT_KEY:
					writeCfgKey (fd, option, taglist);
					break;
				case DT_BOOL:
				case DT_INVBOOL:
					writeCfgBool (fd, option, taglist);
					break;
				case DT_LIST:
					writeCfgList (fd, option, taglist);
					break;
				}
				
				if (option->comment) {
					fpos2 = ftell(fd);
					if ((fpos2 - fpos1) < 8)
						fprintf(fd, "\t");
					fprintf (fd, "\t; %s", _(option->comment));
				}
				fputs("\n", fd);
			}
		}
		fclose(fd);
	}
	free(taglist);  /* free memory for taglist */
}

void
writeCfgInt (FILE *fd, struct cfgoption *option, struct tagitem *taglist)
{
	int val;
	
	val = (int) tagfind (taglist, option->tag1, -1);
	fprintf (fd, "%d", val == -1 ? 0 : val);
}

void
writeCfgString (FILE *fd, struct cfgoption *option, struct tagitem *taglist)
{
	long val;
	
    val = tagfind (taglist, option->tag1, -1);
	if (val == -1 || val == 0) 
		fprintf (fd, "\"\"");
	else	
		fprintf (fd, "\"%s\"", (char *) val);
}

void
writeCfgKey (FILE *fd, struct cfgoption *option, struct tagitem *taglist)
{
	int key, mod;

    key = (int) tagfind (taglist, option->tag1, -1);
    mod = (int) tagfind (taglist, option->tag2, -1);
	fprintf (fd, "%s", code_key_and_modifier(key, mod));
}

char*
code_key_and_modifier (int key, int mod)
{
	static char buffer[30];

	if (key == -1) key = 0;
	if (mod == -1) mod = 0;
	
	snprintf (buffer, sizeof(buffer)-1, "%d", (key & 255));
	if (mod & MOD_SHIFT)
		strcat (buffer, " + shift");
	if (mod & MOD_ALT)
		strcat (buffer, " + alt");
	if (mod & MOD_CTRL)
		strcat (buffer, " + ctrl");
	return buffer;
}

void
writeCfgBool (FILE *fd, struct cfgoption *option, struct tagitem *taglist)
{
    int val;
	
	val = (int) tagfind (taglist, option->tag1, -1);
	if (val == -1) val = 0;
	if (option->type == DT_INVBOOL)	val ^= 1;
	fprintf (fd, "%s", val ? "yes" : "no");
}

void
writeCfgList (FILE *fd, struct cfgoption *option, struct tagitem *taglist)
{
	int val;
	
	val = (int) tagfind (taglist, option->tag1, -1);
	if (val < 0 || val >= option->itemscnt) val = 0;
	fprintf (fd, "%s", option->items[val]);	
}

